<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('JPATH_PLATFORM') or die;

use Joomla\Utilities\ArrayHelper;

JLoader::register('CommunitySurveysHelper', JPATH_ADMINISTRATOR . '/components/com_communitysurveys/helpers/communitysurveys.php');

class CommunitySurveysModelQuestion extends JModelAdmin
{
	protected $text_prefix = 'COM_COMMUNITYSURVEYS';

	public $typeAlias = 'com_communitysurveys.question';

	public function __construct($config)
	{
		parent::__construct($config);
	}
	
	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    $record    A record object.
	 *
	 * @return  boolean  True if allowed to delete the record. Defaults to the permission set in the component.
	 */
	protected function canDelete($record)
	{
		if (!empty($record->id))
		{
			if ($record->published != -2)
			{
				return;
			}
			$user = JFactory::getUser();
			return $user->authorise('core.delete', 'com_communitysurveys.survey.' . (int) $record->survey_id);
		}
	}

	/**
	 * Method to test whether a record can have its state edited.
	 *
	 * @param   object    $record    A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission set in the component.
	 */
	protected function canEditState($record)
	{
		$user = JFactory::getUser();

		// Check for existing article.
		if (!empty($record->id))
		{
			return $user->authorise('core.edit.state', 'com_communitysurveys.survey.' . (int) $record->survey_id);
		}
		else
		{
			return parent::canEditState('com_communitysurveys');
		}
	}

	/**
	 * Returns a Table object, always creating it.
	 *
	 * @param   type      The table type to instantiate
	 * @param   string    A prefix for the table class name. Optional.
	 * @param   array     Configuration array for model. Optional.
	 *
	 * @return  JTable    A database object
	 */
	public function getTable($type = 'Question', $prefix = 'CommunitySurveysTable', $config = array())
	{
		return JTable::getInstance($type, $prefix, $config);
	}
	
	/**
	 * Method to get the record form.
	 *
	 * @param   array      $data        Data for the form.
	 * @param   boolean    $loadData    True if the form is to load its own data (default case), false if not.
	 *
	 * @return  mixed  A JForm object on success, false on failure
	 */
	public function getForm($data = array(), $loadData = true)
	{
		// Get the form.
		$form = $this->loadForm('com_communitysurveys.question', 'question', array('control' => 'jform', 'load_data' => $loadData));
		
		if (empty($form))
		{
			return false;
		}
		$jinput = JFactory::getApplication()->input;
	
		// The front end calls this model and uses a_id to avoid id clashes so we need to check for that first.
		if ($jinput->get('qid'))
		{
			$id = $jinput->get('qid', 0);
		}
		// The back end uses id so we use that the rest of the time and set it to 0 by default.
		else
		{
			$id = $jinput->get('qid', 0);
		}
		// Determine correct permissions to check.
		if ($this->getState('question.id'))
		{
			$id = $this->getState('question.id');
		}
	
		return $form;
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  The form data.
	 *
	 * @return  boolean  True on success.
	 */
	public function save($question)
	{
		$app 	= JFactory::getApplication();
		$user	= JFactory::getUser();
		$db 	= JFactory::getDbo();
		
		if(!$question->id)
		{
			$question->created_by = $user->id;
			
			// get the sort order
			$query = $db->getQuery(true)
				->select('max(sort_order) + 1')
				->from('#__survey_questions')
				->where('survey_id = '.$question->survey_id.' and page_number = '.$question->page_number);
			$db->setQuery($query);
			
			try 
			{
				$question->sort_order = (int) $db->loadResult();
			}
			catch (Exception $e){}
			
			try 
			{
			    $db->insertObject('#__survey_questions', $question);
			}
			catch (Exception $e)
			{
			    throw new Exception( $e->getMessage() . '<br><br> RC: 1<br><br>', 500 );
			}
			
			$question->id = $db->insertid();
			$this->setState($this->getName() . '.id', $question->id);
			$this->setState($this->getName() . '.new', true);
			
			return true; // no need to insert any answers.
		}
		else
		{
			// security check, if the survey id is manipulated
			$query = $db->getQuery(true)
				->select('survey_id')
				->from('#__survey_questions')
				->where('id = '.$question->id);
			
			$db->setQuery($query);
			try 
			{
				$surveyId = (int) $db->loadResult();
				if($surveyId != $question->_survey_id)
				{
					throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
				}
			}
			catch (Exception $e)
			{
				throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
			}
			
			try
			{
			    $db->updateObject('#__survey_questions', $question, 'id');
			}
			catch (Exception $e)
			{
			    throw new Exception( $e->getMessage() . '<br><br> RC: 2<br><br>', 500 );
			}
			
			$this->setState($this->getName() . '.id', $question->id);
		}
		
		// now add the new answers, remove deleted ones, update existing
		switch ($question->question_type)
		{
			case CS_CHOICE_RADIO : // Multiple choice - radio
			case CS_CHOICE_CHECKBOX : // Multiple choice - checkbox
			case CS_CHOICE_SELECT : // Multiple choice - select
			case CS_CHOICE_RANKING: // Choice - Ranking
			case CS_CHOICE_SLIDER: // Choice Slider
				
				$choices = $app->input->post->getArray(array('answers'=>'string'));
				$choices = json_decode($choices['answers']);

				if (count ( $choices ) <= 0) 
				{
					if($question->question_type == CS_CHOICE_SLIDER)
					{
						break;
					}
					
					throw new Exception(JText::_ ( 'COM_COMMUNITYSURVEYS_NO_ANSWERS_ENTERED' ) . '| RC: 3', 403);
				}
				
				/**************************** DELETIONS *******************************/
				$existing = array();
				
				foreach ($choices as $row)
				{
					$row = trim($row);
					if (!empty($row) && strpos($row, '_') !== false) 
					{
						$tokens = explode('_', $row, 2);
						if(count($tokens) == 2)
						{
							$answerId = (int) $tokens[0];
							if($answerId > 0)
							{
								$existing[] = $answerId;
							}
						}
					}
				}

				if(!empty($existing))
				{
					// DELETE all answers which are not present in the request
				    $existing = ArrayHelper::toInteger($existing);
					$query = $db->getQuery(true)
						->delete('#__survey_answers')
						->where('survey_id = '.$question->_survey_id.' and question_id = '.$question->id.' and id not in ('.implode(',', $existing).')');
// echo $query->dump();
					$db->setQuery($query);
					
					try
					{
					    $db->execute();
					}
				    catch (Exception $e)
				    {
				        throw new Exception( $e->getMessage() . '<br><br> RC: 4<br>query: ' . $query->__toString() . '<br><br>', 500 );
				    }
				}
				
				/**************************** UPDATES *******************************/
				
				$updateCols = Array(
						'answer_label' => '`answer_label` = CASE ', 
						'sort_order' => '`sort_order` = CASE ', 
						'image' => '`image` = CASE ');

				$queries = $this->_getAnswerUpdateQuery($updateCols, $choices, $question->_survey_id.','.$question->id.','.$db->q( 'x' ));
// var_dump($queries);
				if(!empty($queries['query_update']))
				{
					$query = $db->getQuery(true)
						->update('#__survey_answers')
						->set($queries['query_update'])
						->where('id in ('.implode(',', $queries['update_ids']).')');
// echo $query->dump();
					$db->setQuery ( $query );
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 5<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				
				/**************************** INSERTS *******************************/
				
				if(!empty($queries['query_insert']))
				{
					$query = $db->getQuery(true)
						->insert('#__survey_answers')
						->columns('survey_id, question_id, answer_type, answer_label, sort_order, image')
						->values($queries['query_insert']);
// echo $query->dump();
					$db->setQuery ( $query );
					
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 6<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				} 
				break;
				
			case CS_IMAGE_CHOOSE_IMAGE:
			case CS_IMAGE_CHOOSE_IMAGES:
			
				$choices = $app->input->post->getArray(array('answers'=>'string'));
				$choices = json_decode($choices['answers']);
			
				if (count ( $choices ) <= 0) 
				{
					throw new Exception( JText::_ ( 'COM_COMMUNITYSURVEYS_NO_ANSWERS_ENTERED' ) . ' RC: 7', 403 );
				}

				/**************************** DELETIONS *******************************/
				$existing = array();
				
				foreach ($choices as $row)
				{
					$row = trim($row);
					if (!empty($row) && strpos($row, '_') !== false) 
					{
						$tokens = explode('_', $row, 2);
						if(count($tokens) == 2)
						{
							$answerId = (int) $tokens[0];
							if($answerId > 0)
							{
								$existing[] = $answerId;
							}
						}
					}
				}
				
				if(!empty($existing))
				{
					$query = $db->getQuery(true)
						->delete('#__survey_answers')
						->where('survey_id = '.$question->_survey_id.' and question_id = '.$question->id.' and id not in ('.implode(',', $existing).')');
// echo $query->dump();
					$db->setQuery($query);
					
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 8<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				
				/**************************** UPDATES *******************************/
				
				$updateCols = Array(
						'answer_label' => '`answer_label` = CASE ', 
						'sort_order' => '`sort_order` = CASE ', 
						'image' => '`image` = CASE ');
				
				$queries = $this->_getAnswerUpdateQuery($updateCols, $choices, $question->_survey_id.','.$question->id.','.$db->q( 'x' ));
				if(!empty($queries['query_update']))
				{
					$query = $db->getQuery(true)
						->update('#__survey_answers')
						->set($queries['query_update'])
						->where('id in ('.implode(',', $queries['update_ids']).')');
// echo $query->dump();
					$db->setQuery ( $query );
					
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 9<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				
				/**************************** INSERTS *******************************/
			
				if(!empty($queries['query_insert']))
				{
					$query = $db->getQuery(true)
						->insert('#__survey_answers')
						->columns('survey_id, question_id, answer_type, answer_label, sort_order, image')
						->values($queries['query_insert']);
// echo $query->dump();
					$db->setQuery ( $query );
					
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 10<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				break;

			case CS_GRID_RADIO : // Grid Radio
			case CS_GRID_CHECKBOX : // Grid Checkbox
			case CS_GRID_MATCHING: // Matching
				
				$gridRows = $app->input->post->getArray(array('answers'=>'string'));
				$gridCols = $app->input->post->getArray(array('columns'=>'string'));
				$gridRows = json_decode($gridRows['answers']);
				$gridCols = json_decode($gridCols['columns']);
				
				if ((count ( $gridRows ) <= 0) || (count ( $gridCols ) <= 0)) 
				{
					throw new Exception( JText::_ ( 'COM_COMMUNITYSURVEYS_NO_ANSWERS_ENTERED' ) . ' RC: 11', 500 );
				}
				
				$update_queries = array();
				$insert_queries = array();
				$updateCols = Array(
						'answer_label' => '`answer_label` = CASE ', 
						'sort_order' => '`sort_order` = CASE ', 
						'image' => '`image` = CASE ');
				
				/************************** GET_QUERIES ********************************/
				
				$row_queries = $this->_getAnswerUpdateQuery($updateCols, $gridRows, $question->_survey_id.','.$question->id.','.$db->q( 'x' ));
				$col_queries = $this->_getAnswerUpdateQuery($updateCols, $gridCols, $question->_survey_id.','.$question->id.','.$db->q( 'y' ), 'y');

				/**************************** DELETIONS *******************************/
				$existing = array();
				$rows = array_merge($gridRows, $gridCols);
				
				foreach ($rows as $row)
				{
					$row = trim($row);
					if (!empty($row) && strpos($row, '_') !== false) 
					{
						$tokens = explode('_', $row, 2);
						if(count($tokens) == 2)
						{
							$answerId = (int) $tokens[0];
							if($answerId > 0)
							{
								$existing[] = $answerId;
							}
						}
					}
				}
				
				if(!empty($existing))
				{
					$query = $db->getQuery(true)
						->delete('#__survey_answers')
						->where('survey_id = '.$question->_survey_id.' and question_id = '.$question->id.' and id not in('.implode(',', $existing).')');
					
					$db->setQuery($query);
					
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 12<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				
				/**************************** UPDATES *************************************/
				if(!empty($row_queries['query_update']))
				{
					$update_queries[] = $row_queries['query_update'];
				}
				
				if(!empty($col_queries['query_update']))
				{
					$update_queries[] = $col_queries['query_update'];
				}
				 
				$update_ids = array_merge($row_queries['update_ids'], $col_queries['update_ids']);
				
				if(count($update_queries) > 0)
				{
					$query = $db->getQuery(true)
						->update('#__survey_answers')
						->set(implode(',', $update_queries))
						->where('id in ('.implode(',', $update_ids).')');
					
					$db->setQuery ( $query );

					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 13<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				
				/**************************** INSERTS **********************************/
				
				if(!empty($row_queries['query_insert']))
				{
					$insert_queries = array_merge($insert_queries, $row_queries['query_insert']);
				}
				
				if(!empty($col_queries['query_insert']))
				{
					$insert_queries = array_merge($insert_queries, $col_queries['query_insert']);
				}
				
				if(count($insert_queries) > 0)
				{
					$query = $db->getQuery(true)
						->insert('#__survey_answers')
						->columns('survey_id, question_id, answer_type, answer_label, sort_order, image')
						->values($insert_queries);
					
					$db->setQuery ( $query );
					
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 14<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				break;
								
			case CS_FREE_TEXT_SINGLE_LINE : // Free text - Single line
			case CS_FREE_TEXT_MULTILINE : // Free text - Multiline
			case CS_FREE_TEXT_PASSWORD : // Free text - Password
			case CS_FREE_TEXT_RICH_TEXT : // Free text - Rich Text
			case CS_FILE_UPLOAD: // File upload
		    case CS_SPECIAL_NAME: // Special - Name
		    case CS_SPECIAL_EMAIL: // Special - Email
		    case CS_SPECIAL_CALENDAR: // special - Calendar
		    case CS_SPECIAL_ADDRESS: // Special - Address
		    case CS_CHOICE_NPS: // Choice - Net Promoter Score
		        
				$query = $db->getQuery(true)
					->insert('#__survey_answers')
					->columns('survey_id, question_id, answer_type')
					->values($question->_survey_id . ',' . $question->id . ',' . $db->q( 'text' ));
					
				$db->setQuery ( $query );
				
				try
				{
				    $db->execute();
				}
				catch (Exception $e)
				{
				    throw new Exception( $e->getMessage() . '<br><br> RC: 15<br>query: ' . $query->__toString() . '<br><br>', 500 );
				}
				break;

			case CS_CHOICE_MULTIRATING: // Matching
			
				$gridRows = $app->input->post->getArray(array('answers'=>'string'));
				$gridRows = json_decode($gridRows['answers']);
			
				if (count ( $gridRows ) <= 0)
				{
					throw new Exception( JText::_ ( 'COM_COMMUNITYSURVEYS_NO_QUESTIONS_ENTERED' ) . ' RC: 11', 500 );
				}
			
				$update_queries = array();
				$insert_queries = array();
				$updateCols = Array(
						'answer_label' => '`answer_label` = CASE ',
						'sort_order' => '`sort_order` = CASE ',
						'image' => '`image` = CASE ');
			
				/************************** GET_QUERIES ********************************/
			
				$row_queries = $this->_getAnswerUpdateQuery($updateCols, $gridRows, $question->_survey_id.','.$question->id.','.$db->q( 'x' ));
			
				/**************************** DELETIONS *******************************/
				$existing = array();
				foreach ($gridRows as $row)
				{
					$row = trim($row);
					if (!empty($row) && strpos($row, '_') !== false)
					{
						$tokens = explode('_', $row, 2);
						if(count($tokens) == 2)
						{
							$answerId = (int) $tokens[0];
							if($answerId > 0)
							{
								$existing[] = $answerId;
							}
						}
					}
				}
			
				if(!empty($existing))
				{
					$query = $db->getQuery(true)
					->delete('#__survey_answers')
					->where('survey_id = '.$question->_survey_id.' and question_id = '.$question->id.' and id not in('.implode(',', $existing).')');
						
					$db->setQuery($query);
					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 12<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
			
				/**************************** UPDATES *************************************/
				if(!empty($row_queries['query_update']))
				{
					$update_queries[] = $row_queries['query_update'];
				}
					
				$update_ids = $row_queries['update_ids'];
				if(count($update_queries) > 0)
				{
					$query = $db->getQuery(true)
					->update('#__survey_answers')
					->set(implode(',', $update_queries))
					->where('id in ('.implode(',', $update_ids).')');
						
					$db->setQuery ( $query );

					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 13<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
			
				/**************************** INSERTS **********************************/
			
				if(!empty($row_queries['query_insert']))
				{
					$insert_queries = array_merge($insert_queries, $row_queries['query_insert']);
				}
			
				if(count($insert_queries) > 0)
				{
					$query = $db->getQuery(true)
					->insert('#__survey_answers')
					->columns('survey_id, question_id, answer_type, answer_label, sort_order, image')
					->values($insert_queries);
						
					$db->setQuery ( $query );

					try
					{
					    $db->execute();
					}
					catch (Exception $e)
					{
					    throw new Exception( $e->getMessage() . '<br><br> RC: 14<br>query: ' . $query->__toString() . '<br><br>', 500 );
					}
				}
				break;
		}
		
		return true;
	}
	
	private function _getAnswerUpdateQuery($updateCols, $values, $fields, $type = 'x')
	{
		$db = JFactory::getDbo();
		$app = JFactory::getApplication();
		$query_insert = array();
		$query_update = '';
		$update_ids = array();
	
		foreach ( $values as $choice )
		{
			$choice = trim($choice);
			if (!empty($choice) && strpos($choice, '_') !== false)
			{
				$tokens = explode('_', $choice, 2);
	
				if(count($tokens) == 2)
				{
					$answerId = (int) $tokens[0];
					$sortOrder = (int) $tokens[1];
					$image = 'null';
					
					$title = $app->input->post->getString(($type == 'y' ? 'col_' : '') . 'title_'.$sortOrder);
					$filename = JFile::makeSafe($app->input->post->getString('image_'.$sortOrder));
					if(!empty($filename) && (JFile::exists(CS_TEMP_STORE.'/'.$filename) || JFile::exists(CS_IMAGES_UPLOAD_DIR.'/'.$filename)))
					{
						if(JFile::exists(CS_TEMP_STORE.'/'.$filename))
						{
							JFile::move(CS_TEMP_STORE.'/'.$filename, CS_IMAGES_UPLOAD_DIR.'/'.$filename);
						}
						$image = $db->q($filename);
					}
					
					if($answerId > 0)
					{
						$updateCols['answer_label'] .= "when `id`='".$answerId. "' THEN " . $db->q($title) . " ";
						$updateCols['sort_order'] .= "when `id`='".$answerId. "' THEN '" . $sortOrder . "' ";
						$updateCols['image'] .= "when `id`='".$answerId. "' THEN " . $image . " ";
						$update_ids[] = $answerId;
					}
					else
					{
						$query_insert[] = $fields.','.$db->q( $title ) .','.$sortOrder.','.$image;
					}
				}
			}
		}
	
		foreach($updateCols as $column_name => $query_part)
		{
			$updateCols[$column_name] .= " ELSE `$column_name` END ";
		}
	
		if(count($update_ids) > 0)
		{
			$query_update = implode(', ', $updateCols);
		}

		return array('query_insert'=>$query_insert, 'query_update'=>$query_update, 'update_ids'=>$update_ids);
	}
	
	public function deleteQuestion($surveyId, $questionId)
	{
		try 
		{
			$db = JFactory::getDbo();
			$query = $db->getQuery(true)
				->delete('#__survey_questions')
				->where('id = '.$questionId.' and survey_id = '.$surveyId);
			$db->setQuery($query);
			
			if($db->execute())
			{
				$query = $db->getQuery(true)
					->delete('#__survey_answers')
					->where('survey_id = '.$surveyId.' and question_id = '.$questionId);
				
				$db->setQuery($query);
				$db->execute();
				return true;
			}
		}
		catch(Exception $e)
		{
			return false;
		}
		
		return false;
	}
	
	public function savePage($page)
	{
		$db = JFactory::getDbo();
		
		if($page->id > 0)
		{
			if($db->updateObject('#__survey_pages', $page, 'id'))
			{
				$this->setState($this->getName() . '.id', $page->id);
				return true;
			}
		}
		else
		{
			// first get the sort order
			// get the sort order
			$query = $db->getQuery(true)
				->select('max(sort_order) + 1')
				->from('#__survey_pages')
				->where('sid = '.$page->sid);
			$db->setQuery($query);
				
			try
			{
				$page->sort_order = (int) $db->loadResult();
			}
			catch (Exception $e){}
			
			if($db->insertObject('#__survey_pages', $page))
			{
				$page->id = $db->insertid();
				$this->setState($this->getName() . '.id', $page->id);
				$this->setState($this->getName() . '.new', true);
				return true;
			}
		}
	}
	
	public function deletePage($surveyId, $pageId)
	{
		$db = JFactory::getDbo();
		$query = $db->getQuery(true)
			->delete('#__survey_pages')
			->where('id = '.$pageId.' and sid = '.$surveyId);
		
		$db->setQuery($query);
		try 
		{
			if($db->execute())
			{
				return true;
			}
		}
		catch (Exception $e){}
		return false;
	}
	
	public function move($surveyId, $questionId, $newPageId)
	{
		$db = JFactory::getDbo();
		$query = $db->getQuery(true)
			->update('#__survey_questions')
			->set('page_number = '.$newPageId)
			->where('id = '.$questionId.' and survey_id = '.$surveyId);
	
		$db->setQuery($query);
		try
		{
			if($db->execute())
			{
				return true;
			}
		}
		catch (Exception $e)
		{
			if(JFactory::getUser()->authorise('core.admin'))
			{
				JFactory::getApplication()->enqueueMessage($e->getMessage());
			}
		}
		return false;
	}

	public function copy($surveyId, $questionId, $newPageId)
	{
		$db = JFactory::getDbo();
		
		// get the sort order
		$query = $db->getQuery(true)
			->select('max(sort_order)')
			->from('#__survey_questions')
			->where('page_number = '.$newPageId);
		
		$db->setQuery($query);
		$order = (int) $db->loadResult();
		
		$query = $db->getQuery(true)
			->insert('#__survey_questions')
			->columns('title, description, survey_id, question_type, page_number, sort_order, mandatory, created_by, custom_choice, orientation, min_selections, max_selections, params')
			->values(
				$db->getQuery(true)
					->select('title, description, survey_id, question_type, '.$newPageId.', '.$order.', mandatory, created_by, custom_choice, orientation, min_selections, max_selections, params')
					->from('#__survey_questions')
					->where('id = '.$questionId.' and survey_id = '.$surveyId)
				);
		
		$db->setQuery($query);
		if(!$db->execute())
		{
			return false;
		}
		
		$newqid = $db->insertid();
		
		$query = $db->getQuery(true)
			->select('id, answer_label, survey_id, question_id, answer_type, sort_order, image')
			->from('#__survey_answers')
			->where('question_id = '.$questionId.' and survey_id = '.$surveyId);
		$db->setQuery($query);
		$answers = $db->loadObjectList();
		
		foreach ($answers as &$answer)
		{
			$answer->_old_id = $answer->id;
			$answer->id = 0;
			$answer->question_id = $newqid;
			
			try
			{
				if(!$db->insertObject('#__survey_answers', $answer))
				{
					return false;
				}
			}
			catch (Exception $e)
			{
				$app->enqueueMessage('Database error. |Error='.$e->getMessage());
				return false;
			}
			
			$answer->id = $db->insertid();
		}
			
		// conditional rules
		$query = $db->getQuery(true)
			->select('survey_id, question_id, rulecontent')
			->from('#__survey_rules')
			->where('survey_id = ' . $surveyId . ' and question_id = '. $questionId);
		$db->setQuery($query);
		$rules = $db->loadObjectList();
		
		if(empty($rules))
		{
			return true;
		}
		
		foreach ($rules as $rule)
		{
			$ruleContent = json_decode($rule->rulecontent);
			$rule->question_id = $newqid;
				
			foreach ($answers as $answer)
			{
				if($answer->_old_id == $ruleContent->answer_id)
				{
					$ruleContent->answer_id = $answer->id;
				}
				else if ($answer->_old_id == $ruleContent->column_id)
				{
					$ruleContent->column_id = $answer->id;
				}
			}
			
			$rule->rulecontent = json_encode($ruleContent);
			
			try
			{
				if(!$db->insertObject('#__survey_rules', $rule))
				{
					return false;
				}
			}
			catch (Exception $e)
			{
				$app->enqueueMessage('Database error. |Error='.$e->getMessage());
				return false;
			}
		}
			
		return true;
	}
	
	public function savePageTitle($surveyId, $pageId, $newTitle)
	{
		$db = JFactory::getDbo();
		$query = $db->getQuery(true)
			->update('#__survey_pages')
			->set('title = '.$db->q($newTitle))
			->where('id = '.$pageId.' and sid = '.$surveyId);
	
		$db->setQuery($query);
		try
		{
			if($db->execute())
			{
				return true;
			}
		}
		catch (Exception $e){}
		return false;
	}
	
	public function saveSortOrder($surveyId, $pageId, $orders)
	{
		$db = JFactory::getDbo();
		$queries = array();
		
		foreach ($orders as $order)
		{
			$parts = explode('_', $order);
			if(count($parts) == 2)
			{
				$queries[] = 'when '.((int) $parts[1]).' then '.((int) $parts[0]);
			}
		}
		
		try
		{
			$query = $db->getQuery(true)
				->update('#__survey_questions')
				->set('sort_order = case id '.implode(' ', $queries).' end')
				->where('survey_id = '.$surveyId.' and page_number = '.$pageId);
			$db->setQuery($query);
			
			if($db->execute())
			{
				return true;
			}
		}
		catch (Exception $e)
		{
			if(JFactory::getUser()->authorise('core.admin'))
			{
				JFactory::getApplication()->enqueueMessage($e->getMessage());
			}
		}
		return false;
	}
	
	public function savePageSortOrder($surveyId, $orders)
	{
		$db = JFactory::getDbo();
		$queries = array();
		
		$order = 1;
		foreach ($orders as $pageId)
		{
			$queries[] = 'when '.((int) $pageId).' then '. $order;
			$order++;
		}
		
		try
		{
			$query = $db->getQuery(true)
				->update('#__survey_pages')
				->set('sort_order = case id '.implode(' ', $queries).' end')
				->where('sid = '.$surveyId);
			$db->setQuery($query);
			
			if($db->execute())
			{
				return true;
			}
		}
		catch (Exception $e)
		{
			if(JFactory::getUser()->authorise('core.admin'))
			{
				JFactory::getApplication()->enqueueMessage($e->getMessage());
			}
		}
		return false;
	}
	
	public function saveConditionalRule($rule)
	{
	    $db = JFactory::getDbo();
	    try
	    {
    	    if($db->insertObject('#__survey_rules', $rule))
    	    {
    	        return true;
    	    }
	    }
	    catch (Exception $e)
	    {
	        return false;
	    }
	    
	    return false;
	}
	
	public function deleteConditionalRule($ruleId, $surveyId, $questionId)
	{
		$db = JFactory::getDbo();
		try
		{
			$query = $db->getQuery(true)
				->delete('#__survey_rules')
				->where('id = '.$ruleId)
				->where('survey_id = '.$surveyId)
				->where('question_id = '.$questionId);
			
			$db->setQuery($query);
			if($db->execute())
			{
				return true;
			}
		}
		catch (Exception $e)
		{
			return false;
		}
		 
		return false;
	}
	
	protected function cleanCache($group = null, $client_id = 0)
	{
		parent::cleanCache('com_communitysurveys');
	}
}