<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die();

use Joomla\Utilities\ArrayHelper;
use Joomla\String\StringHelper;

require_once JPATH_SITE.'/components/com_communitysurveys/helpers/constants.php';

class CommunitySurveysModelSurveys extends JModelList
{
	protected $_item = null;

	public function __construct ($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
					'id', 'a.id',
					'title', 'a.title',
					'alias', 'a.alias',
					'checked_out', 'a.checked_out',
					'checked_out_time', 'a.checked_out_time',
					'catid', 'a.catid', 'category_title',
					'published', 'a.published', 'state', 
					'access', 'a.access', 'access_level',
					'created', 'a.created',
					'created_by', 'a.created_by',
					'ordering', 'a.ordering',
					'language', 'a.language',
					'responses', 'a.responses',
					'rating', 'rtg.rating',
					'publish_up', 'a.publish_up',
					'publish_down', 'a.publish_down'
			);
		}
		
		parent::__construct($config);
	}

	protected function populateState ($ordering = 'ordering', $direction = 'ASC')
	{
		$app = JFactory::getApplication();
		$params = JComponentHelper::getParams('com_communitysurveys');
		$user = JFactory::getUser();
		
		// List state information
		$value = $app->input->get('limit', $app->get('list_limit', 20), 'uint');
		$this->setState('list.limit', $value);
		
		$limitStart = $app->input->get('limitstart', 0, 'uint');
		$this->setState('list.start', $limitStart);
		
		$authorId = $app->input->get('filter_author_id', 0, 'uint');
		$this->setState('filter.author_id', $authorId);
		
		$parentId = $app->input->get('id', 0, 'uint');
		if ($parentId)
		{
			$this->setState('filter.category_id', $parentId);
		}
		
		$categories = $app->input->getArray(array('catid' => 'array'));
		$this->setState('filter.category_id', $categories['catid']);
		
		$orderCol = $app->input->get('filter_order');
		if ( empty($orderCol) || ! in_array($orderCol, $this->filter_fields))
		{
			$orderby = $params->get('orderby_sec', 'rdate');
			$orderCol = CommunitySurveysHelperQuery::orderbySecondary($orderby);
			$this->setState('list.ordering', $orderCol);
		} 
		else
		{
			$listOrder = $app->input->get('filter_order_Dir', 'DESC');
			if (! in_array(strtoupper($listOrder), array('ASC', 'DESC', '')))
			{
				$listOrder = 'DESC';
			}
			
			$this->setState('list.ordering', $orderCol . ' ' . $listOrder);
		}
		
		$filter = $app->input->get('list_filter', '', 'string');
		if (strlen($filter) > 1 || is_numeric($filter))
		{
			$this->setState('list.filter', $filter);
			
			$filterAllKeywords = $app->input->get('list_filter_all_keywords', 0, 'int');
			$this->setState('list.filter_all_keywords', $filterAllKeywords);
			
			$filterField = $app->input->get('list_filter_field', 'title', 'word');
			$this->setState('list.filter_field', $filterField);
		}
		
		if (! $user->authorise('core.edit.state', 'com_communitysurveys') && ! $user->authorise('core.edit', 'com_communitysurveys'))
		{
			// Filter on published for those who do not have edit or edit.state rights.
			$this->setState('filter.published', 1);
		}
		
		$this->setState('filter.language', JLanguageMultilang::isEnabled());
		$this->setState('params', $params);
		
		// Process show_noauth parameter
		if (! $params->get('show_noauth'))
		{
			$this->setState('filter.access', true);
		}
		else
		{
			$this->setState('filter.access', false);
		}
		
		$this->setState('layout', $app->input->getString('layout'));
	}

	protected function getStoreId ($id = '')
	{
		// Compile the store id.
		$id .= ':' . serialize($this->getState('filter.published'));
		$id .= ':' . $this->getState('filter.access');
		$id .= ':' . $this->getState('filter.featured');
		$id .= ':' . $this->getState('filter.survey_id');
		$id .= ':' . $this->getState('filter.survey_id.include');
		$id .= ':' . serialize($this->getState('filter.category_id'));
		$id .= ':' . $this->getState('filter.category_id.include');
		$id .= ':' . serialize($this->getState('filter.author_id'));
		$id .= ':' . $this->getState('filter.author_id.include');
		$id .= ':' . serialize($this->getState('filter.author_alias'));
		$id .= ':' . $this->getState('filter.author_alias.include');
		$id .= ':' . $this->getState('filter.date_filtering');
		$id .= ':' . $this->getState('filter.date_field');
		$id .= ':' . $this->getState('filter.start_date_range');
		$id .= ':' . $this->getState('filter.end_date_range');
		$id .= ':' . $this->getState('filter.relative_date');
		
		return parent::getStoreId($id);
	}

	protected function getListQuery ()
	{
		// Get the current user for authorisation checks
		$user = $this->getUser();
		$params = JComponentHelper::getParams('com_communitysurveys');
		$userDisplayName = $params->get('user_display_name', 'name');
		
		// Create a new query object.
		$db = $this->getDbo();
		$query = $db->getQuery(true);
		
		// Select the required fields from the table.
		$query->select(
				$this->getState('list.select', 
						'a.id, a.title, a.alias, a.description, a.checked_out, a.checked_out_time, a.catid, a.created, a.created_by, a.responses, a.language, a.private,' .
								 'CASE WHEN a.modified = ' . $db->quote($db->getNullDate()) . ' THEN a.created ELSE a.modified END as modified, a.modified_by, uam.'.$userDisplayName.' as modified_by_name,' .
								 'CASE WHEN a.publish_up = ' . $db->quote($db->getNullDate()) . ' THEN a.created ELSE a.publish_up END as publish_up, a.publish_down, ' .
								 'a.attribs, a.metadata, a.metakey, a.metadesc, a.access, ' . $query->length('a.description') . ' AS readmore'));
		
		// Process an Archived survey layout
		if ($this->getState('filter.published') == 2)
		{
			// If badcats is not null, this means that the survey is inside an
			// archived category
			// In this case, the state is set to 2 to indicate Archived (even if
			// the survey state is Published)
			$query->select($this->getState('list.select', 'CASE WHEN badcats.id is null THEN a.published ELSE 2 END AS state'));
		}
		else
		{
			/*
			 * Process non-archived layout If badcats is not null, this means
			 * that the survey is inside an unpublished category In this case,
			 * the state is set to 0 to indicate Unpublished (even if the
			 * survey state is Published)
			 */
			$query->select($this->getState('list.select', 'CASE WHEN badcats.id is not null THEN 0 ELSE a.published END AS state'));
		}
		
		$query->from('#__survey_surveys AS a');
		
		// join over ratings rable
		$query
			->select('rtg.rating')
			->join('LEFT', '#__corejoomla_rating AS rtg on rtg.asset_id = '.CS_ASSET_ID.' and rtg.item_id = a.id');
		
		// Join over the categories.
		$query
			->select('c.title AS category_title, c.path AS category_route, c.access AS category_access, c.alias AS category_alias')
			->join('LEFT', '#__categories AS c ON c.id = a.catid');
		
		// Join over the users for the author and modified_by names.
		$query
			->select("ua.".$userDisplayName." AS author, ua.email AS author_email")
			->join('LEFT', '#__users AS ua ON ua.id = a.created_by')
			->join('LEFT', '#__users AS uam ON uam.id = a.modified_by');
		
		// Join over the categories to get parent category titles
		$query
			->select('parent.title as parent_title, parent.id as parent_id, parent.path as parent_route, parent.alias as parent_alias')
			->join('LEFT', '#__categories as parent ON parent.id = c.parent_id');
		
		// Join to check for category published state in parent categories up the tree
		$query->select('c.published, CASE WHEN badcats.id is null THEN c.published ELSE 0 END AS parents_published');
		$subquery = 'SELECT cat.id as id FROM #__categories AS cat JOIN #__categories AS parent ';
		$subquery .= 'ON cat.lft BETWEEN parent.lft AND parent.rgt ';
		$subquery .= 'WHERE parent.extension = ' . $db->quote('com_communitysurveys');
		
		if ($this->getState('filter.published') == 2)
		{
			// Find any up-path categories that are archived
			// If any up-path categories are archived, include all children in
			// archived layout
			$subquery .= ' AND parent.published = 2 GROUP BY cat.id ';
			
			// Set effective state to archived if up-path category is archived
			$publishedWhere = 'CASE WHEN badcats.id is null THEN a.published ELSE 2 END';
		}
		else
		{
			// Find any up-path categories that are not published
			// If all categories are published, badcats.id will be null, and we
			// just use the survey state
			$subquery .= ' AND parent.published != 1 GROUP BY cat.id ';
			
			// Select state to unpublished if up-path category is unpublished
			$publishedWhere = 'CASE WHEN badcats.id is null THEN a.published ELSE 0 END';
		}
		
		$query->join('LEFT', '(' . $subquery . ') AS badcats ON badcats.id = c.id');
		
		// Filter by access level.
		if ($access = $this->getState('filter.access'))
		{
			$groups = implode(',', $user->getAuthorisedViewLevels());
			$query->where('a.access IN (' . $groups . ')')->where('c.access IN (' . $groups . ')');
		}
		

		if (! $user->authorise('core.edit.state', 'com_communitysurveys') && ! $user->authorise('core.edit', 'com_communitysurveys'))
		{
		    // Filter on published for those who do not have edit or edit.state rights.
		    $query->where('('.$publishedWhere . ' = 1 OR (a.created_by = '.$user->id.' AND a.published = 3))');
		}
		
// 		// Filter by published state
// 		$published = $this->getState('filter.published');
		
// 		if (is_numeric($published))
// 		{
// 			// Use survey state if badcats.id is null, otherwise, force 0 for unpublished
// 			$query->where($publishedWhere . ' = ' . (int) $published);
// 		}
// 		elseif (is_array($published))
// 		{
// 			$published = ArrayHelper::toInteger($published);
// 			$published = implode(',', $published);
			
// 			// Use survey state if badcats.id is null, otherwise, force 0 for unpublished
// 			$query->where($publishedWhere . ' IN (' . $published . ')');
// 		}
		
		// Filter by a single or group of surveys.
		$surveyId = $this->getState('filter.survey_id');
		
		if (is_numeric($surveyId))
		{
			$type = $this->getState('filter.survey_id.include', true) ? '= ' : '<> ';
			$query->where('a.id ' . $type . (int) $surveyId);
		}
		elseif (is_array($surveyId))
		{
			$surveyId = ArrayHelper::toInteger($surveyId);
			$surveyId = implode(',', $surveyId);
			$type = $this->getState('filter.survey_id.include', true) ? 'IN' : 'NOT IN';
			$query->where('a.id ' . $type . ' (' . $surveyId . ')');
		}
		
		// Filter by a single or group of categories
		$categoryId = $this->getState('filter.category_id');
		
		if (is_numeric($categoryId) && $categoryId)
		{
			$type = $this->getState('filter.category_id.include', true) ? '= ' : '<> ';
			
			// Add subcategory check
			$includeSubcategories = $this->getState('filter.subcategories', false);
			$categoryEquals = 'a.catid ' . $type . (int) $categoryId;
			
			if ($includeSubcategories)
			{
				$levels = (int) $this->getState('filter.max_category_levels', '1');
				
				// Create a subquery for the subcategory list
				$subQuery = $db->getQuery(true)
					->select('sub.id')
					->from('#__categories as sub')
					->join('INNER', '#__categories as this ON sub.lft > this.lft AND sub.rgt < this.rgt')
					->where('this.id = ' . (int) $categoryId);
				
				if ($levels >= 0)
				{
					$subQuery->where('sub.level <= this.level + ' . $levels);
				}
				
				// Add the subquery to the main query
				$query->where('(' . $categoryEquals . ' OR a.catid IN (' . $subQuery->__toString() . '))');
			}
			else
			{
				$query->where($categoryEquals);
			}
		}
		elseif (is_array($categoryId) && (count($categoryId) > 0))
		{
			$categoryId = ArrayHelper::toInteger($categoryId);
			$categoryId = implode(',', $categoryId);
			
			if (! empty($categoryId))
			{
				$type = $this->getState('filter.category_id.include', true) ? 'IN' : 'NOT IN';
				$query->where('a.catid ' . $type . ' (' . $categoryId . ')');
			}
		}
		
		if (is_numeric($categoryId) && $categoryId)
		{
		    if( ! $user->authorise('core.manage', 'com_communitysurveys.category.'.$categoryId) )
		    {
		        $query->where('(a.created_by = '.$user->id.' OR a.private = 0)');
		    }
		}
		else 
		{
		    if( ! $user->authorise('core.manage', 'com_communitysurveys') )
		    {
		        $query->where('(a.created_by = '.$user->id.' OR a.private = 0)');
		    }
		}
		
		// Filter by author
		$authorId = $this->getState('filter.author_id');
		
		if (is_numeric($authorId) && $authorId)
		{
			$type = $this->getState('filter.author_id.include', true) ? '= ' : '<> ';
			$query->where('a.created_by ' . $type . (int) $authorId);
		}
		elseif (is_array($authorId))
		{
			$authorId = ArrayHelper::toInteger($authorId);
			$authorId = implode(',', $authorId);
			
			if ($authorId)
			{
				$type = $this->getState('filter.author_id.include', true) ? 'IN' : 'NOT IN';
				$query->where('a.created_by ' . $type . ' (' . $authorId . ')');
			}
		}
		
		// Filter by start and end dates.
		if ((! $user->authorise('core.edit.state', 'com_communitysurveys')) && (! $user->authorise('core.edit', 'com_communitysurveys')))
		{
			$nullDate = $db->quote($db->getNullDate());
			$nowDate = $db->quote(JFactory::getDate()->toSql());
			
			$query
				->where('(a.publish_up = ' . $nullDate . ' OR a.publish_up <= ' . $nowDate . ')')
				->where('(a.publish_down = ' . $nullDate . ' OR a.publish_down >= ' . $nowDate . ')');
		}
		
		// Filter by Date Range or Relative Date
		$dateFiltering = $this->getState('filter.date_filtering', 'off');
		$dateField = $this->getState('filter.date_field', 'a.created');
		
		switch ($dateFiltering)
		{
			case 'range':
				$startDateRange = $db->quote($this->getState('filter.start_date_range', $nullDate));
				$endDateRange = $db->quote($this->getState('filter.end_date_range', $nullDate));
				$query->where('(' . $dateField . ' >= ' . $startDateRange . ' AND ' . $dateField . ' <= ' . $endDateRange . ')');
				break;
			
			case 'relative':
				$relativeDate = (int) $this->getState('filter.relative_date', 0);
				$query->where($dateField . ' >= DATE_SUB(' . $nowDate . ', INTERVAL ' . $relativeDate . ' DAY)');
				break;
			
			case 'off':
			default:
				break;
		}
		
		// Process the filter for list views with user-entered filters
		if ($filter = $this->getState('list.filter'))
		{
			// Clean filter variable
			$filter = StringHelper::strtolower($filter);
			$hitsFilter = (int) $filter;
			$createdByFilter = (int) $filter;
			$idFilter = (int) $filter;
			$filterField = $this->getState('list.filter_field', 'title');
			
			switch ($filterField)
			{
				case 'createdby':
					$query->where('a.created_by = ' . $createdByFilter . ' ');
					break;
				
				case 'author':
					$query->where('ua.name LIKE ' . $db->q('%' . $db->escape($filter, true) . '%', false) . ' ');
					break;
				
				case 'hits':
					$query->where('a.responses >= ' . $hitsFilter . ' ');
					break;
					
				case 'id':
				    $query->where('a.id = ' . $idFilter . ' ');
				    break;
				
				case 'title':
				default:
				    if(is_numeric($filter))
				    {
				        $idFilter = (int) $filter;
				        $query->where('a.id = ' . $idFilter . ' ');
				    }
				    else 
				    {
					   $stopwords = array(
							"a", "about", "above", "above", "across", "after", "afterwards", "again", "against", "all", "almost", "alone", "along", "already", "also","although","always",
							"am","among", "amongst", "amoungst", "amount",  "an", "and", "another", "any","anyhow","anyone","anything","anyway", "anywhere", "are", "around", "as",  "at",
							"back","be","became", "because","become","becomes", "becoming", "been", "before", "beforehand", "behind", "being", "below", "beside", "besides", "between",
							"beyond", "bill", "both", "bottom","but", "by", "call", "can", "cannot", "cant", "co", "con", "could", "couldnt", "cry", "de", "describe", "detail", "do", "done",
							"down", "due", "during", "each", "eg", "eight", "either", "eleven","else", "elsewhere", "empty", "enough", "etc", "even", "ever", "every", "everyone", "everything",
							"everywhere", "except", "few", "fifteen", "fify", "fill", "find", "fire", "first", "five", "for", "former", "formerly", "forty", "found", "four", "from", "front",
							"full", "further", "get", "give", "go", "had", "has", "hasnt", "have", "he", "hence", "her", "here", "hereafter", "hereby", "herein", "hereupon", "hers", "herself",
							"him", "himself", "his", "how", "however", "hundred", "ie", "if", "in", "inc", "indeed", "interest", "into", "is", "it", "its", "itself", "keep", "last", "latter",
							"latterly", "least", "less", "ltd", "made", "many", "may", "me", "meanwhile", "might", "mill", "mine", "more", "moreover", "most", "mostly", "move", "much", "must",
							"my", "myself", "name", "namely", "neither", "never", "nevertheless", "next", "nine", "no", "nobody", "none", "noone", "nor", "not", "nothing", "now", "nowhere", "of",
							"off", "often", "on", "once", "one", "only", "onto", "or", "other", "others", "otherwise", "our", "ours", "ourselves", "out", "over", "own","part", "per", "perhaps",
							"please", "put", "rather", "re", "same", "see", "seem", "seemed", "seeming", "seems", "serious", "several", "she", "should", "show", "side", "since", "sincere", "six",
							"sixty", "so", "some", "somehow", "someone", "something", "sometime", "sometimes", "somewhere", "still", "such", "system", "take", "ten", "than", "that", "the", "their",
							"them", "themselves", "then", "thence", "there", "thereafter", "thereby", "therefore", "therein", "thereupon", "these", "they", "thickv", "thin", "third", "this", "those",
							"though", "three", "through", "throughout", "thru", "thus", "to", "together", "too", "top", "toward", "towards", "twelve", "twenty", "two", "un", "under", "until", "up",
							"upon", "us", "very", "via", "was", "we", "well", "were", "what", "whatever", "when", "whence", "whenever", "where", "whereafter", "whereas", "whereby", "wherein",
							"whereupon", "wherever", "whether", "which", "while", "whither", "who", "whoever", "whole", "whom", "whose", "why", "will", "with", "within", "without", "would", "yet",
							"you", "your", "yours", "yourself", "yourselves", "the");
							
                        $keywords = array_diff(explode(' ', $filter), $stopwords);
                        $filters = array();
    					
    					foreach ($keywords as $keyword)
    					{
    						if (strlen($keyword) > 2)
    						{
    							$filters[] = 'LOWER( a.title ) LIKE ' . $db->q('%' . $db->escape($keyword, true) . '%', false);
    						}
    					}
    					// Default to 'title' if parameter is not valid
    					if ($filterAllKeywords = $this->getState('list.filter_all_keywords'))
    					{
    						$query->where('(' . implode(' AND ', $filters) . ')');
    					}
    					else
    					{
    						$query->where('(' . implode(' OR ', $filters) . ')');
    					}
				    }
				    					
					break;
			}
		}
		
		// Filter by language
		if ($this->getState('filter.language'))
		{
			$query->where('a.language in (' . $db->quote(JFactory::getLanguage()->getTag()) . ',' . $db->quote('*') . ')');
		}
		
		// Add the list ordering clause.
		$query->order($this->getState('list.ordering', 'a.created DESC'));
		
// 		echo $query->dump();
// 		jexit();
		return $query;
	}

	public function getItems ()
	{
		$items = parent::getItems();
		$user = $this->getUser();
		$app = JFactory::getApplication();
		$userId = $user->id;
		$guest = $user->guest;
		$groups = $user->getAuthorisedViewLevels();
		$input = $app->input;
		
		// Get the global params
		$globalParams = JComponentHelper::getParams('com_communitysurveys', true);
		
		// Convert the parameter fields into objects.
		if(!empty($items))
		{
    		foreach ($items as &$item)
    		{
    			$item->slug = $item->alias ? ($item->id . ':' . $item->alias) : $item->id;
    			$item->parent_slug = ($item->parent_alias) ? ($item->parent_id . ':' . $item->parent_alias) : $item->parent_id;
    			
    			// No link for ROOT category
    			if ($item->parent_alias == 'root')
    			{
    				$item->parent_slug = null;
    			}
    			
    			$item->catslug = $item->category_alias ? ($item->catid . ':' . $item->category_alias) : $item->catid;
    			
    			$surveyParams = new JRegistry();
    			$surveyParams->loadString($item->attribs);
    			
    			// Unpack readmore and layout params
    			$item->alternative_readmore = $surveyParams->get('alternative_readmore');
    			$item->layout = $surveyParams->get('layout');
    			
    			$item->params = clone $this->getState('params');
    			
    			/*
    			 * For lists, survey params override menu item params only if menu
    			 * param = 'use_survey' Otherwise, menu item params control the
    			 * layout If menu item is 'use_survey' and there is no survey
    			 * param,
    			 * use global
    			 */
    			if (
    					($input->getString('layout') == 'blog') || 
    					($input->getString('view') == 'featured') ||
    					($this->getState('params')->get('layout_type') == 'blog')
    			)
    			{
    				// Create an array of just the params set to 'use_survey'
    				$menuParamsArray = $this->getState('params')->toArray();
    				$surveyArray = array();
    				
    				foreach ($menuParamsArray as $key => $value)
    				{
    					if ($value === 'use_survey')
    					{
    						// If the survey has a value, use it
    						if ($surveyParams->get($key) != '')
    						{
    							// Get the value from the survey
    							$surveyArray[$key] = $surveyParams->get($key);
    						}
    						else
    						{
    							// Otherwise, use the global value
    							$surveyArray[$key] = $globalParams->get($key);
    						}
    					}
    				}
    				
    				// Merge the selected survey params
    				if (count($surveyArray) > 0)
    				{
    					$surveyParams = new JRegistry();
    					$surveyParams->loadArray($surveyArray);
    					$item->params->merge($surveyParams);
    				}
    			}
    			else
    			{
    				// For non-blog layouts, merge all of the survey params
    				$item->params->merge($surveyParams);
    			}
    			
    			// Get display date
    			switch ($item->params->get('list_show_date'))
    			{
    				case 'modified':
    					$item->displayDate = $item->modified;
    					break;
    				
    				case 'published':
    					$item->displayDate = ($item->publish_up == 0) ? $item->created : $item->publish_up;
    					break;
    				
    				case 'created':
    					$item->displayDate = $item->created;
    					break;
    			}
    			
    			// Compute the asset access permissions.
    			// Technically guest could edit an survey, but lets not check that
    			// to improve performance a little.
    			if (! $guest)
    			{
    				$asset = 'com_communitysurveys.category.' . $item->catid;
    				
    				// Check general edit permission first.
    				if ($user->authorise('core.edit', $asset))
    				{
    					$item->params->set('access-edit', true);
    				}
    				
    				// Now check if edit.own is available.
    				elseif (! empty($userId) && $user->authorise('core.edit.own', $asset))
    				{
    					// Check for a valid user and that they are the owner.
    					if ($userId == $item->created_by)
    					{
    						$item->params->set('access-edit', true);
    					}
    				}
    			}
    			
    			$access = $this->getState('filter.access');
    			
    			if ($access)
    			{
    				// If the access filter has been set, we already have only the
    				// surveys this user can view.
    				$item->params->set('access-view', true);
    			}
    			else
    			{
    				// If no access filter is set, the layout takes some
    				// responsibility for display of limited information.
    				if ($item->catid == 0 || $item->category_access === null)
    				{
    					$item->params->set('access-view', in_array($item->access, $groups));
    				}
    				else
    				{
    					$item->params->set('access-view', in_array($item->access, $groups) && in_array($item->category_access, $groups));
    				}
    			}
    			
    			// Get the tags
    			// $item->tags = new JHelperTags();
    			// $item->tags->getItemTags('com_communitysurveys.survey',
    			// $item->id);
    		}
    		
    		reset($items);
		}
		
		return $items;
	}

	public function getPagination ()
	{
		$page = parent::getPagination();
		$view = JFactory::getApplication()->input->getCmd('view');
		
		if ($view == 'surveys')
		{
			$page->setAdditionalUrlParam('view', 'surveys');
			
			if ($this->state->get('list.ordering') == 'hits')
			{
				$page->setAdditionalUrlParam('filter_order', $this->state->get('list.ordering'));
				$page->setAdditionalUrlParam('filter_order_Dir', $this->state->get('list.direction'));
			}
		}
		
		return $page;
	}

	public function getCategory ()
	{
		if (! is_object($this->_item))
		{
			if (isset($this->state->params))
			{
				$params = $this->state->params;
				$options = array();
				$options['countItems'] = $params->get('show_cat_num_surveys', 1) || ! $params->get('show_empty_categories_cat', 0);
			}
			else
			{
				$options['countItems'] = 0;
			}
			
			$categories = JCategories::getInstance('CommunitySurveys', $options);
			$this->_item = $categories->get($this->getState('filter.category_id', 'root'));
			
			// Compute selected asset permissions.
			if (is_object($this->_item))
			{
				$user = $this->getUser();
				$asset = 'com_communitysurveys.category.' . $this->_item->id;
				
				// Check general create permission.
				if ($user->authorise('core.create', $asset))
				{
					$this->_item->getParams()->set('access-create', true);
				}
				
				// TODO: Why aren't we lazy loading the children and siblings?
				$this->_children = $this->_item->getChildren();
				$this->_parent = false;
				
				if ($this->_item->getParent())
				{
					$this->_parent = $this->_item->getParent();
				}
				
				$this->_rightsibling = $this->_item->getSibling();
				$this->_leftsibling = $this->_item->getSibling(false);
			}
			else
			{
				$this->_children = false;
				$this->_parent = false;
			}
		}
		
		return $this->_item;
	}
	
	private function getUser()
	{
		$user = $this->getState('user');
		if(empty($user))
		{
			$user = JFactory::getUser();
		}
	
		return $user;
	}
}
