<?php
/**
 * @package     corejoomla.site
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die();

use Joomla\Utilities\ArrayHelper;
use Joomla\String\StringHelper;

require_once JPATH_ADMINISTRATOR . '/components/com_communitysurveys/models/invitebase.php';

class CommunitySurveysModelInviteBase extends JModelList
{
	protected $_pagination = null;
	
	public function __construct ($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
					'id', 'a.id',
					'group_id', 'a.group_id'
			);
		}
	
		parent::__construct($config);
	}

	protected function populateState($ordering = null, $direction = null)
	{
		$app = JFactory::getApplication();
	
		// Adjust the context to support modal layouts.
		if ($layout = $app->input->get('ui_layout'))
		{
			$this->context .= '.' . $layout;
		}
		
		$params = JComponentHelper::getParams('com_communitysurveys');
		$this->setState('params', $params);
		
		$pk = $app->input->getInt('id');
		$this->setState($this->getName() . '.id', $pk);

		$limit = $app->input->getInt('limit', 20);
		$this->setState('contacts.limit', $limit);
		
		$limitstart = $app->input->getInt('limitstart', 0);
		$this->setState('contacts.limitstart', $limitstart);
		
		parent::populateState('a.name', 'asc');
	}
	
	public function getItem($pk = null)
	{
		$surveyModel	= JModelLegacy::getInstance( 'survey', 'CommunitySurveysModel' );
		$state 			= $surveyModel->getState(); // access the state first so that it can be modified
		$surveyId 		= $this->getState($this->getName() . '.id');
		
		$surveyModel->setState('filter.survey_id', $surveyId);
		$survey = $surveyModel->getItem($surveyId);
		
		return $survey;
	}
	
	function getKeys($surveyId = null, $limitstart = 0, $limit = 20)
	{
		$user 			= JFactory::getUser();
		$db 			= JFactory::getDbo();
		$surveyId 		= $surveyId ? $surveyId : $this->getState($this->getName() . '.id');

		$query = $db->getQuery(true)
			->select('a.key_name, a.survey_id, a.response_id, a.response_status, a.created')
			->from('#__survey_keys AS a')
			->where('a.survey_id = '.(int) $surveyId)
			->order('a.created desc');
		
		if(!$user->authorise('core.manage', 'com_communitysurveys.survey.'.$surveyId))
		{
			$query
				->join('inner', '#__survey_surveys s on s.id = a.survey_id')
				->where('s.created_by = '.$user->id);
		}
		
		try 
		{
			$db->setQuery($query, $limitstart, $limit);
			$keys = $db->loadObjectList();
		}
		catch (Exception $e){}
	
		return !empty($keys) ? $keys : array();
	}

	/**
	 * Method to get the record form.
	 *
	 * @param   array      $data        Data for the form.
	 * @param   boolean    $loadData    True if the form is to load its own data (default case), false if not.
	 *
	 * @return  mixed  A JForm object on success, false on failure
	 */
	public function getForm($data = array(), $loadData = true)
	{
		// Get the form.
		$form = $this->loadForm('com_communitysurveys.invite', 'invite', array('control' => 'jform', 'load_data' => $loadData));
		if (empty($form))
		{
			return false;
		}
		
		return $form;
	}
	
	protected function loadFormData()
	{
		$db 		= JFactory::getDbo();
		$language 	= JFactory::getLanguage();
		$data 		= array();
		
		$query = $db->getQuery(true)
			->select('title, description')
			->from('#__survey_email_templates')
			->where('published = 1')
			->where('email_type = ' . $db->q('com_communitysurveys.invitation'))
			->where('language in (' . $db->q($language->getTag()) . ',' . $db->q('*') . ')' );
		
		$db->setQuery($query, 0, 1);

		$data['id'] = 1;
		$data['title'] = JText::_('COM_COMMUNITYSURVEYS_INVITE_EMAIL_SUBJECT_CONTENT');
		$data['description'] = JText::_('COM_COMMUNITYSURVEYS_INVITE_EMAIL_BODY_CONTENT');
		
		try 
		{
			$template = $db->loadObject();
			
			if(!empty($template))
			{
				$data['title'] = $template->title;
				$data['description'] = $template->description;
			}
		}
		catch (Exception $e){}
		return $data;
	}

	/**
	 * Returns a Table object, always creating it.
	 *
	 * @param   type      The table type to instantiate
	 * @param   string    A prefix for the table class name. Optional.
	 * @param   array     Configuration array for model. Optional.
	 *
	 * @return  JTable    A database object
	 */
	public function getTable($type = 'Survey', $prefix = 'CommunitySurveysTable', $config = array())
	{
		return JTable::getInstance($type, $prefix, $config);
	}
	
	public function inviteUserGroups($limitstart = 0, $limit = 5000)
	{
		$app		= JFactory::getApplication();
		$db 		= JFactory::getDbo();
		$user		= JFactory::getUser();
		$surveyId 	= $this->getState($this->getName() . '.id');
		$data		= $app->input->post->get('jform', array(), 'array');
		
		if(empty($data['group_id']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_PLEASE_SELECT_USER_GROUPS_TO_INVITE'), 500);
		}
		
		if($user->guest || !$user->authorise('core.manage', 'com_communitysurveys.survey.'.$surveyId))
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
		}
		
		$data['group_id'] = ArrayHelper::toInteger($data['group_id']);
		$query = $db->getQuery(true)
			->select('id, email, name')
			->from('#__users');

		$subQuery = $db->getQuery(true)
			->select('distinct(user_id)')
			->from('#__user_usergroup_map')
			->where('group_id in ('.implode(',', $data['group_id']).')');
			
		$query
			->where('id in ('.$subQuery->__toString().')')
			->where('block = 0')
			->order('id asc');
			
		$db->setQuery($query, $limitstart, $limit);
		$users = $db->loadObjectList();
		
		if(!empty($users))
		{
			$this->sendInvitations($data, $users);
		}
		
		return count($users);
	}
	
	public function inviteRegisteredUsers()
	{
		$app		= JFactory::getApplication();
		$db 		= JFactory::getDbo();
		$user		= JFactory::getUser();
		$surveyId 	= $this->getState($this->getName() . '.id');
		$data		= $app->input->post->get('jform', array(), 'array');
		
		$data['users'] = ArrayHelper::toInteger($data['users']);
		if(empty($data['users']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_PLEASE_SELECT_FEW_USERS_INVITE'), 500);
		}
		
		if($user->guest || !$user->authorise('core.manage', 'com_communitysurveys.survey.'.$surveyId))
		{
			$query = $db->getQuery(true);
			$query
				->select('created_by')
				->from('#__survey_surveys')
				->where('id = '.$surveyId);
			
			$db->getQuery(true);
			$createdBy = (int) $db->loadResult();
			
			if($createdBy != $user->id)
			{
				throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
			}
		}
		
		$query = $db->getQuery(true)
			->select('id, email, name')
			->from('#__users')
			->where('id in ('.implode(',', $data['users']).')')
			->where('block = 0')
			->order('id asc');
			
		$db->setQuery($query);
		$users = $db->loadObjectList();
		
		if(!empty($users))
		{
			$this->sendInvitations($data, $users);
		}
		
		return count($users);
	}
	
	public function inviteCommunityGroups($limitstart = 0, $limit = 20)
	{
		if(file_exists(JPATH_ROOT.'/components/com_community/userpoints.php')) 
		{
			return $this->inviteJomSocialUserGroups($limitstart, $limit);
		}
		else if(file_exists(JPATH_ADMINISTRATOR.'/components/com_easysocial/includes/foundry.php'))
		{
			return $this->inviteEasySocialUserGroups($limitstart, $limit);
		}
	}
	
	protected function inviteJomSocialUserGroups($limitstart = 0, $limit = 20)
	{
		$app		= JFactory::getApplication();
		$db 		= JFactory::getDbo();
		$user		= JFactory::getUser();
		$surveyId 	= $this->getState($this->getName() . '.id');
		$data		= $app->input->post->get('jform', array(), 'array');
	
		if(empty($data['gid']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_PLEASE_SELECT_USER_GROUPS_TO_INVITE'), 500);
		}
	
		if($user->guest || !$user->authorise('core.manage', 'com_communitysurveys.survey.'.$surveyId))
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
		}
	
		$data['gid'] = ArrayHelper::toInteger($data['gid']);
		
		// First check if the user is member of the selected groups
		$query = $db->getQuery(true)
			->select('distinct(m.groupid)')
			->from('#__community_groups_members m')
			->where('m.memberid = '.$user->id)
			->where('m.approved = 1');
		
		$db->setQuery($query);
		$groups = $db->loadColumn();
		$data['gid'] = array_intersect($data['gid'], $groups);
		
		if(empty($data['gid']))
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
		}
		
		// Now get all users in the groups
		$query = $db->getQuery(true)
			->select('id, email, name')
			->from('#__users');
	
		$subQuery = $db->getQuery(true)
			->select('memberid')
			->from('#__community_groups_members')
			->where('groupid in ('.implode(',', $data['gid']).')');
			
		$query
			->where('id in ('.$subQuery->__toString().')')
			->where('block = 0')
			->order('id asc');
			
		$db->setQuery($query, $limitstart, $limit);
		$users = $db->loadObjectList();
	
		if(!empty($users))
		{
			$this->sendInvitations($data, $users);
		}
	
		return count($users);
	}
	
	protected function inviteEasySocialUserGroups($limitstart = 0, $limit = 20)
	{
		$app		= JFactory::getApplication();
		$user		= JFactory::getUser();
		$surveyId 	= $this->getState($this->getName() . '.id');
		$data		= $app->input->post->get('jform', array(), 'array');

		if(empty($data['gid']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_PLEASE_SELECT_USER_GROUPS_TO_INVITE'), 500);
		}
	
		if($user->guest || !$user->authorise('core.manage', 'com_communitysurveys.survey.'.$surveyId))
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
		}
	
		$data['gid'] = ArrayHelper::toInteger($data['gid']);
	
		// First check if the user is member of the selected groups
		require_once JPATH_ADMINISTRATOR.'/components/com_easysocial/includes/foundry.php';
		$model = FD::model('Groups');
		
		foreach ($data['gid'] as $groupId)
		{
			if(!$model->isMember($user->id, $groupId))
			{
				throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
			}
		}
	
		// Now get all users in the groups
		$users = array();
		foreach ($data['gid'] as $groupId)
		{
			$members = $model->getMembers($groupId);
			if(!empty($members))
			{
				foreach ($members as $member)
				{
					$newUser = new stdClass();
					$newUser->id = $member->id;
					$newUser->name = $member->name;
					$newUser->email = $member->email;
					
					$users[] = $newUser;
				}
			}
		}
	
		if(!empty($users))
		{
			$this->sendInvitations($data, $users);
		}
	
		return count($users);
	}
	
	public function inviteContactGroups()
	{
		$app		= JFactory::getApplication();
		$db 		= JFactory::getDbo();
		$user		= JFactory::getUser();
		$data		= $app->input->post->get('jform', array(), 'array');
		
		if(empty($data['cgroup_id']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_PLEASE_SELECT_USER_GROUPS_TO_INVITE'), 500);
		}

		$data['cgroup_id'] = ArrayHelper::toInteger($data['cgroup_id']);
	
		// First check if the user is member of the selected groups
		$query = $db->getQuery(true)
			->select('id')
			->from('#__survey_contactgroups g')
			->where('g.created_by = '.$user->id);
	
		$db->setQuery($query);
		$groups = $db->loadColumn();
		$data['cgroup_id'] = array_intersect($data['cgroup_id'], $groups);
	
		if(empty($data['cgroup_id']))
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
		}
	
		// Now get all users in the groups
		$query = $db->getQuery(true)
			->select('id, email, name')
			->from('#__survey_contacts');
	
		$subQuery = $db->getQuery(true)
			->select('contact_id')
			->from('#__survey_contact_group_map')
			->where('group_id in ('.implode(',', $data['cgroup_id']).')');
			
		$query
			->where('id in ('.$subQuery->__toString().')')
			->order('id asc');
			
		$db->setQuery($query, 0, 1000);
		$users = $db->loadObjectList();
	
		if(!empty($users))
		{
			$this->sendInvitations($data, $users);
		}
	
		return count($users);
	}
	
	protected function sendInvitations($data, $users)
	{
		$config 				= JFactory::getConfig();
		$surveyId 				= $this->getState($this->getName() . '.id');
		$sitename 				= $config->get('sitename');
		
		JLoader::import('survey', JPATH_ADMINISTRATOR.'/components/com_communitysurveys/models');
		$surveyModel			= JModelLegacy::getInstance( 'survey', 'CommunitySurveysModel' );
		$survey					= $surveyModel->getItem($surveyId);

		if(empty($survey))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_NOT_FOUND'), 500);
		}
		
// 		$surveyUrl				= SurveyHelper::buildSefSurveyUrl(CommunitySurveysHelperRoute::getSurveyRoute($survey->id, $survey->catid, $survey->language));
		$description 			= StringHelper::str_ireplace('{SITENAME}', 		$sitename, 			$data['description']);
		$description 			= StringHelper::str_ireplace('{SURVEY_TITLE}', 	$survey->title,		$description);
// 		$description 			= StringHelper::str_ireplace('{SURVEY_URL}', 	$surveyUrl, 		$description);
// 		$description			= StringHelper::str_ireplace('{NAME}',			$user->name,		$description);

		$subject				= StringHelper::str_ireplace('{SURVEY_TITLE}', 	$survey->title,		$data['title']);
		
		$message				= new stdClass();
		$message->asset_id 		= $survey->id;
		$message->asset_name 	= 'com_communitysurveys.invitation';
		$message->subject 		= $subject;
		$message->description 	= $description;
		
		// load survey urls
		JLoader::import('response', JPATH_COMPONENT_SITE.'/models');
		$responseModel = JModelLegacy::getInstance( 'response', 'CommunitySurveysModel' );
		$responseModel->createSurveyKeys($survey->id, count($users), false, false, $users);
		
		JLoader::import('mail', JPATH_ADMINISTRATOR.'/components/com_communitysurveys/models');
		$mailModel = JModelLegacy::getInstance( 'mail', 'CommunitySurveysModel' );
		$mailModel->enqueueMail($message, $users, 'none');
	}
	
	public function getContacts()
	{
		$app		= JFactory::getApplication();
		$user 		= JFactory::getUser();
		$db 		= JFactory::getDbo();
		$limit 		= $this->getState('contacts.limit', 20);
		$limitstart	= $this->getState('contacts.limitstart', 0);
		
		$query = $db->getQuery(true)
			->select('id, name, email')
			->from('#__survey_contacts')
			->where('created_by = '.$user->id)
			->order('name asc');
		
		$groupid = (int) $app->input->getInt('filter_group_id', 0);
		if($groupid)
		{
			$query->where('id IN (select contact_id from #__survey_contact_group_map WHERE group_id = '.$groupid.')');
		}
		
		$search = $app->input->getString('filter_search', '');
		if(!empty($search))
		{
			$search = '%' . $db->escape( $search, true ) . '%';
			$query->where('(name like '.$db->q( $search, false ).' OR email like '.$db->q( $search, false ).')');
		}
		
		// get totals first
		$totalsquery = clone $query;
		$totalsquery->clear('select')->clear('order')->clear('limit')->clear('offset')->select('COUNT(*)');
		$db->setQuery($totalsquery);
		$total = (int) $db->loadResult();
		
		if ($limitstart > $total - $limit)
		{
			$limitstart = max(0, (int) (ceil($total / $limit) - 1) * $limit);
		}
		
		
		$db->setQuery($query, $limitstart, $limit);
		$contacts = $db->loadObjectList();
		
		$this->_pagination = new JPagination($total, $limitstart, $limit);
		
		return $contacts;
	}
	
	public function getPagination()
	{
		if(!empty($this->_pagination))
		{
			return $this->_pagination;
		}
		else
		{
			return parent::getPagination();
		}
	}
	
	public function getContactGroups()
	{
		$db = JFactory::getDbo();
		$user = JFactory::getUser();
		
		$query = $db->getQuery(true)
			->select('id, name, contacts')
			->from('#__survey_contactgroups')
			->where('created_by = '.$user->id)
			->order('name asc');
		
		$db->setQuery($query);
		$contacts = $db->loadObjectList();
		
		return $contacts;
	}
	
	public function getCommunityGroups()
	{
		$user = JFactory::getUser();
		
		if(file_exists(JPATH_ROOT.'/components/com_community/userpoints.php')) 
		{
			$db = JFactory::getDbo();
			$query = $db->getQuery(true)
				->select('distinct(id), name')
				->from('#__community_groups AS g')
				->join('inner', '#__community_groups_members AS m ON g.id = m.groupid')
				->where('m.memberid='.$user->id)
				->where('approved = 1');
			
			$db->setQuery($query);
			$groups = $db->loadObjectList();
			
			return !empty($groups) ? $groups : array();
		}
		else if(file_exists(JPATH_ADMINISTRATOR.'/components/com_easysocial/includes/foundry.php'))
		{
			require_once JPATH_ADMINISTRATOR.'/components/com_easysocial/includes/foundry.php';
			$model = FD::model('Groups');
			$userGroups = $model->getUserGroups($user->id);
			
			if(!empty($userGroups))
			{
				$groups = array();
				foreach ($userGroups as $userGroup)
				{
					$group = new stdClass();
					$group->id = $userGroup->id;
					$group->name = $userGroup->title;
					
					$groups[] = $group;
				}
				
				return $groups;
			}
		}
		
		return array();
	}
	
	public function saveContactGroup()
	{
		$app 	= JFactory::getApplication();
		$user	= JFactory::getUser();
		$db 	= JFactory::getDbo();
		$data 	= $app->input->post->get('jform', array(), 'array');
		
		if(empty($data['group_name']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_REQUIRED_FIELDS_MISSING'), 404);
		}
		
		$group = new stdClass();
		$group->name = $data['group_name'];
		$group->created_by = $user->id;
		
		if($db->insertObject('#__survey_contactgroups', $group))
		{
			return true;
		}
		
		return false;
	}

	public function saveContacts()
	{
		$app 	= JFactory::getApplication();
		$user	= JFactory::getUser();
		$db 	= JFactory::getDbo();
		$data 	= $app->input->post->get('jform', array(), 'array');

		if(empty($data['contact_name']) || empty($data['contact_email']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_REQUIRED_FIELDS_MISSING'), 404);
		}
		
		$query = $db->getQuery(true)
			->insert('#__survey_contacts')
			->columns('name, email, created_by');
		
		foreach ($data['contact_name'] as $i=>$name)
		{
			if(empty($name) || empty($data['contact_email'][$i]))
			{
				continue;
			}
			$query->values($db->q($name).','.$db->q($data['contact_email'][$i]).','.$user->id);
		}

		$db->setQuery($query);
		if($db->execute())
		{
			return true;
		}
	
		return false;
	}
	
	public function deleteContactGroup()
	{
		$app 	= JFactory::getApplication();
		$user	= JFactory::getUser();
		$db 	= JFactory::getDbo();
		$id		= $app->input->post->get('cid', 0, 'uint');
		
		if(!$id)
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_REQUIRED_FIELDS_MISSING'), 404);
		}
		
		$query = $db->getQuery(true)
			->delete('#__survey_contactgroups')
			->where('id = '.$id);
		
		if(!$user->authorise('core.manage', 'com_communitysurveys'))
		{
			$query->where('created_by = '.$user->id);
		}
		
		$db->setQuery($query);
		if($db->execute())
		{
			return true;
		}
		return false;
	}
	
	public function deleteContacts()
	{
		$app 	= JFactory::getApplication();
		$user	= JFactory::getUser();
		$db 	= JFactory::getDbo();
		$data	= $app->input->post->get('jform', array(), 'array');
		
		$data['contact_id'] = ArrayHelper::toInteger($data['contact_id']);
		if(empty($data['contact_id']))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_REQUIRED_FIELDS_MISSING'), 404);
		}
		
		$query = $db->getQuery(true)
			->delete('#__survey_contacts')
			->where('id IN ('.implode(',', $data['contact_id']).')');
		
		if(!$user->authorise('core.manage', 'com_communitysurveys'))
		{
			$query->where('created_by = '.$user->id);
		}
		
		$db->setQuery($query);
		if($db->execute())
		{
			$query = $db->getQuery(true)
				->delete('#__survey_contact_group_map')
				->where('contact_id IN ('.implode(',', $data['contact_id']).')');
			
			$db->setQuery($query);
			$db->execute();
			
			$query = $db->getQuery(true)
				->update('#__survey_contactgroups AS g')
				->set('g.contacts = (select count(*) from #__survey_contact_group_map AS m where m.group_id = g.id)')
				->where('g.created_by = '.$user->id);
			
			$db->setQuery($query);
			$db->execute();
			return true;
		}
		return false;
	}
	
	public function importContacts($filepath)
	{
		$user	= JFactory::getUser();
		$db 	= JFactory::getDbo();
		$contacts = array();
		
		if(($handle = fopen($filepath, 'r')) !== FALSE) 
		{
			set_time_limit(0);
			while(($data = fgetcsv($handle, 1000, ',')) !== FALSE) 
			{
				if(count($data) > 1)
				{
					array_push($contacts, array('name'=>trim($data[0]), 'email'=>trim($data[1])));
				}
			}
			fclose($handle);
		}
		
		if(!empty($contacts))
		{
			$query = $db->getQuery(true)
				->insert('#__survey_contacts')
				->columns('name, email, created_by');
			
			foreach ($contacts as $contact)
			{
				$query->values($db->q($contact['name']).','.$db->q($contact['email']).','.$user->id);
			}
			
			$db->setQuery($query);
			if($db->execute())
			{
				return true;
			}
		}
		
		return false;
	}
	
	public function assignContacts()
	{
		$app 	= JFactory::getApplication();
		$user	= JFactory::getUser();
		$db 	= JFactory::getDbo();
		$data	= $app->input->post->get('jform', array(), 'array');
		
		$data['contact_id'] = ArrayHelper::toInteger($data['contact_id']);
		$gid = (int) $data['assign_group'];
		if(empty($data['contact_id']) || !$gid)
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_REQUIRED_FIELDS_MISSING'), 404);
		}
		
		if(!$user->authorise('core.manage', 'com_communitysurveys'))
		{
			$query2 = $db->getQuery(true)
				->select('created_by')
				->from('#__survey_contactgroups')
				->where('id '. $gid);
			
			$db->setQuery($query2);
			$createdby = (int) $db->loadResult();
			
			if($createdby != $user->id)
			{
				throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
			}
		}
		
		$query = $db->getQuery(true)
			->select('contact_id')
			->from('#__survey_contact_group_map')
			->where('group_id = '.$gid)
			->where('contact_id in ('.implode(',', $data['contact_id']).')');
			
		$db->setQuery($query);
		$existing = $db->loadColumn();
		
		if( empty($data['assignAction']) )
		{
			if(empty($existing))
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_NO_CONTACTS_TO_UNASSIGN'), 500);
			}
			
			$query = $db->getQuery(true);
			$query->delete('#__survey_contact_group_map')
				->where('group_id = '.$gid)
				->where('contact_id in ('.implode(',', $existing).')');
			
			$db->setQuery($query);
			if($db->execute())
			{
				$count = $db->getAffectedRows();
				$query = $db->getQuery(true)
					->update('#__survey_contactgroups')
					->set('contacts = (select count(*) from #__survey_contact_group_map where group_id = '.$gid.')')
					->where('id = '.$gid);
				
				$db->setQuery($query);
				$db->execute();
				
				return $count;
			}
		}
		else
		{
			$data['contact_id'] = array_diff($data['contact_id'], $existing);
			if(empty($data['contact_id']))
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_CONTACTS_ALREADY_ASSIGNED'), 500);
			}
			
			$query = $db->getQuery(true);
			$query
				->insert('#__survey_contact_group_map')
				->columns('contact_id, group_id');
			
			foreach ($data['contact_id'] as $cid)
			{
				$query->values($cid.','.$gid);
			}
			
			$db->setQuery($query);
			if($db->execute())
			{
				$count = $db->getAffectedRows();
				$query = $db->getQuery(true)
					->update('#__survey_contactgroups')
					->set('contacts = (select count(*) from #__survey_contact_group_map where group_id = '.$gid.')')
					->where('id = '.$gid);
				
				$db->setQuery($query);
				$db->execute();
				
				return $count;
			}
		}
		
		return false;
	}
	
	public function getRegisteredUsers()
	{
		$app		= JFactory::getApplication();
		$db 		= JFactory::getDbo();
		$limit 		= $this->getState('users.limit', 100);
		$limitstart	= $this->getState('user.limitstart', 0);
		
		$query = $db->getQuery(true)
			->select('id, name, username')
			->from('#__users')
			->order('name asc');
		
		$search = $app->input->getString('filter_username', '');
		if(!empty($search))
		{
			$search = '%' . $db->escape( $search, true ) . '%';
			$query->where('(name like '.$db->q( $search, false ).' OR username like '.$db->q( $search, false ).')');
		}
		else
		{
			return array();
		}
		
		$db->setQuery($query, $limitstart, $limit);
		$users = $db->loadObjectList();
		
		return $users;
	}
	
	public function createUniqueUrls($surveyId = null)
	{
		$app 		= JFactory::getApplication();
		$numUrls 	= $app->input->getInt('num_urls', 5);
		$surveyId 	= $surveyId ? $surveyId : $this->getState($this->getName() . '.id');
		
		if(!$surveyId || !$numUrls)
		{
			return false;
		}
		
		// load survey urls
		JLoader::import('response', JPATH_COMPONENT_SITE.'/models');
		$responseModel = JModelLegacy::getInstance( 'response', 'CommunitySurveysModel' );
		
		if($responseModel->createSurveyKeys($surveyId, $numUrls))
		{
			return true;
		}
		
		return false;
	}
	
	public function getUniqueUrlsList($surveyId)
	{
	    $db = $this->getDbo(); 
	    
	    $query = $db->getQuery(true)
	    ->select('a.id, a.catid, a.alias, a.language, c.alias as category_alias')
	    ->from('#__survey_surveys AS a')
	    ->join('inner', '#__categories AS c ON a.catid = c.id')
	    ->where('a.id = ' . (int) $surveyId);
	    
	    $db->setQuery($query);
	    $survey = $db->loadObject();
	    
	    $query = $db->getQuery(true)
	    ->select('a.key_name, a.response_id, a.created, a.response_status')
	    ->from('#__survey_keys AS a')
	    ->where('a.survey_id = '.(int) $surveyId)
	    ->order('a.created desc');
	    
	    $db->setQuery($query);
	    $keys = $db->loadRowList();
	    
	    $headers = array();
	    $headers[] = JText::_('COM_COMMUNITYSURVEYS_LABEL_SURVEY_URL');
	    $headers[] = JText::_('ID');
	    $headers[] = JText::_('JDATE');
	    $headers[] = JText::_('JSTATUS');
	    
	    if(empty($keys) || empty($survey))
	    {
	        return array($headers);
	    }
	    
	    $surveyUrl 	= CommunitySurveysHelperRoute::getSurveyRoute($survey->id . ':' . $survey->alias, $survey->catid . ':' . $survey->category_alias, $survey->language);
	    foreach ($keys as $i => $key)
	    {
	        $keys[$i][0] = SurveyHelper::buildSefSurveyUrl($surveyUrl.'&key='.$key[0], true, -1);
	        $keys[$i][3] = $keys[$i][3] == 0 ? 
	           JText::_('COM_COMMUNITYSURVEYS_RESPONSE_YET_TO_START') : 
	           ($keys[$i][3] == 1 ? JText::_('COM_COMMUNITYSURVEYS_RESPONSE_FINALIZED') : JText::_('COM_COMMUNITYSURVEYS_RESPONSE_PENDING')); 
	    }
	    
	    $data = array_merge(array($headers), $keys);
	    return $data;
	}
}