<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('JPATH_PLATFORM') or die;

JLoader::register('CommunitySurveysHelper', JPATH_ADMINISTRATOR . '/components/com_communitysurveys/helpers/communitysurveys.php');

class CommunitySurveysModelResponseBase extends JModelAdmin
{
	protected $text_prefix = 'COM_COMMUNITYSURVEYS';
	public $typeAlias = 'com_communitysurveys.response';

	public function __construct($config)
	{
		parent::__construct($config);
	}

	/**
	 * Custom clean the cache of com_content and content modules
	 */
	protected function cleanCache($group = null, $client_id = 0)
	{
		parent::cleanCache('com_communitysurveys');
	}
	
	public function getAttachmentDetails($rid, $qid)
	{
	    $db = JFactory::getDbo();
	    $query = $db->getQuery(true)
	       ->select('d.free_text as filename, r.survey_id, r.created_by as responder_id, a.created_by as author_id, a.catid')
	       ->from('#__survey_response_details AS d')
	       ->join('INNER', '#__survey_responses AS r ON d.response_id = r.id')
	       ->join('INNER', '#__survey_questions AS q ON d.question_id = q.id')
	       ->join('INNER', '#__survey_surveys AS a ON q.survey_id = a.id')
	       ->where('d.response_id = '.$rid.' AND d.question_id = '.$qid)
	       ->where('d.free_text IS NOT NULL');
	    
	    try 
	    {
	        $db->setQuery($query);
	        $download = $db->loadObject();

	        return $download;
	    }
	    catch (Exception $e)
	    {
	        return false;
	    }
	}
	
	public function getForm($data = array(), $loadData = true)
	{
	    return parent::getForm($data, $loadData);
	}

	public function createPdfResponse($surveyId, $responseId)
	{
		$filename	= JPATH_ROOT.'/media/com_communitysurveys/attachments/responses/'.$surveyId.'_'.$responseId.'.pdf';
		if(file_exists($filename))
		{
			return $filename;
		}
		
		$return		= $this->getResponseDetails($surveyId, $responseId);
		$state		= $this->getState();
		$response	= $return->response;
		$questions	= $return->questions;
		$params		= $state->get('params');
	
		if (count($errors = $this->getErrors()))
		{
		    throw new Exception(implode('\n', $errors), 500);
		}
	
		$html		= '<style>.title {font-size: 18pt; font-weight: bold; margin-bottom: 10px; margin-top: 0;}</style>';
		$html 		= $html . SurveyHelper::renderLayout('pdf.results.header', array('item'=>$response, 'params'=>$params));
	
		foreach ($questions as $question)
		{
			$question->rid = $responseId;
			$html .= SurveyHelper::renderLayout('pdf.results.question', array('item'=>$question, 'params'=>$params));
		}
	
		// 	echo $html;
		// 	jexit();
	
		// create new PDF document
		require_once JPATH_ROOT.'/components/com_communitysurveys/helpers/tcpdf.php';
		$pdf = new CJQPDF('P', PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);
	
		// set default header data
		$pdf->SetHeaderData('', 0, $response->title, '');
	
		// set document information
		$pdf->SetCreator(PDF_CREATOR);
		$pdf->SetAuthor('corejoomla.com');
		$pdf->SetTitle('Survey Report');
		$pdf->SetSubject('Survey Response Report');
		$pdf->SetKeywords('survey, report');
	
		// set header and footer fonts
		$pdf->setHeaderFont(Array(PDF_FONT_NAME_MAIN, '', PDF_FONT_SIZE_MAIN));
		$pdf->setFooterFont(Array(PDF_FONT_NAME_DATA, '', PDF_FONT_SIZE_DATA));
	
		// set default monospaced font
		$pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);
	
		//set margins
		$pdf->SetMargins(PDF_MARGIN_LEFT, PDF_MARGIN_TOP, PDF_MARGIN_RIGHT);
		$pdf->SetHeaderMargin(PDF_MARGIN_HEADER);
		$pdf->SetFooterMargin(PDF_MARGIN_FOOTER);
	
		//set auto page breaks
		$pdf->SetAutoPageBreak(TRUE, PDF_MARGIN_BOTTOM);
	
		//set image scale factor
		$pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);
	
		// ---------------------------------------------------------
	
		// set font
		$fontname = $params->get('pdf_fontname', 'helvetica');
		$pdf->SetFont('zapfdingbats');
		$pdf->SetFont($fontname);
		$pdf->AddPage();
		$pdf->writeHTML($html, true, false, true, false, '');
		$pdf->lastPage();
		$pdf->Output($filename, 'F');
		
		return $filename;
	}
	
	private function getResponseDetails($surveyId, $responseId)
	{
		$surveyId 	= (int) ($surveyId ? $surveyId : $this->getState($this->getName() . '.id'));
		$responseId = (int) ($responseId ? $responseId : $this->getState('response.id')); 
		$db 		= JFactory::getDbo();
		$params 	= JComponentHelper::getParams('com_communitysurveys');
		$userDisplayName = $params->get('user_display_name', 'name');
		
		if(!$surveyId || !$responseId)
		{
			return false;
		}
		
		$query = $db->getQuery(true)
			->select('a.id, a.survey_id, a.created, a.created_by, a.state, a.ip_address, a.completed')
			->from('#__survey_responses AS a');
		
		// join over tracking table
		$query->select('t.ip_address, c.country_name, t.city, t.browser_name, t.browser_version, t.os, t.browser_info')
			->join('LEFT', '#__survey_tracking AS t ON t.post_id = a.id AND t.post_type = 1')
			->join('LEFT', '#__corejoomla_countries AS c ON c.country_code = t.country AND c.language = '.$db->q('*'));
			
		// Join over the categories.
		$query->select('q.title AS title, q.alias, q.checked_out, q.checked_out_time, q.language')
			->join('LEFT', '#__survey_surveys AS q ON q.id = a.survey_id');
			
		// Join over the users for the author.
		$query->select('ua.'.$userDisplayName.' AS author_name')
			->join('LEFT', '#__users AS ua ON ua.id = a.created_by');
			
		// Limit rows.
		$query
			->where('a.id = ' . (int) $responseId)
			->where('a.survey_id = ' . (int) $surveyId);
			
		// Filter by published state
		$query->where('a.state = 1');
		
		$db->setQuery($query);
		$return = new stdClass();
		try
		{
			$return->response = $db->loadObject();
			
			JLoader::import('questions', JPATH_ROOT.'/components/com_communitysurveys/models');
			$questionsModel = JModelLegacy::getInstance( 'questions', 'CommunitySurveysModel' );
			
			$questionsModel->getState();
			$questionsModel->setState('filter.survey_id', $surveyId);
			$questionsModel->setState('filter.category_id', null);
			$questionsModel->setState('filter.page_id', 0);
			$questionsModel->setState('filter.question_id', null);
			$questionsModel->setState('filter.unique', false);
			$questionsModel->setState('filter.response_id', $responseId);
			$questionsModel->setState('list.ordering', 'p.sort_order asc, a.sort_order');
			$questionsModel->setState('list.direction', 'asc');
			$questionsModel->setState('list.limit', 0);
			$return->questions = $questionsModel->getItems();
			
			return $return;
		}
		catch (Exception $e){}
		
		return false;
	}
	
	public function reopen(&$pks)
	{
		$user = $this->getUser();
		$table = $this->getTable();
		$pks = (array) $pks;
	
		// Access checks.
		foreach ($pks as $i => $pk)
		{
			$table->reset();
	
			if ($table->load($pk))
			{
				if (!$this->canEditState($table))
				{
					// Prune items that you can't change.
					unset($pks[$i]);
	
					JLog::add(JText::_('JLIB_APPLICATION_ERROR_EDITSTATE_NOT_PERMITTED'), JLog::WARNING, 'jerror');
	
					return false;
				}
	
				// If the table is checked out by another user, drop it and report to the user trying to change its state.
				if (property_exists($table, 'checked_out') && $table->checked_out && ($table->checked_out != $user->id))
				{
					JLog::add(JText::_('JLIB_APPLICATION_ERROR_CHECKIN_USER_MISMATCH'), JLog::WARNING, 'jerror');
	
					// Prune items that you can't change.
					unset($pks[$i]);
	
					return false;
				}
			}
		}
	
		// Attempt to change the state of the records.
		if (!$table->open($pks, 1, $user->id))
		{
			$this->setError($table->getError());
	
			return false;
		}
	
		// Clear the component's cache
		$this->cleanCache();
	
		return true;
	}
	
	public function delete(&$pks)
	{
        $app = JFactory::getApplication();	    
	    $pks = (array) $pks;
	    $table = $this->getTable();
	    $db = $this->getDbo();
	    
	    // Include the plugins for the delete events.
	    \JPluginHelper::importPlugin($this->events_map['delete']);
	    
	    // Iterate the items to delete each one.
	    foreach ($pks as $i => $pk)
	    {
	        if ($table->load($pk))
	        {
	            if ($this->canDelete($table))
	            {
	                $context = $this->option . '.' . $this->name;
	                
	                // Trigger the before delete event.
	                $result = $app->triggerEvent($this->event_before_delete, array($context, $table));
	                
	                if (in_array(false, $result, true))
	                {
	                    $this->setError($table->getError());
	                    
	                    return false;
	                }
	                
	                if (!$table->delete($pk))
	                {
	                    $this->setError($table->getError());
	                    
	                    return false;
	                }
	                
	                // Trigger the after event.
	                $app->triggerEvent($this->event_after_delete, array($context, $table));
	                
	                // delete response details
	                $query = $db->getQuery(true)->delete('#__survey_response_details')->where('response_id = ' . $pk);
	                $db->setQuery($query);
	                $db->execute();
	                
	                // delete keys
	                $query = $db->getQuery(true)->delete('#__survey_keys')->where('response_id = ' . $pk);
	                $db->setQuery($query);
	                $db->execute();
	                
	                $query = $db->getQuery(true)
	                   ->update('#__survey_surveys')
	                   ->set('responses = (select count(*) from #__survey_responses where survey_id = '. ((int) $table->survey_id).')')
	                   ->where('id = ' . ((int) $table->survey_id));
	                $db->setQuery($query);
	                $db->execute();
	                
	                // delete files
	                JFile::delete(JPATH_ROOT.'/media/com_communitysurveys/attachments/responses/'.$table->survey_id.'_'.$pk.'.pdf');
	            }
	            else
	            {
	                // Prune items that you can't change.
	                unset($pks[$i]);
	                $error = $this->getError();
	                
	                if ($error)
	                {
	                    \JLog::add($error, \JLog::WARNING, 'jerror');
	                    
	                    return false;
	                }
	                else
	                {
	                    \JLog::add(\JText::_('JLIB_APPLICATION_ERROR_DELETE_NOT_PERMITTED'), \JLog::WARNING, 'jerror');
	                    
	                    return false;
	                }
	            }
	        }
	        else
	        {
	            $this->setError($table->getError());
	            
	            return false;
	        }
	    }
	    
	    // Clear the component's cache
	    $this->cleanCache();
	    
	    return true;
	}
	
	public function saveComment()
	{
	    $app = JFactory::getApplication();
	    $comment = new stdClass();
	    $comment->id = $app->input->getInt('id', 0);
	    $comment->rid = $app->input->getInt('rid', 0);
	    $comment->description = $app->input->getString('comment');
	    
	    if(!$comment->id || !$comment->rid || empty($comment->description))
	    {
	        throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'));
	    }
	    
	    $query = $this->_db->getQuery(true)
	       ->update('#__survey_responses')
	       ->set('comment = ' . $this->_db->q($comment->description))
	       ->where('id = ' . $comment->rid)
	       ->where('survey_id = ' . $comment->id);
	    $this->_db->setQuery($query);
	    $this->_db->execute();
	    
	    return $comment;
	}
	
	protected function getUser()
	{
		$user = $this->getState('user');
		if(empty($user))
		{
			$user = JFactory::getUser();
		}
	
		return $user;
	}
}