<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('JPATH_PLATFORM') or die;

class CommunitySurveysModelResponses extends JModelList
{
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'a.id',
				'survey_id', 'a.survey_id',
				'created', 'a.created',
				'created_by', 'a.created_by',
				'ip_address', 'a.ip_address',
				'completed', 'a.completed',
				'state', 'a.state',
				'ip_address', 't.ip_address',
				'country', 't.country',
				'city', 't.city',
				'browser', 't.browser_name',
				'os', 't.os',
				'response_status',	
				'author_id',
			    'start_date',
			    'end_date'
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		$app = JFactory::getApplication();
		$surveyId = $app->input->getInt('id', 0);
		$this->context = 'com_communitysurveys.responses.'.$surveyId;
		
		// List state information.
		parent::populateState('a.id', 'desc');
		
		$this->setState('survey_id', $surveyId);
		
		$search = $app->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);
		
		$startDate = $app->getUserStateFromRequest($this->context . '.filter.start_date', 'start_date');
		$this->setState('filter.start_date', $startDate);
		
		$endDate = $app->getUserStateFromRequest($this->context . '.filter.end_date', 'end_date');
		$this->setState('filter.end_date', $endDate);
		
		$access = $this->getUserStateFromRequest($this->context . '.filter.access', 'filter_access');
		$this->setState('filter.access', $access);

		$published = $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '');
		$this->setState('filter.published', $published);
		
		$authorId = $this->getUserStateFromRequest($this->context . '.filter.author_id', 'filter_author_id');
		$this->setState('filter.author_id', $authorId);

		$country = $this->getUserStateFromRequest($this->context . '.filter.country', 'filter_country');
		$this->setState('filter.country', $country);

		$city = $this->getUserStateFromRequest($this->context . '.filter.city', 'filter_city');
		$this->setState('filter.city', $city);

		$os = $this->getUserStateFromRequest($this->context . '.filter.os', 'filter_os');
		$this->setState('filter.os', $os);

		$browser = $this->getUserStateFromRequest($this->context . '.filter.browser', 'filter_browser');
		$this->setState('filter.browser', $browser);

		$status = $this->getUserStateFromRequest($this->context . '.filter.response_status', 'filter_response_status');
		$this->setState('filter.response_status', $status);

		$conditions = $app->getUserStateFromRequest($this->context . '.filter.conditions', 'filter_conditions', '', 'string');
		$this->setState('advancedfilter.conditions', $conditions);
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * This is necessary because the model is used by the component and
	 * different modules that might need different sets of data or different
	 * ordering requirements.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 */
	protected function getStoreId($id = '')
	{
		// Compile the store id.
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.published');
		$id .= ':' . $this->getState('filter.survey_id');
		$id .= ':' . $this->getState('filter.author_id');
		$id .= ':' . $this->getState('filter.country');
		$id .= ':' . $this->getState('filter.city');
		$id .= ':' . $this->getState('filter.os');
		$id .= ':' . $this->getState('filter.browser');
		$id .= ':' . $this->getState('filter.response_status');

		return parent::getStoreId($id);
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  JDatabaseQuery
	 */
	protected function getListQuery()
	{
		// Create a new query object.
		$db = $this->getDbo();
		$query = $db->getQuery(true);
		$params = JComponentHelper::getParams('com_communitysurveys');
		$userDisplayName = $params->get('user_display_name', 'name');

		// Select the required fields from the table.
		$query->select($this->getState('list.select', 'a.id, a.survey_id, a.created, a.created_by, a.state, a.ip_address, a.completed, a.comment'))
			->select('case when a.completed > a.created then '.
				$db->q(JText::_('COM_COMMUNITYSURVEYS_RESPONSE_FINALIZED')).' else '.
				$db->q(JText::_('COM_COMMUNITYSURVEYS_RESPONSE_PENDING')).' end as completion_status');
		$query->from('#__survey_responses AS a');
		
		// filter conditions
		require_once JPATH_ADMINISTRATOR . '/components/com_communitysurveys/helpers/rulesparser.php';
		$conditions = $this->getState('advancedfilter.conditions', 'string');
		$rulesParser = new CommunitySurveysRulesParser();
		$rulesParser->getResult($conditions, $query, 'a.id');
		
		// join over tracking table
		$query->select('c.country_name, t.city, t.browser_name, t.browser_version, t.os, t.browser_info')
			->join('LEFT', '#__survey_tracking AS t ON t.post_id = a.id AND t.post_type = 1')
			->join('LEFT', '#__corejoomla_countries AS c ON c.country_code = t.country AND c.language = '.$db->q('*'));

		// Join over the categories.
		$query->select('q.title AS title, q.alias, q.checked_out, q.checked_out_time, q.language')
			->join('LEFT', '#__survey_surveys AS q ON q.id = a.survey_id');

		// Join over the users for the author.
		$query->select('ua.'.$userDisplayName.' AS author_name, ua.name, ua.username, ua.email')
			->join('LEFT', '#__users AS ua ON ua.id = a.created_by');

		// Filter by a survey id.
		$surveyId = $this->getState('survey_id');
		if($surveyId > 0)
		{
			$query->where('a.survey_id = ' . (int) $surveyId);
		}
		
		// Filter by published state
		$published = $this->getState('filter.published');
		
		if (is_numeric($published))
		{
			$query->where('a.state = ' . (int) $published);
		}
		elseif ($published === '')
		{
			$query->where('(a.state = 0 OR a.state = 1)');
		}
		
		// Filter by author
		$authorId = $this->getState('filter.author_id');
		if (is_numeric($authorId))
		{
			$type = $this->getState('filter.author_id.include', true) ? '= ' : '<>';
			$query->where('a.created_by ' . $type . (int) $authorId);
		}

		// Filter by country
		$country = $this->getState('filter.country');
		if (!empty($country))
		{
			$query->where('t.country = ' . $db->q($country));
		}

		// Filter by country
		$city = $this->getState('filter.city');
		if (!empty($city))
		{
			$query->where('t.city = ' . $db->q($city));
		}

		// Filter by os
		$platform = $this->getState('filter.platform');
		if (!empty($platform))
		{
			$query->where('t.os = ' . $db->q($platform));
		}

		// Filter by browser
		$browser = $this->getState('filter.browser_name');
		if (!empty($browser))
		{
			$query->where('t.browser_name = ' . $db->q($browser));
		}

		// Filter by browser
		$status = (int) $this->getState('filter.response_status');
		if (!empty($status))
		{
			switch ($status)
			{
				case 1: 
					$query->where('a.completed IS NULL OR a.completed = ' . $db->q( $db->getNullDate()));
					break;
					
				case 2:
					$query->where('a.completed > a.created');
					break;
			}
		}
		
		$startDate = $this->getState('filter.start_date', null);
		if($startDate)
		{
		    $startDate = JFactory::getDate($startDate)->toSql();
		    $query->where('a.created >= '.$db->q($startDate));
		}
		
		$endDate = $this->getState('filter.end_date', null);
		if($endDate)
		{
		    $endDate = JFactory::getDate($endDate)->toSql();
		    $query->where('a.created <= '.$db->q($endDate));
		}
		
		// Filter by search in title.
		$search = $this->getState('filter.search');
		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('a.id = ' . (int) substr($search, 3));
			}
			elseif (stripos($search, 'author:') === 0)
			{
				$search = $db->quote('%' . $db->escape(substr($search, 7), true) . '%');
				$query->where('(ua.name LIKE ' . $search . ' OR ua.username LIKE ' . $search . ')');
			}
			else 
			{
				$search = $db->quote('%' . $db->escape($search, true) . '%');
				$query->where('(ua.name LIKE ' . $search . ' OR ua.username LIKE ' . $search . ' OR c.country_name LIKE ' . $search . ' OR t.city LIKE ' . $search . ')');
			}
		}
		
		if($this->getState('filter.junk'))
		{
			$junkQuery = $db->getQuery(true)
				->select('distinct(tr.response_id)')
				->from('#__survey_response_details tr');
			
			if($surveyId > 0)
			{
				$junkQuery->join('left', '#__survey_questions tq on tr.question_id = tq.id')->where('tq.survey_id = ' . (int) $surveyId);
			}
			
			$query->where('a.id in ( ' . $junkQuery->__toString() . ' )');
		}
		
// var_dump($this->state);
		// Add the list ordering clause.
		$orderCol = $this->state->get('list.ordering', 'a.id');
		$orderDirn = $this->state->get('list.direction', 'desc');
		$query->order($db->escape($orderCol . ' ' . $orderDirn));
// echo $query->dump();jexit();

		return $query;
	}

	/**
	 * Build a list of authors
	 *
	 * @return  JDatabaseQuery
	 */
	public function getAuthors()
	{
		// Create a new query object.
		$db = $this->getDbo();
		$query = $db->getQuery(true);

		// Construct the query
		$query->select('u.id AS value, u.name AS text')
			->from('#__users AS u')
			->join('INNER', '#__survey_responses AS c ON c.created_by = u.id')
			->group('u.id, u.name')
			->order('u.name');
		
		$surveyId = $this->getState('survey_id');
		if($surveyId > 0)
		{
			$query->where('c.survey_id = ' . (int) $surveyId);
		}
		
		// Filter by published state
		$published = $this->getState('filter.published');
		
		if (is_numeric($published))
		{
			$query->where('c.state = ' . (int) $published);
		}
		elseif ($published === '')
		{
			$query->where('(c.state = 0 OR c.state = 1)');
		}
		
		// Filter by author
		$authorId = $this->getState('filter.author_id');
		
		if (is_numeric($authorId))
		{
			$type = $this->getState('filter.author_id.include', true) ? '= ' : '<>';
			$query->where('c.created_by ' . $type . (int) $authorId);
		}
		
		// Setup the query
		$db->setQuery($query);

		// Return the result
		return $db->loadObjectList();
	}
}
