<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die();

use Joomla\Utilities\ArrayHelper;

class CommunitySurveysModelRules extends JModelList
{
	public function __construct ($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
					'id', 'a.id',
					'survey_id', 'a.survey_id',
					'question_id', 'a.question_id'
			);
		}
		
		parent::__construct($config);
	}

	protected function populateState ($ordering = 'sort_order', $direction = 'ASC')
	{
		$app = JFactory::getApplication();
		$params = JComponentHelper::getParams('com_communitysurveys');
		
		// List state information
		$questions = $app->input->getArray(array('qid'=>'array'));
		$this->setState('filter.question_id', $questions['qid']);
		
		$surveyId = $app->input->get('survey_id', 0, 'uint');
		if($surveyId)
		{
			$this->setState('filter.survey_id', $surveyId);
		}
		
		$pageId = $app->input->get('page_id', 0, 'uint');
		if($pageId)
		{
			$this->setState('filter.page_id', $pageId);
		}
		
		$this->setState('params', $params);
	}

	protected function getStoreId ($id = '')
	{
		// Compile the store id.
		$id .= ':' . $this->getState('filter.survey_id');
		$id .= ':' . $this->getState('filter.survey_id.include');
		$id .= ':' . $this->getState('filter.page_id');
		$id .= ':' . $this->getState('filter.page_id.include');
		$id .= ':' . $this->getState('filter.question_id.include');
		
		return parent::getStoreId($id);
	}

	protected function getListQuery ()
	{
		// Create a new query object.
		$db = $this->getDbo();
		$query = $db->getQuery(true);
		
		// Select the required fields from the table.
		$query
			->select($this->getState('list.select', 'id, rulecontent, question_id'))
			->from('#__survey_rules AS a');
		
		// Filter by survey id.
		$surveyId = (int) $this->getState('filter.survey_id');
		$query->where('a.survey_id = '.(int) $surveyId);
		
		// Filter by single or group of questions.
		$questionId = $this->getState('filter.question_id');
		if (is_numeric($questionId) && $questionId)
		{
			// join over surveys table
			$type = $this->getState('filter.question_id.include', true) ? '= ' : '<> ';
			$query->where('a.question_id ' . $type . (int) $questionId);
		}
		elseif (is_array($questionId) && (count($questionId) > 0))
		{
		    $questionId = ArrayHelper::toInteger($questionId);
			$questionId = implode(',', $questionId);
		
			if (! empty($questionId))
			{
				$type = $this->getState('filter.question_id.include', true) ? 'IN' : 'NOT IN';
				$query->where('a.question_id ' . $type . ' (' . $questionId . ')');
			}
		}
		
		// Add the list ordering clause.
		$query->order($this->getState('list.ordering', 'a.id') . ' ' . $this->getState('list.direction', 'ASC'));
// echo $query->dump();
// jexit();
		return $query;
	}

	public function getStart ()
	{
		return $this->getState('list.start');
	}
	
	public function getItems ()
	{
		$items = parent::getItems();
		
		if(!empty($items))
		{
			$db = JFactory::getDbo();
			
			/**************** Get Questions *****************/
			$query = $db->getQuery(true)
				->select('id, title, question_type')
				->from('#__survey_questions as a');
			
			// Filter by survey id.
			$surveyId = (int) $this->getState('filter.survey_id');
			$query->where('a.survey_id = '. $surveyId);
			$db->setQuery($query);
			$questions = $db->loadAssocList('id');
			
			/**************** Get Answers *****************/
			$query = $db->getQuery(true)
				->select('id, answer_label')
				->from('#__survey_answers as a')
				->where('a.survey_id = '. $surveyId);
			
			// Filter by single or group of questions.
			$questionId = $this->getState('filter.question_id');
			if (is_numeric($questionId) && $questionId)
			{
				// join over surveys table
				$type = $this->getState('filter.question_id.include', true) ? '= ' : '<> ';
				$query->where('a.question_id ' . $type . (int) $questionId);
			}
			elseif (is_array($questionId) && (count($questionId) > 0))
			{
			    $questionId = ArrayHelper::toInteger($questionId);
				$questionId = implode(',', $questionId);
			
				if (! empty($questionId))
				{
					$type = $this->getState('filter.question_id.include', true) ? 'IN' : 'NOT IN';
					$query->where('a.question_id ' . $type . ' (' . $questionId . ')');
				}
			}
			
			$db->setQuery($query);
			$answers = $db->loadAssocList('id');
			
			/**************** Get Pages *****************/
			$query = $db->getQuery(true)
				->select('id, title')
				->from('#__survey_pages')
				->where('sid = '.$surveyId);
			
			$db->setQuery($query);
			$pages = $db->loadAssocList('id');
			
			/************** Final Touches **************/
			foreach ($items as &$item)
			{
				$rule = json_decode($item->rulecontent);
				$question = (isset($rule->question) && !empty($questions[$rule->question])) ? $questions[$rule->question]['title'] : 'NA';
				$item->description = '';

				switch ($rule->name)
				{
					case 'answered':
					
						if($rule->page > 0)
						{
							$pageName = isset($pages[$rule->page]) ? $rule->page . ' (' . $pages[$rule->page]['title'] .')' : 'NA';
							$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_ANSWERED_SKIP_TO_PAGE', $pageName);
						}
						else if($rule->finalize == 1)
						{
							$item->description =  JText::_('COM_COMMUNITYSURVEYS_RULE_IF_ANSWERED_FINALIZE_SURVEY');
						} 
						else if($rule->question > 0)
						{
							$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_ANSWERED_SHOW_QUESTION', $question);
						}
					
						break;
					
					case 'unanswered':
					
						if($rule->page > 0)
						{
							$pageName = isset($pages[$rule->page]) ? $rule->page . ' (' . $pages[$rule->page]['title'] .')' : 'NA';
							$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_NOT_ANSWERED_SKIP_TO_PAGE', $pageName);
						}
						else if($rule->finalize == 1)
						{
							$item->description =  JText::_('COM_COMMUNITYSURVEYS_RULE_IF_NOT_ANSWERED_FINALIZE_SURVEY');
						}
					
						break;
					
					case 'selected':
					
						$answer = !empty($answers[$rule->answer_id]) ? $answers[$rule->answer_id]['answer_label'] : $rule->answer_id;
						$column = (!empty($rule->column_id) && !empty($answers[$rule->column_id])) ? $answers[$rule->column_id]['answer_label'] : null;
						
						if(empty($column) && !empty($rule->column_id))
						{
							$column = $rule->column_id;
						}
							
						if($rule->page > 0)
						{
							$pageName = isset($pages[$rule->page]) ? $rule->page . ' (' . $pages[$rule->page]['title'] .')' : 'NA';
							if(empty($column))
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_SELECTED_ANSWER_SKIP_TO_PAGE', $answer, $pageName);
							} 
							else 
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_SELECTED_ANSWER_COLUMN_SKIP_TO_PAGE', $answer, $column, $pageName);
							}
						}
						else if($rule->finalize == 1)
						{
							if(empty($column))
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_SELECTED_ANSWER_FINALIZE_SURVEY', $answer);
							} 
							else 
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_SELECTED_ANSWER_COLUMN_FINALIZE_SURVEY', $answer, $column);
							}
						} 
						else if($rule->question > 0)
						{
							if(empty($column))
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_SELECTED_ANSWER_SHOW_QUESTION', $answer, $question);
							} 
							else 
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_SELECTED_ANSWER_COLUMN_SHOW_QUESTION', $answer, $column, $question);
							}
						}
					
						break;
					
					case 'unselected':
					
						$answer = !empty($answers[$rule->answer_id]) ? $answers[$rule->answer_id]['answer_label'] : $rule->answer_id;
						$column = !empty($rule->column_id) ? $answers[$rule->column_id]['answer_label'] : null;

						if(empty($column) && !empty($rule->column_id))
						{
							$column = $rule->column_id;
						}
						
						if($rule->page > 0)
						{
							$pageName = isset($pages[$rule->page]) ? $rule->page . ' (' . $pages[$rule->page]['title'] .')' : 'NA';
							if(empty($column))
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_NOT_SELECTED_ANSWER_SKIP_TO_PAGE', $answer, $pageName);
							} 
							else 
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_NOT_SELECTED_ANSWER_COLUMN_SKIP_TO_PAGE', $answer, $column, $pageName);
							}
						}
						else if($rule->finalize == 1)
						{
							if(empty($column))
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_NOT_SELECTED_ANSWER_FINALIZE_SURVEY', $answer);
							} 
							else 
							{
								$item->description =  JText::sprintf('COM_COMMUNITYSURVEYS_RULE_IF_NOT_SELECTED_ANSWER_COLUMN_FINALIZE_SURVEY', $answer, $column);
							}
						}
					
						break;
				}
			}
		}
		
		return $items;
	}
	
	public function getPages($surveyId)
	{
		$db = JFactory::getDbo();
		
		try 
		{
			$query = $db->getQuery(true)
				->select('id, title, sort_order')
				->from('#__survey_pages')
				->where('sid = '.(int) $surveyId)
				->order('sort_order asc');
			
			$db->setQuery($query);
			$pages = $db->loadObjectList();
			
			return $pages;
		}
		catch (Exception $e)
		{
			return null;
		}
	}
}