<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('JPATH_PLATFORM') or die;

JLoader::register('CommunitySurveysHelper', JPATH_ADMINISTRATOR . '/components/com_communitysurveys/helpers/communitysurveys.php');

class CommunitySurveysModelSurvey extends JModelAdmin
{
	protected $text_prefix = 'COM_COMMUNITYSURVEYS';

	public $typeAlias = 'com_communitysurveys.survey';
	
	private $_error = '';
	private $_message = '';

	public function __construct($config)
	{
		$config['event_after_delete'] = 'onSurveyAfterDelete';
		$config['event_after_save'] = 'onSurveyAfterSave';
		$config['event_before_delete'] = 'onSurveyBeforeDelete';
		$config['event_before_save'] = 'onSurveyBeforeSave';
		$config['event_change_state'] = 'onSurveyChangeState';
		$config['events_map'] = array(
				'delete' => 'communitysurveys', 
				'save' => 'communitysurveys', 
				'change_state' => 'communitysurveys');
	
		parent::__construct($config);
	}
	
	/**
	 * Batch copy items to a new category or current.
	 *
	 * @param   integer  $value     The new category.
	 * @param   array    $pks       An array of row IDs.
	 * @param   array    $contexts  An array of item contexts.
	 *
	 * @return  mixed  An array of new IDs on success, boolean false on failure.
	 */
	protected function batchCopy($value, $pks, $contexts)
	{
		$categoryId = (int) $value;

		$i = 0;

		if (!parent::checkCategoryId($categoryId))
		{
			return false;
		}

		// Parent exists so we let's proceed
		while (!empty($pks))
		{
			// Pop the first ID off the stack
			$pk = array_shift($pks);

			$this->table->reset();

			// Check that the row actually exists
			if (!$this->table->load($pk))
			{
				if ($error = $this->table->getError())
				{
					// Fatal error
					$this->setError($error);

					return false;
				}
				else
				{
					// Not fatal error
					$this->setError(JText::sprintf('JLIB_APPLICATION_ERROR_BATCH_MOVE_ROW_NOT_FOUND', $pk));
					continue;
				}
			}

			// Alter the title & alias
			$data = $this->generateNewTitle($categoryId, $this->table->alias, $this->table->title);
			$this->table->title = $data['0'];
			$this->table->alias = $data['1'];

			// Reset the ID because we are making a copy
			$this->table->id = 0;

			// Reset hits because we are making a copy
			$this->table->responses = 0;

			// New category ID
			$this->table->catid = $categoryId;

			// TODO: Deal with ordering?
			//$table->ordering	= 1;

			// Check the row.
			if (!$this->table->check())
			{
				$this->setError($this->table->getError());
				return false;
			}

			parent::createTagsHelper($this->tagsObserver, $this->type, $pk, $this->typeAlias, $this->table);

			// Store the row.
			if (!$this->table->store())
			{
				$this->setError($this->table->getError());
				return false;
			}

			// Get the new item ID
			$newId = $this->table->get('id');

			// Add the new ID to the array
			$newIds[$i] = $newId;
			$i++;
		}

		// Clean the cache
		$this->cleanCache();

		return $newIds;
	}

	/**
	 * Method to test whether a record can be deleted.
	 *
	 * @param   object    $record    A record object.
	 *
	 * @return  boolean  True if allowed to delete the record. Defaults to the permission set in the component.
	 */
	protected function canDelete($record)
	{
		if (!empty($record->id))
		{
			if ($record->published != -2)
			{
				return;
			}
			$user = JFactory::getUser();
			return $user->authorise('core.delete', 'com_communitysurveys.survey.' . (int) $record->id);
		}
	}

	/**
	 * Method to test whether a record can have its state edited.
	 *
	 * @param   object    $record    A record object.
	 *
	 * @return  boolean  True if allowed to change the state of the record. Defaults to the permission set in the component.
	 */
	protected function canEditState($record)
	{
		$user = JFactory::getUser();
		$asset = 'com_communitysurveys';

		// Check for existing article.
		if (!empty($record->id))
		{
			$asset = 'com_communitysurveys.survey.' . (int) $record->id;
		}
		// New article, so check against the category.
		elseif (!empty($record->catid))
		{
			$asset = 'com_communitysurveys.category.' . (int) $record->catid;
		}
		
		if($user->authorise('core.edit.state', $asset))
		{
			return true;
		}
		else if($user->id && !empty($record->created_by) && $record->created_by == $user->id)
		{
			if($user->authorise('core.edit.state.own', $asset))
			{
				return true;
			}
		}
		
		return false;
	}

	/**
	 * Prepare and sanitise the table data prior to saving.
	 *
	 * @param   JTable    A JTable object.
	 *
	 * @return  void
	 */
	protected function prepareTable($table)
	{
		// Set the publish date to now
		$db = $this->getDbo();
		if ($table->published == 1 && (int) $table->publish_up == 0)
		{
			$table->publish_up = JFactory::getDate()->toSql();
		}

		if ($table->published == 1 && intval($table->publish_down) == 0)
		{
			$table->publish_down = $db->getNullDate();
		}
	}

	/**
	 * Returns a Table object, always creating it.
	 *
	 * @param   type      The table type to instantiate
	 * @param   string    A prefix for the table class name. Optional.
	 * @param   array     Configuration array for model. Optional.
	 *
	 * @return  JTable    A database object
	 */
	public function getTable($type = 'Survey', $prefix = 'CommunitySurveysTable', $config = array())
	{
		return JTable::getInstance($type, $prefix, $config);
	}

	/**
	 * Method to get a single record.
	 *
	 * @param   integer    The id of the primary key.
	 *
	 * @return  mixed  Object on success, false on failure.
	 */
	public function getItem($pk = null)
	{
		if ($item = parent::getItem($pk))
		{
			// Convert the params field to an array.
			$registry = new JRegistry;
			$registry->loadString($item->attribs);
			$item->attribs = $registry->toArray();
			
			// Convert the metadata field to an array.
			$registry = new JRegistry;
			$registry->loadString($item->metadata);
			$item->metadata = $registry->toArray();

			if (!empty($item->id))
			{
				$item->tags = new JHelperTags;
				$item->tags->getTagIds($item->id, 'com_communitysurveys.survey');
			}
		}

		// Load associated content items
		$app = JFactory::getApplication();
		$assoc = JLanguageAssociations::isEnabled();

		if ($assoc)
		{
			$item->associations = array();

			if ($item->id != null)
			{
				$associations = JLanguageAssociations::getAssociations('com_communitysurveys', '#__survey_surveys', 'com_communitysurveys.item', $item->id);

				foreach ($associations as $tag => $association)
				{
					$item->associations[$tag] = $association->id;
				}
			}
		}

		return $item;
	}

	/**
	 * Method to get the record form.
	 *
	 * @param   array      $data        Data for the form.
	 * @param   boolean    $loadData    True if the form is to load its own data (default case), false if not.
	 *
	 * @return  mixed  A JForm object on success, false on failure
	 */
	public function getForm($data = array(), $loadData = true)
	{
		// Get the form.
		$form = $this->loadForm('com_communitysurveys.survey', 'survey', array('control' => 'jform', 'load_data' => $loadData));
		if (empty($form))
		{
			return false;
		}
		$jinput = JFactory::getApplication()->input;

		// The front end calls this model and uses a_id to avoid id clashes so we need to check for that first.
		if ($jinput->get('s_id'))
		{
			$id = $jinput->get('s_id', 0);
		}
		// The back end uses id so we use that the rest of the time and set it to 0 by default.
		else
		{
			$id = $jinput->get('id', 0);
		}
		// Determine correct permissions to check.
		if ($this->getState('survey.id'))
		{
			$id = $this->getState('survey.id');
			// Existing record. Can only edit in selected categories.
			$form->setFieldAttribute('catid', 'action', 'core.edit');
			// Existing record. Can only edit own articles in selected categories.
			$form->setFieldAttribute('catid', 'action', 'core.edit.own');
		}
		else
		{
			// New record. Can only create in selected categories.
			$form->setFieldAttribute('catid', 'action', 'core.create');
		}

		$user = JFactory::getUser();

		// Check for existing article.
		// Modify the form based on Edit State access controls.
		if ($id != 0 && (!$user->authorise('core.edit.state', 'com_communitysurveys.survey.' . (int) $id))
			|| ($id == 0 && !$user->authorise('core.edit.state', 'com_communitysurveys'))
		)
		{
			// Disable fields for display.
			$form->setFieldAttribute('ordering', 'disabled', 'true');
// 			$form->setFieldAttribute('publish_up', 'disabled', 'true');
// 			$form->setFieldAttribute('publish_down', 'disabled', 'true');
			$form->setFieldAttribute('published', 'disabled', 'true');

			// Disable fields while saving.
			// The controller has already verified this is an article you can edit.
			$form->setFieldAttribute('ordering', 'filter', 'unset');
// 			$form->setFieldAttribute('publish_up', 'filter', 'unset');
// 			$form->setFieldAttribute('publish_down', 'filter', 'unset');
			$form->setFieldAttribute('published', 'filter', 'unset');
		}

		// Prevent messing with article language and category when editing existing article with associations
		$app = JFactory::getApplication();
		$assoc = JLanguageAssociations::isEnabled();

		// Check if article is associated
		if ($this->getState('survey.id') && $app->isClient('site') && $assoc)
		{
			$associations = JLanguageAssociations::getAssociations('com_communitysurveys', '#__survey_surveys', 'com_communitysurveys.item', $id);

			// Make fields read only
			if ($associations)
			{
				$form->setFieldAttribute('language', 'readonly', 'true');
				$form->setFieldAttribute('catid', 'readonly', 'true');
				$form->setFieldAttribute('language', 'filter', 'unset');
				$form->setFieldAttribute('catid', 'filter', 'unset');
			}
		}

		return $form;
	}

	/**
	 * Method to get the data that should be injected in the form.
	 *
	 * @return  mixed  The data for the form.
	 */
	protected function loadFormData()
	{
		// Check the session for previously entered form data.
		$app = JFactory::getApplication();
		$data = $app->getUserState('com_communitysurveys.edit.survey.data', array());

		if (empty($data))
		{
			$data = $this->getItem();

			// Prime some default values.
			if ($this->getState('survey.id') == 0)
			{
				$filters = (array) $app->getUserState('com_communitysurveys.survey.filter');
				$filterCatId = isset($filters['category_id']) ? $filters['category_id'] : null;

				$data->set('catid', $app->input->getInt('catid', $filterCatId));
			}
		}

		$this->preprocessData('com_communitysurveys.survey', $data);
		return $data;
	}

	/**
	 * Method to save the form data.
	 *
	 * @param   array  The form data.
	 *
	 * @return  boolean  True on success.
	 */
	public function save($data)
	{
		$app = JFactory::getApplication();
		$task = $app->input->get('task');

		// Alter the title for save as copy
		if ($task == 'save2copy')
		{
			list($title, $alias) = $this->generateNewTitle($data['catid'], $data['alias'], $data['title']);
			$data['title'] = $title;
			$data['alias'] = $alias;
			$data['published'] = 0;
		}
		
		if(empty($data['id']))
		{
			$data['ip_address'] = CjLibUtils::getUserIpAddress();
			$data['created'] = JFactory::getDate()->toSql();
			$data['published'] = 3;
			$data['survey_key'] = CjLibUtils::getRandomKey(16);
			
			if(!$data['created_by'])
			{
				$data['created_by'] = JFactory::getUser()->id;
			}
		}
		
		$data['restrictions'] = !empty($data['restrictions']) ? implode(',', $data['restrictions']) : '';
		if (parent::save($data))
		{
			$db = JFactory::getDbo();
			$id = (int) $this->getState($this->getName() . '.id');
			$isNew = $this->getState($this->getName() . '.new');
			$item = $this->getItem($id);
			
			if($isNew)
			{
				if ($task == 'save2copy')
				{
					$oldId = $app->input->getInt('id');
					if(!$oldId)
					{
						$oldId = $app->input->getInt('s_id');
					}
					
					$this->copyQuestions($oldId, $id);
				}
				else 
				{
					// create first page
					$query = $db->getQuery(true)
						->insert('#__survey_pages')
						->columns('sid, title, sort_order')
						->values($id.','.$db->q('Page 1').', 1');
					
					try
					{
						$db->setQuery($query);
						$db->execute();
					}
					catch (Exception $e)
					{
						//return false;
					}
				}
			}
			
			$assoc = JLanguageAssociations::isEnabled();
			if ($assoc)
			{
				// Adding self to the association
				$associations = $data['associations'];

				if(count($associations))
				{
					foreach ($associations as $tag => $id)
					{
						if (empty($id))
						{
							unset($associations[$tag]);
						}
					}
				}
				
				// Detecting all item menus
				$all_language = $item->language == '*';

				if ($all_language && !empty($associations))
				{
				    throw new Exception(JText::_('COM_CONTENT_ERROR_ALL_LANGUAGE_ASSOCIATED'), 403);
				}

				$associations[$item->language] = $item->id;

				// Deleting old association for these items
				$assoc_vals = implode(',', $associations);
				if(!empty($assoc_vals))
				{
					$query = $db->getQuery(true)
						->delete('#__associations')
						->where('context=' . $db->quote('com_communitysurveys.item'))
						->where('id IN (' . $assoc_vals . ')');
					$db->setQuery($query);
					$db->execute();
	
					if ($error = $db->getErrorMsg())
					{
						$this->setError($error);
						return false;
					}
								
					if (!$all_language && count($associations))
					{
						// Adding new association for these items
						$key = md5(json_encode($associations));
						$query = $db->getQuery(true)
							->insert('#__associations');
	
						foreach ($associations as $id)
						{
							$query->values($id . ',' . $db->quote('com_communitysurveys.item') . ',' . $db->quote($key));
						}
	
						$db->setQuery($query);
						$db->execute();
	
						if ($error = $db->getErrorMsg())
						{
							$this->setError($error);
							return false;
						}
					}
				}
			}

			return true;
		}

		return false;
	}
	
	public function copyQuestions($oldSurveyId, $newSurveyId)
	{
		$db 		= JFactory::getDbo();
		$app		= JFactory::getApplication();
		$pages 		= null;
		$questions 	= null;
		
		$query = $db->getQuery(true)
			->select('id, sort_order, title')
			->from('#__survey_pages')
			->where('sid = '.$oldSurveyId)
			->order('sort_order');
		
		$db->setQuery($query);
		try
		{
			$pages = $db->loadObjectList();
		}
		catch (Exception $e)
		{
			$app->enqueueMessage('Database error. |Error='.$e->getMessage());
			return false;
		}
		
		$query = $db->getQuery(true)
			->select('id, title, description, question_type, page_number, sort_order, mandatory, created_by')
			->select('custom_choice, orientation, min_selections, max_selections, params')
			->from('#__survey_questions')
			->where('survey_id = '.$oldSurveyId);
		 
		$db->setQuery($query);
		try 
		{
			$questions = $db->loadObjectList();
		}
		catch (Exception $e)
		{
			$app->enqueueMessage('Database error. |Error='.$e->getMessage());
			return false;
		}
		
		if(empty($pages) || empty($questions))
		{
			return false;
		}
		
		foreach ($pages as &$page)
		{
			$query = $db->getQuery(true)
				->insert('#__survey_pages')
				->columns('sid, sort_order, title')
				->values($newSurveyId.','.$page->sort_order.','.$db->q($page->title));
			
			$db->setQuery($query);
			try 
			{
				if(!$db->execute())
				{
					return false;
				}
			}
			catch (Exception $e)
			{
				$app->enqueueMessage('Database error. |Error='.$e->getMessage());
				return false;
			}
			
			$page->old_id = $page->id;
			$page->id = $db->insertid();
			
			if($page->id <= 0)
			{
				return false;
			}
			
			foreach ($questions as &$qn)
			{
				if($qn->page_number != $page->old_id)
				{
					continue;
				}
				
				$oldQnId = $qn->id;
				$qn->id = 0;
				$qn->survey_id = $newSurveyId;
				$qn->page_number = $page->id;
				
				try
				{
					if(!$db->insertObject('#__survey_questions', $qn))
					{
						return false;
					}
				}
				catch (Exception $e)
				{
					$app->enqueueMessage('Database error. |Error='.$e->getMessage());
					return false;
				}
				
				$qn->old_id = $oldQnId;
				$qn->id = $db->insertid();
				
				if($qn->id<= 0)
				{
					return false;
				}
				
				$query = $db->getQuery(true)
					->select('id, survey_id, question_id, answer_type, answer_label, sort_order, image')
					->from('#__survey_answers')
					->where('survey_id = '.$oldSurveyId.' and question_id = '.$qn->old_id);
				$db->setQuery($query);
				$qn->answers = $db->loadObjectList();
				
				foreach ($qn->answers as &$answer)
				{
					$oldAnsweId = $answer->id;
					$answer->id = 0;
					$answer->survey_id = $newSurveyId;
					$answer->question_id = $qn->id;
					
					try
					{
						if(!$db->insertObject('#__survey_answers', $answer))
						{
							return false;
						}
						
						$answer->old_id = $oldAnsweId;
						$answer->id = $db->insertid();
						
						if(empty($answer->id))
						{
							$app->enqueueMessage('Database error. Invalid answer id for '.$oldAnsweId.'. |Error='.$e->getMessage());
							return false;
						}
					}
					catch (Exception $e)
					{
						$app->enqueueMessage('Database error. |Error='.$e->getMessage());
						return false;
					}
				}
			}
		}
		
		// conditional rules
		$query = $db->getQuery(true)
			->select('survey_id, question_id, rulecontent')
			->from('#__survey_rules')
			->where('survey_id = ' . $oldSurveyId);
		$db->setQuery($query);
		$rules = $db->loadObjectList();
		
		if(empty($rules))
		{
			return true;
		}
		
		foreach ($rules as $rule)
		{
			$rule->survey_id = $newSurveyId;
			$ruleContent = json_decode($rule->rulecontent);
			$oldQnId = $rule->question_id;
			
			foreach ($questions as $question)
			{
				if($rule->question_id != $question->old_id)
				{
					continue;
				}
				
				$rule->question_id = $question->id;
				
				foreach ($question->answers as $answer)
				{
					if($answer->old_id == $ruleContent->answer_id)
					{
						$ruleContent->answer_id = $answer->id;
					} 
					else if ($answer->old_id == $ruleContent->column_id)
					{
						$ruleContent->column_id = $answer->id;
					}
				}
				
				break;
			}
			
			if($oldQnId == $rule->question_id)
			{
				// no question found, do not process this rule
				continue;
			}
			
			if($ruleContent->page > 0)
			{
				foreach ($pages as $page)
				{
					if($ruleContent->page == $page->old_id)
					{
						$ruleContent->page = $page->id;
						break;
					}
				}
			}
			
			if($ruleContent->question > 0)
			{
				foreach ($questions as $question)
				{
					if($question->old_id == $ruleContent->question)
					{
						$ruleContent->question = $question->id;
						break;
					}
				}
			}
			
			$rule->rulecontent = json_encode($ruleContent);
			
			try
			{
				if(!$db->insertObject('#__survey_rules', $rule))
				{
					return false;
				}
			}
			catch (Exception $e)
			{
				$app->enqueueMessage('Database error. |Error='.$e->getMessage());
				return false;
			}
		}
		 
		return true;
	}

	/**
	 * A protected method to get a set of ordering conditions.
	 *
	 * @param   object    A record object.
	 *
	 * @return  array  An array of conditions to add to add to ordering queries.
	 */
	protected function getReorderConditions($table)
	{
		$condition = array();
		$condition[] = 'catid = ' . (int) $table->catid;
		return $condition;
	}

	/**
	 * Auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @return  void
	 */
	protected function preprocessForm(JForm $form, $data, $group = 'content')
	{
		// Association content items
		$app = JFactory::getApplication();
		$assoc = JLanguageAssociations::isEnabled();
		if ($assoc)
		{
			$languages = JLanguageHelper::getLanguages('lang_code');
			$addform = new SimpleXMLElement('<form />');
			$fields = $addform->addChild('fields');
			$fields->addAttribute('name', 'associations');
			$fieldset = $fields->addChild('fieldset');
			$fieldset->addAttribute('name', 'item_associations');
			$fieldset->addAttribute('description', 'COM_COMMUNITYSURVEYS_ITEM_ASSOCIATIONS_FIELDSET_DESC');
			$add = false;
			foreach ($languages as $tag => $language)
			{
				if (empty($data->language) || $tag != $data->language)
				{
					$add = true;
					$field = $fieldset->addChild('field');
					$field->addAttribute('name', $tag);
					$field->addAttribute('type', 'modal_survey');
					$field->addAttribute('language', $tag);
					$field->addAttribute('label', $language->title);
					$field->addAttribute('translate_label', 'false');
					$field->addAttribute('edit', 'true');
					$field->addAttribute('clear', 'true');
				}
			}
			if ($add)
			{
				$form->load($addform, false);
			}
		}
		
		parent::preprocessForm($form, $data, $group);
	}

	/**
	 * Custom clean the cache of com_content and content modules
	 */
	protected function cleanCache($group = null, $client_id = 0)
	{
		parent::cleanCache('com_communitysurveys');
	}

	public function delete(&$pks)
	{
		$db = JFactory::getDbo();
    	$queries = array();
    	$id = implode(',', $pks);
    	 
    	$queries[] = 'delete from #__survey_response_details where response_id in (select id from #__survey_responses where survey_id in ('.$id.'))';
    	$queries[] = 'delete from #__survey_responses where survey_id in ('.$id.')';
    	$queries[] = 'delete from #__survey_answers where survey_id in ('.$id.')';
    	$queries[] = 'delete from #__survey_questions where survey_id in ('.$id.')';
    	$queries[] = 'delete from #__survey_pages where sid in ('.$id.')';
    	 
    	foreach ($queries as $query)
    	{
    	    $db->setQuery($query);
    	    if(!$db->query())
    	    {
    	        return false;
    	    }
    	}
    	
    	return parent::delete($pks);
	}
    
    public function makeSurveyOnline($id)
    {
        $user = JFactory::getUser();
        $db = JFactory::getDbo();
        
        try
        {
        	$query = $db->getQuery(true)
        		->select('id, catid, created_by, published')
        		->from('#__survey_surveys')
        		->where('id = '. (int) $id);
        	$db->setQuery($query);
        	$survey = $db->loadObject();	
        	
        	if($user->id != $survey->created_by)
        	{
	            if( 
	            		!$user->authorise('core.edit.state', 'com_communitysurveys.category.'.$survey->catid) &&
	            		!$user->authorise('core.manage', 'com_communitysurveys.category.'.$survey->catid) 
	            )
	            {
	            	$this->setError(JText::_('JERROR_ALERTNOAUTHOR'));
	            }
        	}
        	else if($survey->published != 1 && !$user->authorise('core.approve', 'com_communitysurveys.category.'.$survey->catid))
        	{
        		$this->setMessage(JText::_('COM_COMMUNITYSURVEYS_SURVEY_SENT_FOR_REVIEW'));
        		return true;
        	}
        	
        	
            $query = $db
            	->getQuery(true)
            	->update('#__survey_surveys')
            	->set('published = 1')
            	->where('id = '.(int) $id);
            $db->setQuery($query);
            
            if($db->execute())
            {
            	$table  = $this->getTable();
            	$table->load($id);
            	
            	JPluginHelper::importPlugin('communitysurveys');
            	$result = JFactory::getApplication()->triggerEvent('onSurveyAfterSave', array('com_communitysurveys.form', $table, true));
            	
            	if (in_array(false, $result, true))
            	{
            		$this->setError($table->getError());
            	
            		return false;
            	}
            	
            	$this->setMessage(JText::_('COM_COMMUNITYSURVEYS_SURVEY_SUCCESSFULLY_PUBLISHED'));
                return true;
            }
        }
        catch (Exception $e)
        {
            return false;
        }
        
        return false;
    }
	
    public function getError($i = null, $toString = true){
    	
    	return $this->_error;
    }
    
    public function setError($error){
    	
    	$this->_error = $error;
    }
	
    public function getMessage($i = null, $toString = true){
    	
    	return $this->_message;
    }
    
    public function setMessage($message){
    	
    	$this->_message = $message;
    }
}