<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die;

class CommunitySurveysModelTranslations extends JModelList
{
	/**
	 * Constructor.
	 *
	 * @param   array  $config  An optional associative array of configuration settings.
	 * @see     JController
	 */
	public function __construct($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
				'id', 'a.id',
				'title',
				'published', 'a.published',
				'created', 'a.created',
				'created_by', 'a.created_by',
				'item_type', 'a.item_type',
			    'survey_id', 'a.survey_id',
				'language', 'a.language'
			);
		}

		parent::__construct($config);
	}

	/**
	 * Method to auto-populate the model state.
	 *
	 * Note. Calling getState in this method will result in recursion.
	 *
	 * @param   string  $ordering   An optional ordering field.
	 * @param   string  $direction  An optional direction (asc|desc).
	 *
	 * @return  void
	 */
	protected function populateState($ordering = null, $direction = null)
	{
		$app = JFactory::getApplication();

		// Adjust the context to support modal layouts.
		if ($layout = $app->input->get('ui_layout'))
		{
			$this->context .= '.' . $layout;
		}
		
		$surveyId = $app->input->getInt('survey_id');
		$this->setState('survey.id', $surveyId);
		
		$language = $app->input->getInt('language');
		$this->setState('survey.language', $language);

		$search = $this->getUserStateFromRequest($this->context . '.filter.search', 'filter_search');
		$this->setState('filter.search', $search);

		$published = $this->getUserStateFromRequest($this->context . '.filter.published', 'filter_published', '');
		$this->setState('filter.published', $published);

		$language = $this->getUserStateFromRequest($this->context . '.filter.language', 'filter_language', '');
		$this->setState('filter.language', $language);

		// List state information.
		parent::populateState('a.id', 'asc');

		// Force a language
		$forcedLanguage = $app->input->get('forcedLanguage');

		if (!empty($forcedLanguage))
		{
			$this->setState('filter.language', $forcedLanguage);
			$this->setState('filter.forcedLanguage', $forcedLanguage);
		}
	}

	/**
	 * Method to get a store id based on model configuration state.
	 *
	 * This is necessary because the model is used by the component and
	 * different modules that might need different sets of data or different
	 * ordering requirements.
	 *
	 * @param   string  $id  A prefix for the store id.
	 *
	 * @return  string  A store id.
	 */
	protected function getStoreId($id = '')
	{
		// Compile the store id.
		$id .= ':' . $this->getState('filter.search');
		$id .= ':' . $this->getState('filter.published');
		$id .= ':' . $this->getState('filter.language');

		return parent::getStoreId($id);
	}

	/**
	 * Build an SQL query to load the list data.
	 *
	 * @return  JDatabaseQuery
	 */
	protected function getListQuery()
	{
		// Create a new query object.
		$db = $this->getDbo();
		$query = $db->getQuery(true);
		$surveyId = (int) $this->getState('survey.id');
		$languageId = (int) $this->getState('survey.language');
		
		if(!$languageId || !$surveyId)
		{
		    throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
		}
		
		// Select the required fields from the table.
		$query->select(
			$this->getState(
				'list.select',
				'a.id, a.language_id, a.item_id, a.item_type, a.translation, a.published, a.created, a.created_by'
			)
		);
		$query->from('#__survey_translations AS a');
		
		// join over survey languages
		$query
        ->select('l.survey_id, l.language')
        ->join('inner', '#__survey_languages AS l ON l.id = '.$languageId);
		
		// Join over the users for the author.
		$query->select('ua.name AS author_name')
		->join('LEFT', '#__users AS ua ON ua.id = a.created_by');
		
		$query->select(
		    'CASE a.item_type '.
		    'WHEN '.CS_TRANSLATION_SURVEY_TITLE.' THEN s1.title '.
		    'WHEN '.CS_TRANSLATION_SURVEY_DESC.' THEN s1.description '.
		    'WHEN '.CS_TRANSLATION_SURVEY_ENDTEXT.' THEN s1.endtext '.
		    'WHEN '.CS_TRANSLATION_SURVEY_HEADER.' THEN s1.custom_header '.
		    'WHEN '.CS_TRANSLATION_PAGE_TITLE.' THEN p1.title '.
		    'WHEN '.CS_TRANSLATION_QUESTION_TITLE.' THEN q1.title '.
		    'WHEN '.CS_TRANSLATION_QUESTION_DESC.' THEN q1.description '.
		    'WHEN '.CS_TRANSLATION_ANSWER_TITLE.' THEN a1.answer_label '.
		    'ELSE \'\' '.
		    'END AS source_value'
		    );
		
		$query->select('CASE a.item_type WHEN 7 THEN q2.title WHEN 8 THEN q2.title ELSE \'\' END AS question_title');

		$query->join('LEFT', '#__survey_surveys s1 ON s1.id = '.$surveyId.' AND a.item_type IN ('.CS_TRANSLATION_SURVEY_TITLE.','.CS_TRANSLATION_SURVEY_DESC.','.CS_TRANSLATION_SURVEY_ENDTEXT.','.CS_TRANSLATION_SURVEY_HEADER.')');
		$query->join('LEFT', '#__survey_pages p1 ON a.item_id = p1.id AND a.item_type = '.CS_TRANSLATION_PAGE_TITLE);
		$query->join('LEFT', '#__survey_questions q1 ON a.item_id = q1.id AND a.item_type IN ('.CS_TRANSLATION_QUESTION_TITLE.','.CS_TRANSLATION_QUESTION_DESC.')');
		$query->join('LEFT', '#__survey_questions q2 ON a.parent_id = q2.id AND a.item_type IN ('.CS_TRANSLATION_QUESTION_DESC.','.CS_TRANSLATION_ANSWER_TITLE.')');
		$query->join('LEFT', '#__survey_answers a1 ON a.item_id = a1.id AND a.item_type = '.CS_TRANSLATION_ANSWER_TITLE);
		
		$query->where('a.language_id = ' . $languageId);

		// Filter by published state
		$published = $this->getState('filter.published');

		if (is_numeric($published))
		{
			$query->where('a.published = ' . (int) $published);
			$query->where('l.published = ' . (int) $published);
		}
		elseif ($published === '')
		{
			$query->where('(a.published = 0 OR a.published = 1)');
			$query->where('(l.published = 0 OR l.published = 1)');
		}

		// Filter by search in title.
		$search = $this->getState('filter.search');

		if (!empty($search))
		{
			if (stripos($search, 'id:') === 0)
			{
				$query->where('a.id = ' . (int) substr($search, 3));
			}
			elseif (stripos($search, 'author:') === 0)
			{
				$search = $db->quote('%' . $db->escape(substr($search, 7), true) . '%');
				$query->where('(ua.name LIKE ' . $search . ' OR ua.username LIKE ' . $search . ')');
			}
			else
			{
				$search = $db->quote('%' . str_replace(' ', '%', $db->escape(trim($search), true) . '%'));
				$query->where('(a.translation LIKE ' . $search . ')');
			}
		}

		// Filter on the language.
		if ($language = $this->getState('filter.language'))
		{
			$query->where('a.language_id = ' . $db->quote($language));
		}

		// Add the list ordering clause.
		$orderCol = $this->state->get('list.ordering', 'a.id');
		$orderDirn = $this->state->get('list.direction', 'asc');

		// SQL server change
		if ($orderCol == 'language')
		{
			$orderCol = 'l.title';
		}

		$query->order($db->escape($orderCol . ' ' . $orderDirn));
// echo $query->dump();jexit();
		return $query;
	}
	
	public function getSurvey()
	{
	    $app = JFactory::getApplication();
	    $surveyId = $app->input->getInt('survey_id', 0);
	    if(!$surveyId)
	    {
	        throw new Exception(JText::_('JERROR_ALERTNOAUTHOR'), 403);
	    }
	    
	    $db = $this->getDbo();
	    $query = $db->getQuery(true)->select('id, title, alias, language')->from('#__survey_surveys')->where('id = ' . $surveyId);
	    $db->setQuery($query);
	    $survey = $db->loadObject();
	    
	    return $survey;
	}
}
