<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die();

use Joomla\Utilities\ArrayHelper;
use Joomla\String\StringHelper;

require_once JPATH_SITE.'/components/com_communitysurveys/helpers/constants.php';

class CommunitySurveysModelMyResponses extends JModelList
{
	protected $_item = null;

	public function __construct ($config = array())
	{
		if (empty($config['filter_fields']))
		{
			$config['filter_fields'] = array(
					'id', 'a.id',
					'title', 'a.title',
					'alias', 'a.alias',
					'checked_out', 'a.checked_out',
					'checked_out_time', 'a.checked_out_time',
					'catid', 's.catid', 'category_title',
					'published', 's.published', 'state', 
					'access', 'a.access', 'access_level',
					'created', 'a.created',
					'created_by', 'a.created_by',
					'ordering', 'a.ordering',
					'language', 'a.language',
					'publish_up', 'a.publish_up',
					'publish_down', 'a.publish_down'
			);
		}
		
		parent::__construct($config);
		
		$this->populateState();
	}

	protected function populateState ($ordering = 'ordering', $direction = 'ASC')
	{
		$app = JFactory::getApplication();
		$params = $app->getParams();
		
		// List state information
		$value = $app->input->get('limit', $app->getCfg('list_limit', 20), 'uint');
		$this->setState('list.limit', $value);
		
		$limitStart = $app->input->get('limitstart', 0, 'uint');
		$this->setState('list.start', $limitStart);
		
		$categories = $app->input->getArray(array('catid' => 'array'));
		$this->setState('filter.category_id', $categories['catid']);
		
		$orderCol = $app->input->get('filter_order', 'a.created');
		if (! in_array($orderCol, $this->filter_fields))
		{
			$orderCol = 'a.created';
		}
		
		$this->setState('list.ordering', $orderCol);
		
		$listOrder = $app->input->get('filter_order_Dir', 'DESC');
		if (! in_array(strtoupper($listOrder), array('ASC', 'DESC', '')))
		{
			$listOrder = 'DESC';
		}
		
		$filter = $app->input->get('list_filter', '', 'string');
		if (strlen($filter) > 1 || is_numeric($filter))
		{
			$this->setState('list.filter', $filter);
			
			$filterAllKeywords = $app->input->get('list_filter_all_keywords', 0, 'int');
			$this->setState('list.filter_all_keywords', $filterAllKeywords);
			
			$filterField = $app->input->get('list_filter_field', 'title', 'word');
			$this->setState('list.filter_field', $filterField);
		}
		
		$this->setState('list.direction', $listOrder);
		
		$recent = $app->input->getBool('recent', false);
		$this->setState('list.recent', $recent);
		$this->setState('params', $params);
		
		// Filter on published for those who do not have edit or edit.state rights.
		$this->setState('filter.published', 1);
		$this->setState('filter.language', JLanguageMultilang::isEnabled());
		$this->setState('filter.access', false);
		
		$this->setState('layout', $app->input->getString('layout'));
	}

	protected function getStoreId ($id = '')
	{
		// Compile the store id.
		$id .= ':' . serialize($this->getState('filter.published'));
		$id .= ':' . $this->getState('filter.access');
		$id .= ':' . serialize($this->getState('filter.category_id'));
		$id .= ':' . $this->getState('filter.category_id.include');
		$id .= ':' . $this->getState('filter.date_filtering');
		$id .= ':' . $this->getState('filter.date_field');
		$id .= ':' . $this->getState('filter.start_date_range');
		$id .= ':' . $this->getState('filter.end_date_range');
		$id .= ':' . $this->getState('filter.relative_date');
		
		return parent::getStoreId($id);
	}

	protected function getListQuery ()
	{
		// Get the current user for authorisation checks
		$user = JFactory::getUser();
		
		// Create a new query object.
		$db = $this->getDbo();
		$query = $db->getQuery(true);
		
		// Select the required fields from the table.
		$query->select(
				$this->getState('list.select',
						'a.id, a.survey_id, a.created, a.completed, a.state, s.title, s.alias, s.catid, s.attribs'));
		
		$query
			->select($this->getState('list.select', 's.published AS state'))
			->from('#__survey_responses AS a')
			->join('inner', '#__survey_surveys AS s on s.id = a.survey_id')
			->where('a.created_by = '. (int) $user->id);
		
		// Join over the categories.
		$query
			->select('c.title AS category_title, c.path AS category_route, c.access AS category_access, c.alias AS category_alias')
			->join('LEFT', '#__categories AS c ON c.id = s.catid');
		
		// Join over the categories to get parent category titles
		$query
			->select('parent.title as parent_title, parent.id as parent_id, parent.path as parent_route, parent.alias as parent_alias')
			->join('LEFT', '#__categories as parent ON parent.id = c.parent_id');
		
		// Join to check for category published state in parent categories up the tree
		$query->select('c.published');
		
		// Filter by a single or group of categories
		$categoryId = $this->getState('filter.category_id');
		
		if (is_numeric($categoryId) && $categoryId)
		{
			$type = $this->getState('filter.category_id.include', true) ? '= ' : '<> ';
			
			// Add subcategory check
			$includeSubcategories = $this->getState('filter.subcategories', false);
			$categoryEquals = 's.catid ' . $type . (int) $categoryId;
			
			if ($includeSubcategories)
			{
				$levels = (int) $this->getState('filter.max_category_levels', '1');
				
				// Create a subquery for the subcategory list
				$subQuery = $db->getQuery(true)
					->select('sub.id')
					->from('#__categories as sub')
					->join('INNER', '#__categories as this ON sub.lft > this.lft AND sub.rgt < this.rgt')
					->where('this.id = ' . (int) $categoryId);
				
				if ($levels >= 0)
				{
					$subQuery->where('sub.level <= this.level + ' . $levels);
				}
				
				// Add the subquery to the main query
				$query->where('(' . $categoryEquals . ' OR s.catid IN (' . $subQuery->__toString() . '))');
			}
			else
			{
				$query->where($categoryEquals);
			}
		}
		elseif (is_array($categoryId) && (count($categoryId) > 0))
		{
		    $categoryId = ArrayHelper::toInteger($categoryId);
			$categoryId = implode(',', $categoryId);
			
			if (! empty($categoryId))
			{
				$type = $this->getState('filter.category_id.include', true) ? 'IN' : 'NOT IN';
				$query->where('s.catid ' . $type . ' (' . $categoryId . ')');
			}
		}
		// Filter by Date Range or Relative Date
		$dateFiltering = $this->getState('filter.date_filtering', 'off');
		$dateField = $this->getState('filter.date_field', 'a.created');
		$nullDate = $db->getNullDate();
		$nowDate = JFactory::getDate()->toSql();
		
		switch ($dateFiltering)
		{
			case 'range':
				$startDateRange = $db->quote($this->getState('filter.start_date_range', $nullDate));
				$endDateRange = $db->quote($this->getState('filter.end_date_range', $nullDate));
				$query->where('(' . $dateField . ' >= ' . $startDateRange . ' AND ' . $dateField . ' <= ' . $endDateRange . ')');
				break;
			
			case 'relative':
				$relativeDate = (int) $this->getState('filter.relative_date', 0);
				$query->where($dateField . ' >= DATE_SUB(' . $nowDate . ', INTERVAL ' . $relativeDate . ' DAY)');
				break;
			
			case 'off':
			default:
				break;
		}
		
		// Process the filter for list views with user-entered filters
		if ($filter = $this->getState('list.filter'))
		{
			// Clean filter variable
			$filter = StringHelper::strtolower($filter);
			$idFilter = (int) $filter;
			$filterField = $this->getState('list.filter_field', 'title');
			
			switch ($filterField)
			{
				case 'id':
				    $query->where('a.id = ' . $idFilter . ' ');
				    break;
				
				case 'title':
				default:
				    if(is_numeric($filter))
				    {
				        $idFilter = (int) $filter;
				        $query->where('a.id = ' . $idFilter . ' ');
				    }
				    else 
				    {
					   $stopwords = array(
							"a", "about", "above", "above", "across", "after", "afterwards", "again", "against", "all", "almost", "alone", "along", "already", "also","although","always",
							"am","among", "amongst", "amoungst", "amount",  "an", "and", "another", "any","anyhow","anyone","anything","anyway", "anywhere", "are", "around", "as",  "at",
							"back","be","became", "because","become","becomes", "becoming", "been", "before", "beforehand", "behind", "being", "below", "beside", "besides", "between",
							"beyond", "bill", "both", "bottom","but", "by", "call", "can", "cannot", "cant", "co", "con", "could", "couldnt", "cry", "de", "describe", "detail", "do", "done",
							"down", "due", "during", "each", "eg", "eight", "either", "eleven","else", "elsewhere", "empty", "enough", "etc", "even", "ever", "every", "everyone", "everything",
							"everywhere", "except", "few", "fifteen", "fify", "fill", "find", "fire", "first", "five", "for", "former", "formerly", "forty", "found", "four", "from", "front",
							"full", "further", "get", "give", "go", "had", "has", "hasnt", "have", "he", "hence", "her", "here", "hereafter", "hereby", "herein", "hereupon", "hers", "herself",
							"him", "himself", "his", "how", "however", "hundred", "ie", "if", "in", "inc", "indeed", "interest", "into", "is", "it", "its", "itself", "keep", "last", "latter",
							"latterly", "least", "less", "ltd", "made", "many", "may", "me", "meanwhile", "might", "mill", "mine", "more", "moreover", "most", "mostly", "move", "much", "must",
							"my", "myself", "name", "namely", "neither", "never", "nevertheless", "next", "nine", "no", "nobody", "none", "noone", "nor", "not", "nothing", "now", "nowhere", "of",
							"off", "often", "on", "once", "one", "only", "onto", "or", "other", "others", "otherwise", "our", "ours", "ourselves", "out", "over", "own","part", "per", "perhaps",
							"please", "put", "rather", "re", "same", "see", "seem", "seemed", "seeming", "seems", "serious", "several", "she", "should", "show", "side", "since", "sincere", "six",
							"sixty", "so", "some", "somehow", "someone", "something", "sometime", "sometimes", "somewhere", "still", "such", "system", "take", "ten", "than", "that", "the", "their",
							"them", "themselves", "then", "thence", "there", "thereafter", "thereby", "therefore", "therein", "thereupon", "these", "they", "thickv", "thin", "third", "this", "those",
							"though", "three", "through", "throughout", "thru", "thus", "to", "together", "too", "top", "toward", "towards", "twelve", "twenty", "two", "un", "under", "until", "up",
							"upon", "us", "very", "via", "was", "we", "well", "were", "what", "whatever", "when", "whence", "whenever", "where", "whereafter", "whereas", "whereby", "wherein",
							"whereupon", "wherever", "whether", "which", "while", "whither", "who", "whoever", "whole", "whom", "whose", "why", "will", "with", "within", "without", "would", "yet",
							"you", "your", "yours", "yourself", "yourselves", "the");
							
                        $keywords = array_diff(explode(' ', $filter), $stopwords);
                        $filters = array();
    					
    					foreach ($keywords as $keyword)
    					{
    						if (strlen($keyword) > 2)
    						{
    							$filters[] = 'LOWER( s.title ) LIKE ' . $db->q('%' . $db->escape($keyword, true) . '%', false);
    						}
    					}
    					// Default to 'title' if parameter is not valid
    					if ($filterAllKeywords = $this->getState('list.filter_all_keywords'))
    					{
    						$query->where('(' . implode(' AND ', $filters) . ')');
    					}
    					else
    					{
    						$query->where('(' . implode(' OR ', $filters) . ')');
    					}
				    }
				    					
					break;
			}
		}
		
		// Add the list ordering clause.
		$query->order($this->getState('list.ordering', 'a.created') . ' ' . $this->getState('list.direction', 'DESC'));
		
// 		echo $query->dump();
// 		jexit();
		return $query;
	}

	public function getItems ()
	{
		$items = parent::getItems();
		$app = JFactory::getApplication();
		$input = $app->input;
		
		// Get the global params
		$globalParams = JComponentHelper::getParams('com_communitysurveys', true);
		
		// Convert the parameter fields into objects.
		if(!empty($items))
		{
    		foreach ($items as &$item)
    		{
    			$item->slug = $item->alias ? ($item->id . ':' . $item->alias) : $item->id;
    			$item->parent_slug = ($item->parent_alias) ? ($item->parent_id . ':' . $item->parent_alias) : $item->parent_id;
    			
    			// No link for ROOT category
    			if ($item->parent_alias == 'root')
    			{
    				$item->parent_slug = null;
    			}
    			
    			$item->catslug = $item->category_alias ? ($item->catid . ':' . $item->category_alias) : $item->catid;
    			
    			$surveyParams = new JRegistry();
    			$surveyParams->loadString($item->attribs);
    			
    			// Unpack readmore and layout params
    			$item->alternative_readmore = $surveyParams->get('alternative_readmore');
    			$item->layout = $surveyParams->get('layout');
    			
    			$item->params = clone $this->getState('params');
    			
    			/*
    			 * For lists, survey params override menu item params only if menu
    			 * param = 'use_survey' Otherwise, menu item params control the
    			 * layout If menu item is 'use_survey' and there is no survey param, use global
    			 */
    			if (
    					($input->getString('layout') == 'blog') || 
    					($input->getString('view') == 'featured') ||
    					($this->getState('params')->get('layout_type') == 'blog')
    			)
    			{
    				// Create an array of just the params set to 'use_survey'
    				$menuParamsArray = $this->getState('params')->toArray();
    				$surveyArray = array();
    				
    				foreach ($menuParamsArray as $key => $value)
    				{
    					if ($value === 'use_survey')
    					{
    						// If the survey has a value, use it
    						if ($surveyParams->get($key) != '')
    						{
    							// Get the value from the survey
    							$surveyArray[$key] = $surveyParams->get($key);
    						}
    						else
    						{
    							// Otherwise, use the global value
    							$surveyArray[$key] = $globalParams->get($key);
    						}
    					}
    				}
    				
    				// Merge the selected survey params
    				if (count($surveyArray) > 0)
    				{
    					$surveyParams = new JRegistry();
    					$surveyParams->loadArray($surveyArray);
    					$item->params->merge($surveyParams);
    				}
    			}
    			else
    			{
    				// For non-blog layouts, merge all of the survey params
    				$item->params->merge($surveyParams);
    			}
    			
    			// Get display date
    			$item->displayDate = $item->created;
    		}
    		
    		reset($items);
		}
		
		return $items;
	}

	public function getPagination ()
	{
		$page = parent::getPagination();
		$view = JFactory::getApplication()->input->getCmd('view');
		
		if ($view == 'myresponses')
		{
			$page->setAdditionalUrlParam('view', 'myresponses');
			
			if ($this->state->get('list.ordering') == 'hits')
			{
				$page->setAdditionalUrlParam('filter_order', $this->state->get('list.ordering'));
				$page->setAdditionalUrlParam('filter_order_Dir', $this->state->get('list.direction'));
			}
		}
		
		return $page;
	}
}
