<?php
/**
 * @package     corejoomla.administrator
 * @subpackage  com_communitysurveys
 *
 * @copyright   Copyright (C) 2009 - 2019 corejoomla.com. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */
defined('_JEXEC') or die();

use Joomla\Utilities\ArrayHelper;

jimport('joomla.application.component.modelitem');
require_once JPATH_ADMINISTRATOR . '/components/com_communitysurveys/models/responsebase.php';

class CommunitySurveysModelResponse extends CommunitySurveysModelResponseBase
{
	protected function populateState ()
	{
		$app = JFactory::getApplication('site');
	
		// Load state from the request.
		$surveyId = $app->input->getInt('id');
		$this->setState('filter.survey_id', $surveyId);
		
		$responseId = $app->input->getInt('rid');
		$this->setState('filter.response_id', $responseId);
		
		$uniqueKey = $app->input->getCmd('key');
		$this->setState('filter.unique_key', $uniqueKey);

		$surveyKey = $app->input->getCmd('skey');
		$this->setState('filter.survey_key', $surveyKey);
		
		$pageId = $app->input->getInt('pid');
		$this->setState('filter.page_id', $pageId);
		
		$offset = $app->input->getUInt('limitstart');
		$this->setState('list.offset', $offset);
	
		// Load the parameters.
		$params = JComponentHelper::getParams('com_communitysurveys');
		$this->setState('params', $params);
	
		// TODO: Tune these values based on other permissions.
		$user = JFactory::getUser();
	
		if ((! $user->authorise('core.edit.state', 'com_communitysurveys')) && (! $user->authorise('core.edit', 'com_communitysurveys')))
		{
			$this->setState('filter.published', 1);
			$this->setState('filter.archived', 2);
		}
	
		$this->setState('filter.language', JLanguageMultilang::isEnabled());
	}
	
	protected $_error = '';

	function __construct ($config)
	{
		$config['event_before_save'] 	= 'onResponseBeforeSave';
		$config['event_after_save'] 	= 'onResponseAfterSave';
		$config['event_after_delete'] 	= 'onResponseAfterDelete';
	    $config['event_before_delete'] 	= 'onResponseBeforeDelete';
	    $config['event_change_state'] 	= 'onResponseChangeState';
	    $config['events_map'] = array(
	            'delete'       => 'communitysurveys',
	            'save'         => 'communitysurveys',
	            'change_state' => 'communitysurveys'
	    );
	    
		parent::__construct($config);
		
		$this->populateState();
	}

	public function getError ($i = null, $toString = true)
	{
		return $this->_error;
	}

	public function setError ($error)
	{
		$this->_error = $error;
	}
	
	protected function getUser()
	{
		$user = $this->getState('user');
		if(empty($user))
		{
			$user = JFactory::getUser();
		}
	
		return $user;
	}
	
	/**
	 * Creates the new response or updates the response if already exists
	 * and returns $response instance
	 *
	 * @throws Exception with message
	 * @return survey Survey instance
	 */
	public function createOrUpdateResponse ($surveyId = null, $skipCreate = false, $findResponse = false)
	{
		$app 			= JFactory::getApplication();
		$user 			= $this->getUser();
		$db 			= JFactory::getDbo();
		$params			= JComponentHelper::getParams('com_communitysurveys');
		
		$surveyId 		= (! empty($surveyId)) ? $surveyId : (int) $this->getState('filter.survey_id');
		$responseId 	= (int) $this->getState('filter.response_id');
		$uniqueKey 		= $this->getState('filter.unique_key');
		$surveyKey 		= $this->getState('filter.survey_key');
		$debug			= $params->get('debug_mode') ? true : false;
		$objResponse	= null;
		
		$created = 'now';
		$userId = $user->id;
		
		if(!$surveyId && !$uniqueKey && $surveyKey)
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNAUTHORISED').($debug ? '| RC=100' : ''), 403);
		}
		
		if($debug)
		{
			$app->enqueueMessage('State 1: Survey ID=' . $surveyId . '| Response ID=' .$responseId . '| Unique Key='.$uniqueKey.'| Survey Key='.$surveyKey); 
		}
		
		if(!empty($uniqueKey))
		{
			$query = $db->getQuery(true)
				->select('survey_id, user_id')
				->from('#__survey_keys')
				->where('key_name = '.$db->q($uniqueKey));
			
			$db->setQuery ( $query );
			$keyObj = $db->loadObject();
				
			if($keyObj)
			{
				$surveyId = $keyObj->survey_id;
		
				if(!empty($keyObj->user_id))
				{
					$userId = $keyObj->user_id;
				}
			}
		}
		else if(!empty($surveyKey))
		{
			$query = $db->getQuery(true)
				->select('id')
				->from('#__survey_surveys')
				->where('survey_key = '.$db->q($surveyKey));
			
			$db->setQuery($query);
			$surveyId = $db->loadResult();
		}
		
		if($debug)
		{
			$app->enqueueMessage('State 2: Survey ID=' . $surveyId . '| Response ID=' .$responseId . '| Unique Key='.$uniqueKey.'| Survey Key='.$surveyKey);
		}
		
		if(!$surveyId)
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNAUTHORISED').($debug ? '| RC=1' : ''), 403);
		}
		
		$surveyModel	= JModelLegacy::getInstance( 'survey', 'CommunitySurveysModel' );
		$surveyModel->getState(); // access the state first so that it can be modified
		$surveyModel->setState('filter.survey_id', $surveyId);
		$survey 		= $surveyModel->getItem($surveyId);
		
		if( ! $survey )
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNAUTHORISED').($debug ? '| RC=2' : ''), 403);
		}
		
		if ( $survey->published != 1 )
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNAUTHORISED').($debug ? '| RC=3' : ''), 403);
		}
		
		if(!$user->authorise('core.respond', 'com_communitysurveys.survey.'.$surveyId)) 
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNAUTHORISED').($debug ? '| RC=4' : ''), 403);
		}
		
		if($survey->private == 1 && empty($surveyKey) && empty($uniqueKey))
		{
			throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNAUTHORISED').($debug ? '| RC=5' : ''), 403);
		}
		
		if($survey->publish_up != '0000-00-00 00:00:00')
		{
			$date = JFactory::getDate($survey->publish_up);
			$compareTo = JFactory::getDate();
		
			if(($compareTo->toUnix() - $date->toUnix()) < 0)
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_NOT_YET_UP').($debug ? '| RC=6' : ''), 403);
			}
		}
		
		if($survey->publish_down != '0000-00-00 00:00:00')
		{
			$date = JFactory::getDate($survey->publish_down);
			$compareTo = JFactory::getDate();
		
			if(($compareTo->toUnix() - $date->toUnix()) > 0)
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_TIME_UP').($debug ? '| RC=7' : ''), 403);
			}
		}
		
		$survey->key = $uniqueKey;
		if ($skipCreate && !$findResponse) 
		{
			// return here if no need to find response from chosen restriction methods
			return $survey;
		}
		
		if(!empty($uniqueKey))
		{
			$query = $db->getQuery(true)
				->select('id, completed > created as completed, created_by, survey_key')
				->from('#__survey_responses')
				->where('survey_key = '.$db->q($uniqueKey))
				->order('created desc');
			
			$db->setQuery($query);
			$objResponse = $db->loadObject();
			
			if($debug)
			{
				$app->enqueueMessage('State 3: Survey ID=' . $surveyId . '| Response ID=' .$objResponse->id . '| Unique Key='.$uniqueKey.'| Survey Key='.$surveyKey);
			}
			
			if(!empty($objResponse) && $objResponse->completed == 1)
			{
				$message =  JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_ALREADY_TOOK');
				if($survey->params->get('show_report', 0) == 1 && $user->authorise('core.results', 'com_communitysurveys.survey.'.$survey->id))
				{
					$resultsUrl = JRoute::_(CommunitySurveysHelperRoute::getResponseRoute($survey->id, $objResponse->id).'&key='.$uniqueKey);
					$message = $message . ' ' . JText::sprintf('COM_COMMUNITYSURVEYS_CLICK_HERE_TO_VEW_ANSWERS', $resultsUrl);
				}
				
				throw new Exception($message.($debug ? '| RC=8' : ''), 403);
			}
		}
		else
		{
			if(strpos($survey->restrictions, 'cookie') !== false)
			{
				$cookieName = trim(JApplicationHelper::getHash($app->getName() . CS_COOKIE_PREFIX . $surveyId));
				$cookieKey = trim($app->input->cookie->getCmd($cookieName, ''));
				
				if($debug)
				{
					$app->enqueueMessage('State 4: Survey ID=' . $surveyId . '| Response ID=' .$responseId . '| Unique Key='.$uniqueKey.'| Survey Key='.$surveyKey . '| Cookie Key='.$cookieKey);
				}

				if(!empty($cookieKey))
				{
					$query = $db->getQuery(true)
						->select('id, completed > created as completed, created_by, survey_key')
						->from('#__survey_responses')
						->where('survey_key = '.$db->q($cookieKey))
						->order('created desc');
					
					$db->setQuery($query);
					$objResponse = $db->loadObject();
				}
				
				if($debug)
				{
					$app->enqueueMessage('State 5: Survey ID=' . $surveyId . '| Response ID=' .print_r($objResponse, true) . '| Unique Key='.$uniqueKey.'| Survey Key='.$surveyKey . '| Cookie Key='.$cookieKey);
				}
				
				if(!empty($objResponse) && $objResponse->completed == 1)
				{
					$message =  JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_ALREADY_TOOK');
					if($survey->params->get('show_report', 0) == 1 && $user->authorise('core.results', 'com_communitysurveys.survey.'.$survey->id))
					{
						$resultsUrl = JRoute::_(CommunitySurveysHelperRoute::getResponseRoute($survey->id, $objResponse->id).'&key='.$cookieKey);
						$message = $message . ' ' . JText::sprintf('COM_COMMUNITYSURVEYS_CLICK_HERE_TO_VEW_ANSWERS', $resultsUrl);
					}
					
					throw new Exception($message.($debug ? '| RC=9' : ''), 403);
				}
			}
			
			if(empty($objResponse) && strpos($survey->restrictions, 'ip') !== false)
			{
				$ipAddress = CjLibUtils::getUserIpAddress();
				$query = $db->getQuery(true)
					->select('id, completed > created as completed, created_by, survey_key')
					->from('#__survey_responses')
					->where('survey_id = '.$surveyId.' and ip_address='.$db->q($ipAddress));
			
				$db->setQuery ( $query );
				$objResponse = $db->loadObject();
				
				if($debug)
				{
					$app->enqueueMessage('State 6: Survey ID=' . $surveyId . '| Response ID=' .$objResponse->id . '| Unique Key='.$uniqueKey.'| Survey Key='.$surveyKey);
				}
			
				if(!empty($objResponse) && $objResponse->completed == 1)
				{
					$message =  JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_ALREADY_TOOK');
					if($survey->params->get('show_report', 0) == 1 && $user->authorise('core.results', 'com_communitysurveys.survey.'.$survey->id))
					{
						$resultsUrl = JRoute::_(CommunitySurveysHelperRoute::getResponseRoute($survey->id, $objResponse->id).'&key='.$objResponse->survey_key);
						$message = $message . ' ' . JText::sprintf('COM_COMMUNITYSURVEYS_CLICK_HERE_TO_VEW_ANSWERS', $resultsUrl);
					}
					
					throw new Exception($message.($debug ? '| RC=10' : ''), 403);
				}
			}
			
			if(!$user->guest && empty($objResponse))
			{
				$query = $db->getQuery(true)
					->select('id, completed > created as completed, created_by, survey_key')
					->from('#__survey_responses')
					->where('survey_id = '.$surveyId.' and created_by = '.$user->id.' and (state = 1 || state = 2)')
					->order('created desc');
			
				$db->setQuery ( $query );
				$objResponse = $db->loadObject();
				
				if($debug)
				{
					$app->enqueueMessage('State 7: Survey ID=' . $surveyId . '| Response ID=' . ($objResponse ? $objResponse->id : 0) . '| Unique Key='.$uniqueKey.'| Survey Key='.$surveyKey);
				}
					
				if(!empty($objResponse) && $objResponse->completed == 1)
				{
					if(strpos($survey->restrictions, 'username') !== false)
					{
						$message =  JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_ALREADY_TOOK');
						if($survey->params->get('show_report', 0) == 1 && $user->authorise('core.results', 'com_communitysurveys.survey.'.$survey->id))
						{
							$resultsUrl = JRoute::_(CommunitySurveysHelperRoute::getResponseRoute($survey->id, $objResponse->id).'&key='.$objResponse->survey_key);
							$message = $message . ' ' . JText::sprintf('COM_COMMUNITYSURVEYS_CLICK_HERE_TO_VEW_ANSWERS', $resultsUrl);
						}
						
						throw new Exception($message.($debug ? '| RC=11' : ''), 403);
					} 
					else 
					{
						$objResponse = null;
					}
				}
			}
			
			$countries = $survey->params->get('countries');
			if($survey->params->get('anonymous', 0) != 1 && !empty($countries))
			{
				$ipAddress = CjLibUtils::getUserIpAddress();
				if(!empty($ipAddress))
				{
					$location = CjLibUtils::getUserLocation($ipAddress);
					if(!empty($location['country_code']) && !in_array($location['country_code'], $countries))
					{
						throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_NOT_AVAILABLE_IN_YOUR_COUNTRY').($debug ? '| RC=12' : ''), 403);
					}
				}
			}
		}
		
		if(!empty($objResponse))
		{
			if(($survey->params->get('anonymous', 1) == 1) || ($objResponse->created_by == $userId))
			{
				$responseId = $objResponse->id;
				$uniqueKey = $objResponse->survey_key;
			}
		} 
		else if(!empty($uniqueKey))
		{
			// new survey with created key from invite page. no response should exist with this key so new response is created next
			// check if this key is legimate
			$query = $db->getQuery(true)
				->select('count(*)')
				->from('#__survey_keys')
				->where('key_name = '.$db->q($uniqueKey));
			
			$db->setQuery($query);
			$count = (int) $db->loadResult();
		
			if($count <= 0) 
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNAUTHORISED').($debug ? '| RC=13' : ''), 403);
			}
		}
		
		if($skipCreate && $findResponse)
		{
			// we don't want to create new response, just return whatever is discovered
			$survey->response_id = $responseId;
			$survey->key = $uniqueKey;
			$survey->curr_page = 0;
			
			if($responseId)
			{
				$query = $db->getQuery(true)
					->select('curr_page')
					->from('#__survey_responses')
					->where('id = ' . $responseId);
				$db->setQuery($query);
				$survey->curr_page = 0;
				
				try
				{
					$survey->curr_page = (int) $db->loadResult();
				}
				catch (Exception $e){}
			}
			
			return $survey;
		}
		
		if(!$responseId || empty($uniqueKey))
		{
			// No response id, so there is no way to track if he has responded. Create a survey response now.
			$query = $db->getQuery(true)
				->select('count(*)')
				->from('#__survey_responses')
				->where('survey_id = '.$surveyId);
			
			$db->setQuery($query);
			$maxResponses = $db->loadResult();
			
			if(($survey->max_responses > 0) && ($survey->max_responses <= $maxResponses)) 
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_SURVEY_REACHED_MAX_LIMIT').($debug ? '| RC=14' : ''), 403);
			}
			
			if(!$uniqueKey)
			{
				$uniqueKey = $this->createSurveyKeys($surveyId, 1, false, true);
			}
			
			if(empty($uniqueKey) || count($uniqueKey) == 0)
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_NOT_ENOUGH_CREDITS_AVAILABLE').($debug ? '| RC=15' : ''), 403);
			}
			else
			{
				$uniqueKey = is_array($uniqueKey) ? $uniqueKey[0] : $uniqueKey;
				$hash = JApplicationHelper::getHash($app->getName() . CS_COOKIE_PREFIX . $surveyId);
				$expire = (int)$survey->params->get('cookie_lifetime', 525600);
				$expire = $expire ? $expire : 525600;
				$app->input->cookie->set($hash, $uniqueKey, time() + 60 * $expire,	'/');
			}
			
			$created 				= JFactory::getDate()->toSql();
			$newRec 				= new stdClass();
			$newRec->survey_id 		= $surveyId;
			$newRec->created_by 	= 0;
// 			$newRec->completed 		= '0000-00-00 00:00:00';
			$newRec->created 		= $created;
			$newRec->survey_key		= $uniqueKey;
			$newRec->ip_address		= '0.0.0.0';
			$newRec->state			= 1;

			if( $survey->params->get('anonymous', 0) != 1 )
			{
				$newRec->created_by = $userId;
				$newRec->ip_address = CjLibUtils::getUserIpAddress();
			}

			JPluginHelper::importPlugin($this->events_map['save']);
			$context 				= $this->option . '.' . $this->name;
			$result 				= $app->triggerEvent($this->event_before_save, array($context, $newRec, true));
				
			if (in_array(false, $result, true))
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNABLE_TO_CREATE_RESPONSE').($debug ? '| RC=16' : ''), 500);
			}
			
			try
			{
				if (! $db->insertObject('#__survey_responses', $newRec))
				{
					// DB error
					throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNABLE_TO_CREATE_RESPONSE').($debug ? '| RC=17' : ''), 500);
				}
			}
			catch (Exception $e)
			{
				throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNABLE_TO_CREATE_RESPONSE').($debug ? '| RC=18'.print_r($e,true) : ''), 500);
			}
			
			$responseId = (int) $db->insertid();
			if($responseId)
			{
				$query = $db->getQuery(true)
					->update('#__survey_keys')
					->set('response_id = '. $responseId)
					->set('response_status = 2')
					->where('key_name = '.$db->q($uniqueKey) . ' and survey_id = '.$surveyId);
				
				$db->setQuery($query);
				try
				{
					if (! $db->execute())
					{
						// DB error
						throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNABLE_TO_CREATE_RESPONSE').($debug ? '| RC=19' : ''), 500);
					}
				}
				catch (Exception $e)
				{
					throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_UNABLE_TO_CREATE_RESPONSE').($debug ? '| RC=20' : ''), 500);
				}
			}
			
			if( $survey->params->get('anonymous', 0) != 1 )
			{
				// update tracking information
				$tracking 					= new stdClass();
				$location 					= CjLibUtils::getUserLocation($newRec->ip_address);
				$browser 					= CJFunctions::get_browser();
					
				$tracking->post_id 			= $responseId;
				$tracking->post_type 		= 1;
				$tracking->ip_address 		= $newRec->ip_address;
				$tracking->country 			= $location['country_code'];
				$tracking->city				= $location['city'];
				$tracking->os 				= $browser['platform'];
				$tracking->browser_name 	= $browser['name'];
				$tracking->browser_version 	= $browser['version'];
				$tracking->browser_info 	= $browser['userAgent'];
					
				try
				{
					$db->insertObject('#__survey_tracking', $tracking);
				}
				catch (Exception $e){}
			}
		}
		
		$query = $db->getQuery(true)
			->select('curr_page')
			->from('#__survey_responses')
			->where('id = ' . $responseId);
		$db->setQuery($query);
		$survey->curr_page = 0;
		
		try
		{
			$survey->curr_page = (int) $db->loadResult();
		}
		catch (Exception $e){}
		
		$survey->response_id = $responseId;
		$survey->key = $uniqueKey;
		
		return $survey;
	}

	/**
	 * Saves the response details.
	 * Saves the questions in the current page if page id is given or saves
	 * questions given in <code>questionIds</code> parameter.
	 *
	 * @param number $surveyId
	 *        	Id of the survey
	 * @param number $responseId
	 *        	Id of the response
	 * @param number $pageId
	 *        	If of the current page
	 * @param array $questionIds
	 *        	Array of question ids in this page
	 * @throws Exception on error
	 * @return boolean true if success else false
	 */
	public function save ($data = null)
	{
		$app 		= JFactory::getApplication();
		$user 		= $this->getUser();
		$db 		= JFactory::getDbo();
		$context    = $this->option . '.' . $this->name;
		
		JPluginHelper::importPlugin('communitysurveys');
		
		try
		{
			$surveyId = (int) $this->getState('filter.survey_id');
			$responseId = (int) $this->getState('filter.response_id');
			$pageId = (int) $this->getState('filter.page_id');
			
			$surveyModel	= JModelLegacy::getInstance( 'survey', 'CommunitySurveysModel' );
			$surveyModel->getState(); // access the state first so that it can be modified
			$surveyModel->setState('filter.survey_id', $surveyId);
			$survey 		= $surveyModel->getItem($surveyId);
			
			// if it is anoymous survey, we need to check only response id with key.
			if( ! $survey->params->get('anonymous', 0) )
			{
				// validate if legimate user
				$query = $db->getQuery(true)
					->select('created_by')
					->from('#__survey_responses')
					->where('id = ' . $responseId);
				
				$db->setQuery($query);
				$created_by = (int) $db->loadResult();
				
				if ($created_by != $user->id)
				{
					// if the private key available, make use of it even if user not logged in
					$key = $app->input->getCmd('key');
					if($key)
					{
						$query = $db->getQuery(true)
							->select('user_id')
							->from('#__survey_keys')
							->where('key_name = '.$db->q($key))
							->where('survey_id = '.$surveyId);
						
						$db->setQuery($query);
						$created_by = (int) $db->loadResult();
						
						if(!$created_by)
						{
							throw new Exception(JText::_('JERROR_ALERTNOAUTHOR').'| RC=21', 403);
						}
					}
					else 
					{
						throw new Exception(JText::_('JERROR_ALERTNOAUTHOR').'| RC=22', 403);
					}
				}
			}
			
			// check if response belong to this survey
			$query = $db->getQuery(true)
				->select('count(*)')
				->from('#__survey_responses')
				->where('id = ' . $responseId)
				->where('survey_id = '.$surveyId);
			$db->setQuery($query);
			$count = (int) $db->loadResult();
			
			if(!$count)
			{
				throw new Exception(JText::_('JERROR_ALERTNOAUTHOR').'| RC=23', 403);
			}
			
			JLoader::import('questions', JPATH_ROOT.'/components/com_communitysurveys/models');
			$questionsModel = JModelLegacy::getInstance( 'questions', 'CommunitySurveysModel' );
			
			$questionsModel->getState(); // access the state first so that it can be modified
			$questionsModel->setState('filter.page_id', $pageId);
			$questionsModel->setState('list.ordering', 'a.sort_order');
			$questionsModel->setState('list.direction', 'asc');
			$questionsModel->setState('list.limit', 0);
			$questionsModel->setState('filter.survey_id', $surveyId);
			
			$questions = $questionsModel->getItems();
			$return = new stdClass();
			$return->page_id = $return->finalize = 0;

			if (empty($questions))
			{
				return false;
			}
			
			$answers = array();
			foreach ($questions as $question)
			{
				$free_text = null;
				switch ($question->question_type)
				{
					case CS_CHOICE_RADIO: // Choice - Radio
					case CS_CHOICE_SELECT: // Choice - Select box
					case CS_CHOICE_NPS: // Choice - Net Promoter Score
					case CS_IMAGE_CHOOSE_IMAGE: // Image - Radio
					case CS_CHOICE_SLIDER: // Choice - Slider
						
						$answer_id = $app->input->post->getInt('answer-' . $question->id, 0);
						$free_text = $app->input->post->getString('free-text-' . $question->id, null);
						
						if ($answer_id)
						{
							$answer = array();
							$answer['question_id'] = $question->id;
							$answer['answer_id'] = $answer_id;
							$answer['column_id'] = 0;
							$answer['free_text'] = null;
							array_push($answers, $answer);
						}
						
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, array($answer_id), $question->rules, $return, 1);
						}
						
						break;
					
					case CS_CHOICE_CHECKBOX: // Choice - Checkbox
					case CS_IMAGE_CHOOSE_IMAGES: // Image - Checkbox
						
						$answer_ids = $app->input->post->getArray(array('answer-' . $question->id => 'array'));
						$free_text = $app->input->post->getString('free-text-' . $question->id, null);
						
						$answer_ids = $answer_ids['answer-' . $question->id];
						$answer_ids = ArrayHelper::toInteger($answer_ids);
						
						if (! empty($answer_ids))
						{
							
							foreach ($answer_ids as $answer_id)
							{
								
								$answer = array();
								$answer['question_id'] = $question->id;
								$answer['answer_id'] = $answer_id;
								$answer['column_id'] = 0;
								$answer['free_text'] = null;
								array_push($answers, $answer);
							}
						}
						
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, $answer_ids, $question->rules, $return, 1);
						}
						break;
					
					case CS_GRID_RADIO: // Grid - Radio
					case CS_GRID_MATCHING: // Matching
						
						$rows = array();
						$grid_answers = array();
						
						foreach ($question->answers as $answer)
						{
							if ($answer->answer_type == 'x')
							{
								$rows[] = $answer;
							}
						}
						
						$free_text = $app->input->post->getString('free-text-' . $question->id, null);
						foreach ($rows as $row)
						{
							$column_id = $app->input->post->getInt('answer-' . $question->id . '-' . $row->id, 0);
							if ($column_id)
							{
								$answer = array();
								$answer['question_id'] = $question->id;
								$answer['answer_id'] = $row->id;
								$answer['column_id'] = $column_id;
								$answer['free_text'] = null;
								
								array_push ( $grid_answers, array($row->id => $column_id) );
								array_push($answers, $answer);
							}
						}
						
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, $grid_answers, $question->rules, $return, 2);
						}
						break;
					
					case CS_GRID_CHECKBOX: // Grid - Checkbox
						
						$rows = array();
						$grid_answers = array();
						
						foreach ($question->answers as $answer)
						{
							if ($answer->answer_type == 'x')
							{
								$rows[] = $answer;
							}
						}
						
						$free_text = $app->input->post->getString('free-text-' . $question->id, null);
						foreach ($rows as $row)
						{
							$column_ids = $app->input->post->getArray(array('answer-' . $question->id . '-' . $row->id => 'array'));
							$column_ids = $column_ids['answer-' . $question->id . '-' . $row->id];
							$column_ids = ArrayHelper::toInteger($column_ids);
							
							if (! empty($column_ids))
							{
								foreach ($column_ids as $column_id)
								{
									$answer = array();
									$answer['question_id'] = $question->id;
									$answer['answer_id'] = $row->id;
									$answer['column_id'] = $column_id;
									$answer['free_text'] = null;
									
									array_push ( $grid_answers, array($row->id => $column_id) );
									array_push($answers, $answer);
								}
							}
						}
						
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, $grid_answers, $question->rules, $return, 2);
						}
						
						break;
					
					case CS_FREE_TEXT_SINGLE_LINE: // Freetext - Singleline
					case CS_FREE_TEXT_MULTILINE: // Freetext - Multiline
					case CS_FREE_TEXT_PASSWORD: // Freetext - Password
					case CS_SPECIAL_EMAIL: // Special - Email
					case CS_SPECIAL_SIGNATURE: // Special - Signature
						
						$free_text = $app->input->post->getString('free-text-' . $question->id, null);
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, array($free_text), $question->rules, $return, 3);
						}
						break;
						
					case CS_SPECIAL_CALENDAR: // Special - Calendar
							
						$free_text = $app->input->post->getString('free-text-' . $question->id, null);
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, array($free_text), $question->rules, $return, 3);
						}
						break;

					case CS_FREE_TEXT_RICH_TEXT: // Freetext - Rich text
						
					    $free_text = JComponentHelper::filterText($app->input->post->get('free-text-' . $question->id, '', 'raw'));
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$text = strip_tags($free_text);
							$this->validateRules($question->id, array($text), $question->rules, $return, 3);
						}
						break;
						
					case CS_FILE_UPLOAD: // File upload
						
					    $allowedExtensions = $question->params->get('allowed_file_types', null);
					    $free_text = $this->uploadFile('free-text-' . $question->id, $allowedExtensions);
					    
				        if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
				        {
				        	$this->validateRules($question->id, array($free_text), $question->rules, $return, 3);
				        }
					    break;
					
					case CS_SPECIAL_NAME: // Special - Name
						
						$names = $app->input->getArray(array('user-name-'.$question->id=>'array'));
						if(count($names['user-name-'.$question->id]) == 3)
						{
							$free_text = implode('|', $names['user-name-'.$question->id]);
							$tmp_text = str_replace('|', '', $free_text);
							
							if(empty($tmp_text))
							{
								$free_text = '';
							}
							
							$this->validateRules($question->id, array($free_text), $question->rules, $return, 3);
						}
						break;
						
					case CS_SPECIAL_ADDRESS: // Special - Address
						
						$addr_name = $app->input->post->getString('address-name-'.$question->id, '');
						$addr_line1 = $app->input->post->getString('address-line1-'.$question->id, '');
						$addr_line2 = $app->input->post->getString('address-line2-'.$question->id, '');
						$addr_city = $app->input->post->getString('address-city-'.$question->id, '');
						$addr_state = $app->input->post->getString('address-state-'.$question->id, '');
						$addr_country = $app->input->post->getString('address-country-'.$question->id, '');
						$addr_zip = $app->input->post->getString('address-zip-'.$question->id, '');
						
						if(!empty($addr_name) && !empty($addr_line1) && !empty($addr_city) && !empty($addr_state) && !empty($addr_zip))
						{
							$free_text = $addr_name.'|||'.$addr_line1.'|||'.$addr_line2.'|||'.$addr_city.'|||'.$addr_state.'|||'.$addr_country.'|||'.$addr_zip;
							
							$this->validateRules($question->id, array($free_text), $question->rules, $return, 3);
						}
						break;
						
					case CS_CHOICE_MULTIRATING: // Multiple rating
						
						$rows = array();
						$grid_answers = array();
						
						foreach ($question->answers as $answer)
						{
							
							if ($answer->answer_type == 'x')
							{
								
								$rows[] = $answer;
							}
						}
						
						foreach ($rows as $row)
						{
							
							$column_id = $app->input->post->getInt('answer-' . $question->id . '-' . $row->id, 0);
							
							if ($column_id)
							{
								$answer = array();
								$answer['question_id'] = $question->id;
								$answer['answer_id'] = $row->id;
								$answer['column_id'] = $column_id;
								$answer['free_text'] = $app->input->post->getString('custom-text-' . $row->id, null);
								
								array_push ( $grid_answers, array($row->id => $column_id) );
								array_push($answers, $answer);
							}
						}
						
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, $grid_answers, $question->rules, $return, 2);
						}
						break;
					
					case CS_CHOICE_RANKING: // Ranking
						
						$rows = array();
						$grid_answers = array();
						$free_text = $app->input->post->getString('free-text-' . $question->id, null);
						
						foreach ($question->answers as $answer)
						{
							
							if ($answer->answer_type == 'x')
							{
								
								$rows[] = $answer;
							}
						}
						
						foreach ($rows as $row)
						{
							
							$column_id = $app->input->post->getInt('answer-' . $question->id . '-' . $row->id, 0);
							$answer = array();
							$answer['question_id'] = $question->id;
							$answer['answer_id'] = $row->id;
							$answer['column_id'] = $column_id;
							$answer['free_text'] = null;
							
							array_push ( $grid_answers, array($row->id => $column_id) );
							array_push($answers, $answer);
						}
						
						if(!empty($question->rules) && empty($return->finalize) && empty($return->page_id))
						{
							$this->validateRules($question->id, $grid_answers, $question->rules, $return, 2);
						}
						break;
				}
				
				if ($free_text)
				{
					$answer = array();
					$answer['question_id'] = $question->id;
					$answer['answer_id'] = 0;
					$answer['column_id'] = 0;
					$answer['free_text'] = $free_text;
					array_push($answers, $answer);
				}
			}
			
			$app->triggerEvent('onResponseBeforeSaveQuestions', array($context, $questions, $answers));

			if(!empty($answers))
			{
    			$query = $db->getQuery(true)
    				->delete('#__survey_response_details')
    				->where('response_id=' . $responseId);
    			
    			if ($pageId)
    			{
    				$query->where(
    						'question_id in (select id from #__survey_questions where survey_id=' . $surveyId . ' and page_number=' . $pageId . ')');
    			}
    			
    			$db->setQuery($query);
    			if ($db->execute())
    			{
    				$query = $db->getQuery(true)
    					->insert('#__survey_response_details')
    					->columns('response_id, question_id, answer_id, column_id, free_text');
    				
    				foreach ($answers as $answer)
    				{
    					if (empty($answer['free_text']))
    					{
    						$answer['free_text'] = 'null';
    					}
    					else
    					{
    						$answer['free_text'] = $db->q($answer['free_text']);
    					}
    					
    					$query->values(
    							$responseId . ',' . 
    							$answer['question_id'] . ',' . 
    							$answer['answer_id'] . ',' . 
    							$answer['column_id'] . ',' .
    							$answer['free_text']);
    				}
    				
    				$db->setQuery($query);
    				$db->execute();
    			}
			}
			
			if($pageId && $responseId && $this->getState('update_curr_page', true))
			{
				$query = $db->getQuery(true)
					->update('#__survey_responses')
					->set('curr_page = ' . $pageId)
					->where('id = ' . $responseId);
				$db->setQuery($query);
				$db->execute();
			}
			
			$app->triggerEvent('onResponseAfterSaveQuestions', array($context, $questions, $answers));
				
			return $return;
		}
		catch (Exception $e)
		{
			throw $e; // TODO: change it to throw with different message
		}
		
		return false;
	}
	
	// type: 1 = choice, 2 = grid, 3 = free text
	function validateRules($question_id, $answer_ids, $rules, &$return, $type)
	{
		foreach ($rules as $rawRule)
		{
			if($rawRule->question_id == $question_id && !empty($rawRule->rulecontent))
			{
				$rule = json_decode($rawRule->rulecontent);
				if($rule->name == 'unanswered' && ( (in_array($type, array(1, 3)) && empty($answer_ids[0]))  || empty($answer_ids) ) )
				{
					if($rule->finalize == '1')
					{
						$return->finalize = 1;
					}
					else
					{
						if(!empty($rule->page))
						{
							$return->page_id = intval($rule->page);
						}
					}
					break;
				}

				if($return->finalize > 0 || $return->page_id > 0) break;
				$selected = false;
				foreach($answer_ids as $answer_id)
				{
					switch ($rule->name)
					{
						case 'answered':
							
							if(!empty($answer_id))
							{
								if($rule->finalize == '1')
								{
									$return->finalize = 1;
								}
								else
								{
									if(!empty($rule->page))
									{
										$return->page_id = intval($rule->page);
									}
								}
							}
							break;
	
						case 'selected':

							if(
							( ($type == 1) && ($answer_id == $rule->answer_id) ) ||
							( ($type == 2) && (key($answer_id) == $rule->answer_id) && ($answer_id[key($answer_id)] == $rule->column_id) )
							)
							{
								if($rule->finalize == '1')
								{
									$return->finalize = 1;
								} 
								else if(!empty($rule->page))
								{
									$return->page_id = intval($rule->page);
								}
							}
	
							break;
	
						case 'unselected':
	
							if(
							( ($type == 1) && ($answer_id == $rule->answer_id) ) ||
							( ($type == 2) && (key($answer_id) == $rule->answer_id) && ($answer_id[key($answer_id)] == $rule->column_id) )
							)
							{
								$selected = true;
							}
	
							break;
					}
				}
	
				if($rule->name == 'unselected' && !$selected)
				{
					if($rule->finalize == '1')
					{
						$return->finalize = 1;
					} 
					else if(!empty($rule->page))
					{
						$return->page_id = intval($rule->page);
					}
				}
			}
		}
	}
	
	private function uploadFile($fieldName, $allowedExtensions = null)
	{
	    $app = JFactory::getApplication();
	    $params = $this->getState('params');
	    $allowedExtensions = !empty($allowedExtensions) ? $allowedExtensions : $params->get('allowed_uploadfile_types', 'jpg,png,gif,zip,gz,tar,doc,docx,xls,xlsx,ppt,pptx,txt');
	    $allowedSize = ((int)$params->get('max_attachment_size', 2048))*1024;
	    
	    if(!empty($allowedExtensions))
	    {
	        $tmp_file = $app->input->files->get($fieldName);
	        if(empty($tmp_file) || $tmp_file['error'] > 0)
	        {
	            return false;
	        }
	        else
	        {
	            $temp_file_path = $tmp_file['tmp_name'];
	            $temp_file_name = $tmp_file['name'];
	            $temp_file_ext = JFile::getExt($temp_file_name);
	    
	            if (!in_array(strtolower($temp_file_ext), explode(',', strtolower($allowedExtensions))))
	            {
	                throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_INVALID_FILE_TYPE') . '| RC=2', 403);
	            }
	            else if ($tmp_file['size'] > $allowedSize)
	            {
	                throw new Exception(JText::_('COM_COMMUNITYSURVEYS_ERROR_MAX_SIZE_FAILURE') . '| RC=3', 403);
	            }
	            else
	            {
	                $fileName = CjLibUtils::getRandomKey(25, 'abcdefghijklmnopqrstuvwxyz1234567890').'.'.$temp_file_ext;
	                if(JFile::upload($temp_file_path, CS_FILE_UPLOAD_DIR.'/'.$fileName))
	                {
	                    return $fileName;
	                }
	                else
	                {
	                    throw new Exception(JText::_('JERROR_AN_ERROR_HAS_OCCURRED') . '| RC=4', 500);
	                }
	            }
	        }
	    
	    }
	    
	    return false;
	}
	
	public function getPagination($pageId = null)
	{
		$surveyId = (int) $this->getState('filter.survey_id');
		$pageId = (int) $pageId ? $pageId : $this->getState('filter.page_id');
		
		return new SurveyPagination($surveyId, $pageId);
	}
	
	public function getForm($data = array(), $loadData = true)
	{
	    return parent::getForm($data, $loadData);
	}
	
	public function createSurveyKeys($surveyId, $count, $unlimited = false, $skipUserCheck = false, $userList = null){
	
		$user = $this->getUser();
		$db = JFactory::getDbo();
		
		$query = $db->getQuery(true)
			->select('id, catid, title, created_by, language')
			->from('#__survey_surveys')
			->where('id = '.$surveyId);
		
		$db->setQuery($query);
		$survey = $db->loadObject();
	
		if(!$skipUserCheck && (!$survey || ($survey->created_by != $user->id)) && !$user->authorise('core.manage', 'com_communitysurveys.survey.'.$survey->id))
		{
			return false;
		}
	
		$params = JComponentHelper::getParams('com_communitysurveys');
		$pointsApp = $params->get('points_component', 'none');
		$pointsPerCredit = (int)$params->get('points_per_credit', 0);
	
		if(!$unlimited && $pointsPerCredit > 0 && $pointsApp != 'none')
		{
			//Check user has enough credits
			$credits = $this->checkUserCredits($survey->created_by);
			if($credits == 0)
			{
				return false;
			}
			else
			{
				if($credits > 0 && $credits < $count)
				{
					$count = $credits;
				}
			}
		}
	
		$keys = array();
		$created = JFactory::getDate()->toSql();
		$query = $db->getQuery(true)
			->insert('#__survey_keys')
			->columns('key_name, survey_id, response_id, response_status, created, user_id');
	
		for ($i=0; $i < $count; $i++)
		{
			$user_id 			= 0;
			$key 				= CjLibUtils::getRandomKey(16);
			array_push($keys, $key);
			
			if(!empty($userList[$i]))
			{
				$invitee 		= $userList[$i];
				$user_id 		= (int) $invitee->id;
				$url 			= CommunitySurveysHelperRoute::getSurveyRoute($survey->id, $survey->catid, $survey->language).'&key='.$key;
				$invitee->url 	= SurveyHelper::buildSefSurveyUrl($url, true, -1);
			}
			
			$query->values($db->q($key).','.$surveyId.', 0, 0, '.$db->q($created).','.$user_id);
		}
		
		$db->setQuery($query);
		if($db->execute())
		{
			if(!$unlimited && $pointsPerCredit > 0 && $pointsApp != 'none')
			{
				$this->useCredits($survey->created_by, $count, JText::sprintf('COM_COMMUNITYSURVEYS_CREDITS_USED', $survey->title));
			}
		} 
	
		return $keys;
	}
	
	public function checkUserCredits($userId=0)
	{
		if(!$userId)
		{
			$user = JFactory::getUser ();
			$userId = $user->id;
		}
	
		$params = JComponentHelper::getParams('com_communitysurveys');
		$pointsPerCredit = (int)$params->get('points_per_credit', 0);
		if(!$pointsPerCredit) return -1;
	
		switch ($params->get('points_component', 'none'))
		{
			case 'cjforum':
				$api = JPATH_ROOT.'/components/com_cjforum/lib/api.php';
				if($api)
				{
					require_once $api;
					$profileApi = CjForumApi::getProfileApi();
					$profile = $profileApi->getUserProfile($userId);
					
					if(!empty($profile))
					{
						return floor($profile['points'] / $pointsPerCredit);
					}
				}
				break;
				
			case 'cjblog':
	
				$api = JPATH_ROOT.'/components/com_cjblog/api.php';
				if(file_exists($api))
				{
					include_once $api;
					$profile = CjBlogApi::get_user_profile($userId);
					
					if(!empty($profile))
					{
						return floor($profile['points'] / $pointsPerCredit);
					}
				}
				break;
					
			case 'aup':
	
				$api_AUP = JPATH_SITE.'/components/com_alphauserpoints/helper.php';
				if ( file_exists($api_AUP))
				{
					require_once ($api_AUP);
					$profile = AlphaUserPointsHelper::getUserInfo('', $userId);
					
					if(!empty($profile))
					{
						return floor($profile->points / $pointsPerCredit);
					}
				}
				break;
	
			case 'jomsocial':
	
				$db = JFactory::getDbo();
				$query = $db->getQuery(true)
					->select('points')
					->from('#__community_users')
					->where('userid = '.$userId);
				
				$db->setQuery($query);
				$points = (int)$db->loadResult();
	
				return floor($points / $pointsPerCredit);
	
			case 'easysocial':

				$db = JFactory::getDbo();
				$query = $db->getQuery(true)
					->select('sum(points)')
					->from('#__social_points_history')
					->where('user_id = '.$userId.' and state = 1');
				
				$db->setQuery($query);
				$points = (int)$db->loadResult();
	
				return floor($points / $pointsPerCredit);
		}
	
		return -1;
	}
	
	public function userCredits($userId, $credits, $info)
	{
		$api = new CjLibApi();
		$params = JComponentHelper::getParams('com_communitysurveys');
		$pointsComponent = $params->get('points_component', 'none');
		$pointsPerCredit = (int)$params->get('points_per_credit', 0);
		$points = -($credits * $pointsPerCredit);
		$function = '';
	
		switch ($pointsComponent)
		{
			case 'cjforum':
			case 'cjblog':
			case 'jomsocial':

				$function = 'com_communitysurveys.credits';
				break;
	
			case 'aup':
	
				$function = 'com_communitysurveys.credits';
				break;
					
			case 'easysocial':
	
				$api = JPATH_ADMINISTRATOR.'/components/com_easysocial/includes/foundry.php';
	
				if( file_exists($api) )
				{
					require_once $api;
					Foundry::points()->assignCustom( $userId , $points , $info );
				}
				// now return from here as the points are already added.
				return true;
	
			default:
				return false;
		}
		
		$title = JText::_('COM_COMMUNITYSURVEYS_POINTS_REDEEMED_FOR_SURVEYS_CREDITS');
		$options = array('function'=>$function, 'reference'=>'', 'info'=>$info, 'component'=>'com_communitysurveys', 'title'=>$title, 'points' => $points);
		$api->awardPoints($pointsComponent, $userId, $options);
	}

	public function finalize($survey, $responseId)
	{
		$db = JFactory::getDbo();
		$createdate = JFactory::getDate()->toSql();
	
		try
		{
			$query = $db->getQuery(true)
				->update('#__survey_responses')
				->set('completed = '.$db->q($createdate))
				->where('id = '.$responseId);
				
			$db->setQuery($query);
				
			if($db->execute())
			{
				$query = $db->getQuery(true)
					->update('#__survey_surveys')
					->set('responses = responses + 1')
					->where('id = '.$survey->id);
	
				$db->setQuery($query);
				$db->execute();
	
				// 				$query = $db->getQuery(true)
				// 					->update('#__survey_users')
				// 					->set('responses = (select count(*) from #__survey_responses where created_by = '.$user->id.' and completed > created)')
				// 					->where('id = '.$user->id);
	
				// 				$db->setQuery($query);
				// 				$db->execute();
	
				$query = $db->getQuery(true)
					->update('#__survey_keys')
					->set('response_id = '. $responseId)
					->set('response_status = 1')
					->where('key_name = (select survey_key from #__survey_responses where id ='.$responseId.') and survey_id = '.$survey->id);
	
				$db->setQuery($query);
				$db->execute();
	
				// Generate pdf response for usage
				if($survey->params->get('attach_pdf_report', false))
				{
					@$this->createPdfResponse($survey->id, $responseId);
				}
	
				$survey->response_id = $responseId;
				JPluginHelper::importPlugin($this->events_map['save']);
				JFactory::getApplication()->triggerEvent($this->event_after_save, array('com_communitysurveys.response', $survey, true));
	
				return true;
			}
		}
		catch (Exception $e)
		{
			throw $e;
			// TODO: handle exception
		}
	
		return false;
	}
	
	public function getResponse($pk = 0)
	{
		$db = JFactory::getDbo();
		$user = $this->getUser();
		$pk = $pk ? $pk : (int) $this->getState('filter.response_id');
		
		if($user->guest || !$pk)
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR').'| RC=1', 403);
		}
		
		$query = $db->getQuery(true)
			->select('survey_id')
			->from('#__survey_responses')
			->where('id = '.$pk.' and created_by = '.$user->id);
		
		$db->setQuery($query);
		$surveyId = (int) $db->loadResult();
		
		if(!$surveyId)
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR').'| RC=2', 403);
		}
		
		JLoader::import('survey', JPATH_ROOT.'/components/com_communitysurveys/models');
		$surveyModel	= JModelLegacy::getInstance( 'survey', 'CommunitySurveysModel' );
		
		$surveyModel->getState(); // access the state first so that it can be modified
		$surveyModel->setState('filter.survey_id', $surveyId);
		$survey 		= $surveyModel->getItem($surveyId);
		
		if(!$survey)
		{
			throw new Exception(JText::_('JERROR_ALERTNOAUTHOR').'| RC=3', 403);
		}
		
		JLoader::import('questions', JPATH_ROOT.'/components/com_communitysurveys/models');
		$questionsModel = JModelLegacy::getInstance( 'questions', 'CommunitySurveysModel' );
		
		$questionsModel->getState();
		$questionsModel->setState('filter.survey_id', $surveyId);
		$questionsModel->setState('filter.category_id', null);
		$questionsModel->setState('filter.page_id', 0);
		$questionsModel->setState('filter.question_id', null);
		$questionsModel->setState('filter.unique', false);
		$questionsModel->setState('filter.response_id', $pk);
		$questionsModel->setState('list.ordering', 'p.sort_order asc, a.sort_order');
		$questionsModel->setState('list.direction', 'asc');
		$questionsModel->setState('list.limit', 0);
		$questions = $questionsModel->getItems();
		
		$return 			= new stdClass();
		$return->item 		= $survey;
		$return->questions 	= $questions;
		
		return $return;
	}
}