<?php
/**
 * Prohibit direct script loading
 */
defined('ABSPATH') || die('No direct script access allowed!');

/**
 * Class WpmsRedirectManager
 */
class WpmsRedirectManager
{
    /**
     * Redirect option
     *
     * @var WpmsRedirectOption
     */
    protected $redirect_option;
    /**
     * Redirect format
     *
     * @var string
     */
    protected $redirect_format;
    /**
     * Exporters
     *
     * @var array|null
     */
    protected $exporters;

    /**
     * Default export
     *
     * @return array
     */
    public static function defaultExport()
    {
        $exporters[] = new WpmsRedirectOptionExporter();
        return $exporters;
    }

    /**
     * WpmsRedirectManager constructor.
     *
     * @param string                  $redirect_format Redirect format
     * @param null                    $exporters       Exporters
     * @param WpmsRedirectOption|null $option          Option
     */
    public function __construct($redirect_format = 'plain', $exporters = null, WpmsRedirectOption $option = null)
    {
        if (null === $option) {
            $option = new WpmsRedirectOption();
        }
        $this->redirect_option = $option;
        $this->redirect_format = $redirect_format;
        $this->exporters       = ($exporters) ? $exporters : self::defaultExport();
    }

    /**
     * Export the redirects to the specified sources.
     *
     * @return void
     */
    public function exportRedirects()
    {
        $redirects = $this->redirect_option->getAll();

        foreach ($this->exporters as $exporter) {
            $exporter->export($redirects);
        }
    }

    /**
     * Create a new redirect
     *
     * @param WpmsRedirect $redirect WpmsRedirect class
     *
     * @return boolean
     */
    public function createRedirect(WpmsRedirect $redirect)
    {
        if ($this->redirect_option->add($redirect)) {
            $this->saveRedirects();

            return true;
        }

        return false;
    }

    /**
     * Save the redirect
     *
     * @param WpmsRedirect $current_redirect Current redirect
     * @param WpmsRedirect $redirect         New redirect
     *
     * @return boolean
     */
    public function updateRedirect(WpmsRedirect $current_redirect, WpmsRedirect $redirect)
    {
        if ($this->redirect_option->update($current_redirect, $redirect)) {
            $this->saveRedirects();

            return true;
        }

        return false;
    }

    /**
     * Delete the redirects
     *
     * @param array $delete_redirects Redirects URl list
     *
     * @return boolean
     */
    public function deleteRedirects($delete_redirects)
    {
        $deleted = false;
        foreach ($delete_redirects as $delete_redirect) {
            if ($this->redirect_option->delete($delete_redirect)) {
                $deleted = true;
            }
        }

        if ($deleted === true) {
            $this->saveRedirects();
        }

        return $deleted;
    }

    /**
     * Return redirect
     *
     * @param string $origin URL
     *
     * @return boolean|mixed
     */
    public function getRedirect($origin)
    {
        return $this->redirect_option->get($origin);
    }

    /**
     * Save the redirect option
     *
     * @return void
     */
    public function saveRedirects()
    {
        // Update option
        $this->redirect_option->save();
        // Save redirect file.
        $this->exportRedirects();
    }
}
