<?php

/**
 * @package         Google Structured Data
 * @version         4.3.1 Pro
 * 
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2018 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
*/

defined('_JEXEC') or die('Restricted access');

class GSDModelItems extends JModelList
{
    /**
     * Constructor.
     *
     * @param    array    An optional associative array of configuration settings.
     *
     * @see        JController
     */
    public function __construct($config = array())
    {
        if (empty($config['filter_fields']))
        {
            $config['filter_fields'] = array(
                'id', 'a.id',
                'state', 'a.state',
                'title', 'a.title',
                'created', 'a.created',
                'search',
                'ordering', 'a.ordering',
                'plugin', 'a.plugin',
                'contenttype', 'a.contenttype',
                'targetpages'
            );
        }

        parent::__construct($config);
    }
   
    /**
     * Method to build an SQL query to load the list data.
     *
     * @return      string  An SQL query
     */
    protected function getListQuery()
    {
        // Create a new query object.
        $db = JFactory::getDBO();
        $query = $db->getQuery(true);

        // Select some fields from the item table
        $query
            ->select('a.*')
            ->from('#__gsd a');

        // Filter by search
        $search = $this->getState('filter.search');
        if (!empty($search))
        {
            if (stripos($search, 'id:') === 0)
            {
                $query->where('a.id = ' . (int) substr($search, 3));
            }
            else
            {
                $query->where('a.title LIKE ' . $db->quote('%' . trim($search) . '%') . '');
            }
        }  

        // Filter State
        $state = $this->getState('filter.state');
        if (is_numeric($state))
        {
            $query->where('a.state = ' . (int) $state);
        }

        if (strpos($state, ',') !== false)
        {
            $query->where('a.state IN (' . $state . ')');
        }

        if ($state == '')
        {
            $query->where('a.state IN (0,1,2)');
        }

        // Filter Content Type
        if ($thing = $this->getState('filter.contenttype'))
        {
            $query->where('a.contenttype = ' . $db->q($thing));
        }

        // Filter Plugin
        if ($plugin = $this->getState('filter.plugin'))
        {
            $query->where('a.plugin = ' . $db->q($plugin));
        }

        // Filter Publishing Rules
        if ($targetpages = $this->getState('filter.targetpages'))
        {
            $operator = ($targetpages == 'specific') ? 'LIKE' : 'NOT LIKE';
            $query->where($db->quoteName('a.params') . $operator . $db->q('%assignment_state":"1"%'));
        }

        // Add the list ordering clause.
        $orderCol  = $this->state->get('list.ordering', 'a.id');
        $orderDirn = $this->state->get('list.direction', 'desc');

        $query->order($db->escape($orderCol . ' ' . $orderDirn));

        return $query;
    }

    /**
     *  [getItems description]
     *
     *  @return  object
     */
    public function getItems()
    {
        $items = parent::getItems();

        foreach ($items as $key => &$item)
        {
            $item = (object) array_merge((array) $item, (array) json_decode($item->params));
        }

        return $items;
    }
}