<?php

/**
 * @package         Google Structured Data
 * @version         4.3.1 Pro
 *
 * @author          Tassos Marinos <info@tassos.gr>
 * @link            http://www.tassos.gr
 * @copyright       Copyright © 2018 Tassos Marinos All Rights Reserved
 * @license         GNU GPLv3 <http://www.gnu.org/licenses/gpl.html> or later
 */

defined('_JEXEC') or die('Restricted access');

use GSD\Helper;
use GSD\PluginBaseEvent;

/**
 *  Event Booking Google Structured Data Plugin
 */
class plgGSDEventBooking extends PluginBaseEvent
{
	/**
	 *  Get article's data
	 *
	 *  @return  array
	 */
	public function viewEvent()
	{	
		// Make sure we have a valid ID
		if (!$id = $this->getThingID())
		{
			return;
		}

		// Load current item via model
		$item = EventbookingHelperDatabase::getEvent($id);

		if (!is_object($item))
		{
			return;
		}

		// EventBooking doesn't save dates into the database in UTC format.
		$publish_up   = Helper::date($item->publish_up, false);
		$publish_down = Helper::date($item->publish_down, false);
		$startDate    = Helper::date($item->event_date, false);
		$endDate      = Helper::date($item->event_end_date, false);
		$regDate      = !empty($item->registration_start_date) ? Helper::date($item->registration_start_date, false) : $startDate;

		// Array data
		$payload = [
			'id'   		  		  => $item->id,
			'alias'       		  => $item->alias,
			'headline'    		  => $item->title,
			'description' 		  => empty($item->short_description) ? $item->description : $item->short_description,
			'introtext'   		  => $item->short_description,
			'fulltext'   	      => $item->description,
			'image'       		  => $item->image,
            'imagetext'	   		  => Helper::getFirstImageFromString($item->short_description . $item->description),
			'startDate'	  		  => $startDate,
			'endDate'	  		  => $endDate, 
			'created_by'  		  => $item->created_by,
			'publish_up'  		  => $publish_up,
			'publish_down'        => $publish_down,
        	'metakey'	  		  => $item->meta_keywords,
			'metadesc'	  		  => $item->meta_description,
			'offerStartDate' 	  => $regDate,
			'offerPrice'    	  => $item->individual_price,
			'offerCurrency'       => $item->currency_code,
			'offerInventoryLevel' => $item->event_capacity,
			'performerName'		  => Helper::getSiteName()
		];

		if ($location = EventbookingHelperDatabase::getLocation($item->location_id))
		{
			$payload['locationName'] = $location->name;
			$payload['locationAddress'] = $location->address;
		}

		return $payload;
	}

	/**
	 * Listening to the onAfterRender Joomla event
	 *
	 * @return void
	 */
	public function onAfterRender()
	{
		$this->removeMicrodata();
	}

	/**
	 * Helper method to remove the extra and incomplete Event microdata generated by the Event Booking extension
	 *
	 * @return void
	 */
	private function removeMicrodata()
	{
		if ((bool) !$this->params->get('removemicrodata', true))
		{
			return;
		}

        if (!$this->passContext() || $this->getView() != 'event')
        {
            return;
		}
		
		// Get document buffer
		$body = $this->app->getBody();

		// Simple check to decide whether the plugin should procceed or not.
		if (\JString::strpos($body, 'schema.org/Event') === false)
		{
			return;
		}

		// Replacement patterns
		$patterns = [
			'/itemscope itemtype=(\"?)http(s?):\/\/schema.org\/(Event|AggregateOffer)(\"?)/',
			'/<meta itemprop="(startDate|endDate|url)"[^>]+>/',
			'/itemprop="(lowPrice|offers)"/'
		];

		$body = preg_replace_callback($patterns, function($match) use (&$result)
		{
			return '';
		}, $body);

		$this->app->setBody($body);
	}
}
