<?php
/**
 * Manages interactions with the admin.
 *
 * @package Tasty_Recipes
 */

namespace Tasty_Recipes;

use Tasty_Recipes;
use Tasty_Recipes\Converters\EasyRecipe;

/**
 * Manages interactions with the admin.
 */
class Admin {

	/**
	 * Capability required to manage settings.
	 *
	 * @var string
	 */
	const CAPABILITY = 'manage_options';

	/**
	 * Plugin name for EDD registration.
	 *
	 * @var string
	 */
	const ITEM_AUTHOR = 'Tasty Recipes';

	/**
	 * Plugin author for EDD registration.
	 *
	 * @var string
	 */
	const ITEM_NAME = 'Tasty Recipes';

	/**
	 * Cache key used to store license check data.
	 *
	 * @var string
	 */
	const LICENSE_CHECK_CACHE_KEY = 'tasty-recipes-license-check';

	/**
	 * Key used for nonce authentication.
	 *
	 * @var string
	 */
	const NONCE_KEY = 'tasty-recipes-settings';

	/**
	 * URL to the plugin store.
	 *
	 * @var string
	 */
	const STORE_URL = 'https://www.wptasty.com';

	/**
	 * Parent page for the settings page.
	 *
	 * @var string
	 */
	const PAGE_BASE = 'options-general.php';

	/**
	 * Slug for the settings page.
	 *
	 * @var string
	 */
	const PAGE_SLUG = 'tasty-recipes';

	/**
	 * Group used for authentication settings.
	 *
	 * @var string
	 */
	const SETTINGS_GROUP_LICENSE = 'tasty-recipes-settings-license';

	/**
	 * Section used for authentication settings.
	 *
	 * @var string
	 */
	const SETTINGS_SECTION_LICENSE = 'tasty-recipes-license';

	/**
	 * Group used for card settings.
	 *
	 * @var string
	 */
	const SETTINGS_GROUP_CARD = 'tasty-recipes-settings-card';

	/**
	 * Section used for card settings.
	 *
	 * @var string
	 */
	const SETTINGS_SECTION_CARD_DESIGN = 'tasty-recipes';

	/**
	 * Registers the updater object.
	 */
	public static function action_admin_init() {
		new Updater(
			self::STORE_URL,
			TASTY_RECIPES_PLUGIN_FILE,
			array(
				'version'   => TASTY_RECIPES_PLUGIN_VERSION,
				'license'   => get_option( Tasty_Recipes::LICENSE_KEY_OPTION ),
				'item_name' => self::ITEM_NAME,
				'author'    => self::ITEM_AUTHOR,
			)
		);
		if ( get_option( Tasty_Recipes::PLUGIN_ACTIVATION_OPTION, false ) ) {
			delete_option( Tasty_Recipes::PLUGIN_ACTIVATION_OPTION );
			if ( ! isset( $_GET['activate-multi'] ) ) {
				wp_safe_redirect( add_query_arg( 'tab', 'about', menu_page_url( self::PAGE_SLUG, false ) ) );
				exit;
			}
		}
	}

	/**
	 * Includes PHP and plugin versions in the user agent for update checks.
	 *
	 * @param array  $r   An array of HTTP request arguments.
	 * @param string $url The request URL.
	 * @return array
	 */
	public static function filter_http_request_args( $r, $url ) {
		if ( self::STORE_URL !== $url
			|| 'POST' !== $r['method']
			|| empty( $r['body'] )
			|| ! is_array( $r['body'] )
			|| empty( $r['body']['item_name'] )
			|| self::ITEM_NAME !== $r['body']['item_name'] ) {
			return $r;
		}

		$r['user-agent'] = rtrim( $r['user-agent'], ';' )
			. '; PHP/' . PHP_VERSION . '; '
			. self::ITEM_NAME . '/' . TASTY_RECIPES_PLUGIN_VERSION;
		return $r;
	}

	/**
	 * Registers used settings.
	 */
	public static function action_admin_menu() {
		register_setting(
			self::SETTINGS_GROUP_LICENSE,
			Tasty_Recipes::LICENSE_KEY_OPTION,
			array(
				'sanitize_callback' => 'sanitize_text_field',
			)
		);
		register_setting(
			self::SETTINGS_GROUP_CARD,
			Tasty_Recipes::TEMPLATE_OPTION,
			array(
				'sanitize_callback' => 'sanitize_title',
			)
		);
		register_setting(
			self::SETTINGS_GROUP_CARD,
			Tasty_Recipes::QUICK_LINKS_OPTION,
			array(
				'sanitize_callback' => 'sanitize_title',
			)
		);
		register_setting(
			self::SETTINGS_GROUP_CARD,
			Tasty_Recipes::DEFAULT_AUTHOR_LINK_OPTION,
			array(
				'sanitize_callback' => 'esc_url_raw',
			)
		);
		register_setting(
			self::SETTINGS_GROUP_CARD,
			Tasty_Recipes::INSTAGRAM_HANDLE_OPTION,
			array(
				'sanitize_callback' => 'sanitize_text_field',
			)
		);
		register_setting(
			self::SETTINGS_GROUP_CARD,
			Tasty_Recipes::INSTAGRAM_HASHTAG_OPTION,
			array(
				'sanitize_callback' => 'sanitize_text_field',
			)
		);
		register_setting(
			self::SETTINGS_GROUP_CARD,
			Tasty_Recipes::SHAREASALE_OPTION,
			array(
				'sanitize_callback' => 'sanitize_text_field',
			)
		);

		$page_title = __( 'Tasty Recipes', 'tasty-recipes' );
		add_submenu_page( self::PAGE_BASE, $page_title, $page_title, self::CAPABILITY, self::PAGE_SLUG, array( __CLASS__, 'handle_settings_page' ) );
	}

	/**
	 * Renders the admin notice nag when license key isn't set or invalid.
	 */
	public static function action_admin_notices_license_key() {

		$screen = get_current_screen();
		if ( $screen && 'post' === $screen->base ) {
			return;
		}

		$license_check = self::get_license_check();

		if ( ! empty( $license_check ) && 'valid' !== $license_check->license ) :
			?>
	<div class="error" style="display:block !important;">
		<form method="post" action="options.php">
			<p>
				<strong><?php esc_html_e( 'To enable updates and support for Tasty Recipes', 'tasty-recipes' ); ?></strong>, <label style="vertical-align: baseline;" for="<?php echo esc_attr( Tasty_Recipes::LICENSE_KEY_OPTION ); ?>"><?php esc_html_e( 'enter a valid license key', 'tasty-recipes' ); ?></label>
				<input type="text" style="margin-left: 5px; margin-right: 5px; " class="code regular-text" id="<?php echo esc_attr( Tasty_Recipes::LICENSE_KEY_OPTION ); ?>" name="<?php echo esc_attr( Tasty_Recipes::LICENSE_KEY_OPTION ); ?>" value="<?php echo esc_attr( get_option( Tasty_Recipes::LICENSE_KEY_OPTION ) ); ?>" />
				<input type="submit" value="<?php _e( 'Save license key', 'tasty-recipes' ); ?>" class="button-primary" />
			</p>
			<p>
				<strong><?php esc_html_e( "Think you've reached this message in error?", 'tasty-recipes' ); ?></strong> <a href="http://support.wptasty.com" target="_blank"><?php esc_html_e( 'Submit a support ticket', 'tasty-recipes' ); ?></a>, <?php esc_html_e( "and we'll do our best to help out.", 'tasty-recipes' ); ?>
			</p>
			<div style="display:none;"><pre><?php echo json_encode( $license_check ); ?></pre></div>
			<?php
			settings_fields( self::SETTINGS_GROUP_LICENSE );
			do_settings_sections( self::SETTINGS_SECTION_LICENSE );
			?>
		</form>

	</div>
			<?php
		endif;

	}

	/**
	 * Activates the license when the option is updated.
	 */
	public static function action_update_option_register_license() {
		$api_params = array(
			'edd_action' => 'activate_license',
			'license'    => get_option( Tasty_Recipes::LICENSE_KEY_OPTION ),
			'item_name'  => self::ITEM_NAME, // the name of our product in EDD.
			'url'        => home_url(),
		);
		wp_remote_post(
			self::STORE_URL,
			array(
				'timeout' => 15,
				'body'    => $api_params,
			)
		);
	}

	/**
	 * Clears the license and version check cache when license key is updated.
	 */
	public static function action_update_option_clear_transient() {
		delete_transient( self::LICENSE_CHECK_CACHE_KEY );
		$cache_key = md5( 'edd_plugin_' . sanitize_key( plugin_basename( TASTY_RECIPES_PLUGIN_FILE ) ) . '_version_info' );
		delete_site_transient( $cache_key );
		$cache_key = 'edd_api_request_' . substr( md5( serialize( basename( TASTY_RECIPES_PLUGIN_FILE, '.php' ) ) ), 0, 15 );
		delete_site_transient( $cache_key );
		delete_site_transient( 'update_plugins' );
	}

	/**
	 * Adds a 'Remove license' link to plugin settings.
	 *
	 * @param array $links Existing plugin action links.
	 * @return array
	 */
	public static function filter_plugin_action_links( $links ) {
		if ( get_option( Tasty_Recipes::LICENSE_KEY_OPTION ) ) {
			$links['remove_key'] = '<a href="' . add_query_arg(
				array(
					'action' => 'tasty_recipes_remove_license_key',
					'nonce'  => wp_create_nonce( 'tasty_recipes_remove_license_key' ),
				),
				admin_url( 'admin-ajax.php' )
			) . '">' . esc_html__( 'Remove license', 'tasty-recipes' ) . '</a>';
		}
		return $links;
	}

	/**
	 * Handles a request to remove the license key.
	 */
	public static function handle_wp_ajax_remove_license_key() {

		if ( ! current_user_can( 'manage_options' ) || ! wp_verify_nonce( $_GET['nonce'], 'tasty_recipes_remove_license_key' ) ) {
			wp_safe_redirect( admin_url( 'plugins.php' ) );
			exit;
		}

		$api_params = array(
			'edd_action' => 'deactivate_license',
			'license'    => get_option( Tasty_Recipes::LICENSE_KEY_OPTION ),
			'item_name'  => self::ITEM_NAME,
			'url'        => home_url(),
		);
		$response   = wp_remote_post(
			self::STORE_URL,
			array(
				'timeout' => 15,
				'body'    => $api_params,
			)
		);
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			if ( is_wp_error( $response ) ) {
				$message = $response->get_error_message();
			} else {
				$message = __( 'An error occurred, please try again.', 'tasty-recipes' );
			}
			wp_die( $message );
		}

		self::action_update_option_clear_transient();
		delete_option( Tasty_Recipes::LICENSE_KEY_OPTION );
		wp_safe_redirect( admin_url( 'plugins.php' ) );
		exit;
	}

	/**
	 * Handles an AJAX request to get the number of matching recipes to convert.
	 */
	public static function handle_wp_ajax_get_count() {
		if ( ! current_user_can( self::CAPABILITY ) || ! wp_verify_nonce( $_GET['nonce'], self::NONCE_KEY ) ) {
			wp_send_json_error(
				array(
					'message' => __( "Sorry, you don't have permission to do this.", 'tasty-recipes' ),
				)
			);
		}

		$class      = false;
		$converters = Tasty_Recipes::get_converters();
		if ( isset( $converters[ $_GET['type'] ] ) ) {
			$class = $converters[ $_GET['type'] ]['class'];
		} else {
			wp_send_json_error(
				array(
					'message' => __( "Couldn't count recipes. Please contact support for assistance.", 'tasty-recipes' ),
				)
			);
		}

		wp_send_json_success(
			array(
				'count' => (int) $class::get_count(),
			)
		);
	}

	/**
	 * Handles an AJAX request to batch convert some recipes.
	 */
	public static function handle_wp_ajax_convert() {
		if ( ! current_user_can( self::CAPABILITY ) || ! wp_verify_nonce( $_GET['nonce'], self::NONCE_KEY ) ) {
			wp_send_json_error(
				array(
					'message' => __( "Sorry, you don't have permission to do this.", 'tasty-recipes' ),
				)
			);
		}

		$class      = false;
		$converters = Tasty_Recipes::get_converters();
		if ( isset( $converters[ $_GET['type'] ] ) ) {
			$class = $converters[ $_GET['type'] ]['class'];
		} else {
			wp_send_json_error(
				array(
					'message' => __( "Couldn't convert recipes. Please contact support for assistance.", 'tasty-recipes' ),
				)
			);
		}

		$post_ids  = $class::get_post_ids();
		$converted = 0;
		foreach ( $post_ids as $post_id ) {
			$type = 'shortcode';
			if ( function_exists( 'has_blocks' )
				&& has_blocks( $post_id ) ) {
				$type = 'block';
			}
			$recipe = $class::convert_post( $post_id, $type );
			if ( $recipe ) {
				$converted++;
			} else {
				wp_send_json_error(
					array(
						// translators: Post id that couldn't be translated.
						'message' => sprintf( __( "Couldn't convert recipe in post %d. Please contact support for assistance.", 'tasty-recipes' ), $post_id ),
					)
				);
			}
		}

		$after_count = $class::get_count();
		wp_send_json_success(
			array(
				'converted' => (int) $converted,
				'count'     => (int) $after_count,
			)
		);
	}

	/**
	 * Renders the Tasty Recipes settings page.
	 */
	public static function handle_settings_page() {
		$select_options = array(
			'name'        => Tasty_Recipes::TEMPLATE_OPTION,
			'options'     => array(
				''               => 'Default',
				'bold'           => 'Bold',
				'fresh'          => 'Fresh',
				'simple'         => 'Simple',
				'modern-compact' => 'Modern Compact',
				'elegant'        => 'Elegant',
			),
			// translators: Link to the template documentation.
			'description' => sprintf( __( 'For more details, <a href="%s" target="_blank">read the WP Tasty templates documentation</a>.', 'tasty-recipes' ), 'https://support.wptasty.com/tasty-recipes/what-do-the-recipe-card-templates-look-like' ),
		);
		if ( file_exists( get_stylesheet_directory() . '/tasty-recipes.php' ) ) {
			$select_options['options']     = array(
				'custom' => 'Custom',
			);
			$select_options['disabled']    = true;
			$select_options['description'] = __( 'To enable template selection, please remove the custom tasty-recipes.php file in your theme.', 'tasty-recipes' );
		}
		add_settings_section(
			self::SETTINGS_SECTION_CARD_DESIGN,
			__( 'Card Design', 'tasty-recipes' ),
			false,
			self::PAGE_SLUG
		);
		add_settings_field(
			Tasty_Recipes::TEMPLATE_OPTION,
			__( 'Template', 'tasty-recipes' ),
			array( __CLASS__, 'render_select_field' ),
			self::PAGE_SLUG,
			self::SETTINGS_SECTION_CARD_DESIGN,
			$select_options
		);
		add_settings_field(
			Tasty_Recipes::QUICK_LINKS_OPTION,
			__( 'Quick Links', 'tasty-recipes' ),
			array( __CLASS__, 'render_select_field' ),
			self::PAGE_SLUG,
			self::SETTINGS_SECTION_CARD_DESIGN,
			array(
				'name'        => Tasty_Recipes::QUICK_LINKS_OPTION,
				'description' => __( 'Enable or disable quick links below the post title.', 'tasty-recipes' ),
				'options'     => array(
					''      => __( 'Disabled', 'tasty-recipes' ),
					'both'  => __( 'Show \'Jump to Recipe\' and \'Print Recipe\' links', 'tasty-recipes' ),
					'jump'  => __( 'Only show \'Jump to Recipe\' link', 'tasty-recipes' ),
					'print' => __( 'Only show \'Print Recipe\' link', 'tasty-recipes' ),
				),
			)
		);
		add_settings_field(
			Tasty_Recipes::DEFAULT_AUTHOR_LINK_OPTION,
			__( 'Default Author Link', 'tasty-recipes' ),
			array( __CLASS__, 'render_input_field' ),
			self::PAGE_SLUG,
			self::SETTINGS_SECTION_CARD_DESIGN,
			array(
				'name' => Tasty_Recipes::DEFAULT_AUTHOR_LINK_OPTION,
			)
		);
		add_settings_field(
			Tasty_Recipes::INSTAGRAM_HANDLE_OPTION,
			__( 'Instagram Handle', 'tasty-recipes' ),
			array( __CLASS__, 'render_input_field' ),
			self::PAGE_SLUG,
			self::SETTINGS_SECTION_CARD_DESIGN,
			array(
				'name' => Tasty_Recipes::INSTAGRAM_HANDLE_OPTION,
			)
		);
		add_settings_field(
			Tasty_Recipes::INSTAGRAM_HASHTAG_OPTION,
			__( 'Instagram Hashtag', 'tasty-recipes' ),
			array( __CLASS__, 'render_input_field' ),
			self::PAGE_SLUG,
			self::SETTINGS_SECTION_CARD_DESIGN,
			array(
				'name' => Tasty_Recipes::INSTAGRAM_HASHTAG_OPTION,
			)
		);
		add_settings_field(
			Tasty_Recipes::SHAREASALE_OPTION,
			__( 'ShareASale Affiliate ID', 'tasty-recipes' ),
			array( __CLASS__, 'render_input_field' ),
			self::PAGE_SLUG,
			self::SETTINGS_SECTION_CARD_DESIGN,
			array(
				'name'        => Tasty_Recipes::SHAREASALE_OPTION,
				// translators: Various links.
				'description' => sprintf( __( '<a href="%1$s" target="_blank">Apply for the affiliate program</a>, or <a href="%2$s" target="_blank">find your affiliate ID</a>.', 'tasty-recipes' ), 'https://www.wptasty.com/affiliates', 'https://www.wptasty.com/affiliate-id' ),
			)
		);
		echo Tasty_Recipes::get_template_part( 'settings' );
	}

	/**
	 * Render an input field.
	 *
	 * @param array $args Configuration arguments used by the input field.
	 */
	public static function render_input_field( $args ) {
		$defaults = array(
			'type' => 'text',
			'name' => '',
		);
		$args     = array_merge( $defaults, $args );
		if ( empty( $args['name'] ) ) {
			return;
		}
		$value = get_option( $args['name'] );
		?>
		<input type="<?php echo esc_attr( $args['type'] ); ?>"
			name="<?php echo esc_attr( $args['name'] ); ?>"
			value="<?php echo esc_attr( $value ); ?>"
			class="regular-text"
		/>
		<?php if ( ! empty( $args['description'] ) ) : ?>
			<p class="description"><?php echo wp_kses_post( $args['description'] ); ?></p>
			<?php
		endif;
	}

	/**
	 * Render a select field.
	 *
	 * @param array $args Configuration arguments used by the select field.
	 */
	public static function render_select_field( $args ) {
		$defaults = array(
			'name'    => '',
			'options' => array(),
		);
		$args     = array_merge( $defaults, $args );
		if ( empty( $args['name'] ) || empty( $args['options'] ) ) {
			return;
		}
		$value = get_option( $args['name'] );
		?>
		<select name="<?php echo esc_attr( $args['name'] ); ?>"
		<?php
		if ( ! empty( $args['disabled'] ) ) {
			echo 'disabled';
		}
		?>
		>
			<?php foreach ( $args['options'] as $key => $label ) : ?>
				<option value="<?php echo esc_attr( $key ); ?>" <?php selected( $key, $value ); ?>><?php echo esc_html( $label ); ?></option>
			<?php endforeach; ?>
		</select>
		<?php if ( ! empty( $args['description'] ) ) : ?>
			<p class="description"><?php echo wp_kses_post( $args['description'] ); ?></p>
			<?php
		endif;
	}

	/**
	 * Gets the license check object.
	 *
	 * @return object
	 */
	public static function get_license_check() {
		if ( ! get_option( Tasty_Recipes::LICENSE_KEY_OPTION ) ) {
			return false;
		}
		$license_check = get_transient( self::LICENSE_CHECK_CACHE_KEY );
		if ( false === $license_check ) {
			$api_params = array(
				'edd_action' => 'check_license',
				'license'    => get_option( Tasty_Recipes::LICENSE_KEY_OPTION ),
				'item_id'    => false,
				'item_name'  => self::ITEM_NAME,
				'author'     => self::ITEM_AUTHOR,
				'url'        => home_url(),
			);

			$license_check = wp_remote_post(
				self::STORE_URL,
				array(
					'timeout' => 15,
					'body'    => $api_params,
				)
			);

			if ( ! is_wp_error( $license_check ) ) {
				$license_check = json_decode( wp_remote_retrieve_body( $license_check ) );
			}
			set_transient( self::LICENSE_CHECK_CACHE_KEY, $license_check, 60 * HOUR_IN_SECONDS );
		}
		return $license_check;
	}

}
