<?php
/**
 * Generates distribution metadata.
 *
 * @package Tasty_Recipes
 */

namespace Tasty_Recipes;

use Tasty_Recipes;
use Tasty_Recipes\Content_Model;
use Tasty_Recipes\Utils;
use Tasty_Recipes\Objects\Recipe;

/**
 * Generates distribution metadata.
 */
class Distribution_Metadata {

	/**
	 * Regex used to match list items.
	 *
	 * @var string
	 */
	const LI_MATCH_REGEX = '#<li[^\>]*>(.*)<\/li>#U';

	/**
	 * Image sizes used in JSON+LD.
	 *
	 * @var array
	 */
	static $json_ld_image_sizes = array(
		'1x1'  => array( 225, 225 ),
		'4x3'  => array( 260, 195 ),
		'16x9' => array( 320, 180 ),
	);

	/**
	 * Renders JSON+LD markup on singular views with recipes.
	 */
	public static function action_wp_head_google_schema() {
		if ( ! is_singular() ) {
			return;
		}

		// Use Yoast SEO v11 or greater when present.
		if ( interface_exists( 'WPSEO_Graph_Piece' ) ) {
			return;
		}

		$recipes = Tasty_Recipes::get_recipes_for_post(
			get_queried_object()->ID,
			array(
				'disable-json-ld' => false,
			)
		);
		if ( empty( $recipes ) ) {
			return;
		}

		$schemas = array();
		foreach ( $recipes as $recipe ) {
			$schemas[] = self::get_enriched_google_schema_for_recipe( $recipe, get_queried_object() );
		}

		foreach ( $schemas as $schema ) {
			?>
<script type="application/ld+json">
			<?php echo json_encode( $schema, JSON_PRETTY_PRINT ) . PHP_EOL; ?>
</script>
			<?php
		}
	}

	/**
	 * Adds noindex and rel=canonical to print template rendering.
	 */
	public static function action_wp_head_noindex() {
		if ( ! tasty_recipes_is_print() ) {
			return;
		}
		echo '<meta name="robots" content="noindex">' . PHP_EOL;
		// WordPress will add rel=canonical by default
		// but Yoast SEO might remove it.
		if ( ! has_action( 'wp_head', 'rel_canonical' ) ) {
			echo '<link rel="canonical" href="' . get_permalink( get_queried_object_id() ) . '" />' . PHP_EOL;
		}
	}

	/**
	 * Incorporate Tasty Recipes into Yoast SEO when present.
	 *
	 * @param array  $pieces  Existing Yoast SEO graph pieces.
	 * @param object $context Yoast SEO context instance.
	 * @return array
	 */
	public static function filter_wpseo_schema_graph_pieces( $pieces, $context ) {
		$pieces[] = new \Tasty_Recipes\Integrations\Recipe_Graph_Piece( $context );
		return $pieces;
	}

	/**
	 * Gets the enriched form of a Google Schema for a recipe.
	 *
	 * @param Recipe $recipe Recipe object.
	 * @param object $post   Post object.
	 * @return array
	 */
	public static function get_enriched_google_schema_for_recipe( Recipe $recipe, $post ) {
		$schema = self::get_google_schema_for_recipe( $recipe );
		$schema = self::enrich_google_schema_with_post_author( $schema, $recipe, $post );
		$schema = self::enrich_google_schema_with_post_video( $schema, $recipe, $post );
		$schema = self::enrich_google_schema_with_comments( $schema, $recipe, self::get_wp_query_comments() );

		if ( '0000-00-00 00:00:00' !== $post->post_date ) {
			$schema['datePublished'] = date( 'Y-m-d', strtotime( $post->post_date ) );
		}
		return $schema;
	}

	/**
	 * Generates the Google Schema for a recipe object.
	 *
	 * @param Recipe $recipe Recipe object.
	 * @return array
	 */
	public static function get_google_schema_for_recipe( Recipe $recipe ) {
		$recipe_json = $recipe->to_json();
		$schema      = array(
			'@context'    => 'https://schema.org/',
			'@type'       => 'Recipe',
			'name'        => $recipe_json['title'],
			'description' => $recipe_json['description'],
		);

		if ( ! empty( $recipe_json['author_name'] ) ) {
			$author = array(
				'@type' => 'Person',
				'name'  => $recipe_json['author_name'],
			);
			$url    = get_option( Tasty_Recipes::DEFAULT_AUTHOR_LINK_OPTION, '' );
			if ( $url ) {
				$author['url'] = $url;
			}
			$schema['author'] = $author;
		}

		if ( ! empty( $recipe_json['keywords'] ) ) {
			$schema['keywords'] = $recipe_json['keywords'];
		}

		$image_urls = array();
		foreach ( self::$json_ld_image_sizes as $image_size ) {
			$image_url = $recipe->get_image_size( $image_size );
			if ( $image_url ) {
				$image_urls[] = $image_url['url'];
			}
		}
		$full_size = self::get_recipe_image_size_with_fallback( $recipe, 'full' );
		if ( ! empty( $image_urls ) ) {
			if ( ! empty( $full_size ) ) {
				$image_urls[] = $full_size['url'];
			}
			$schema['image'] = array_values( array_unique( $image_urls ) );
		} else {
			if ( ! empty( $full_size ) ) {
				$schema['image'] = array(
					'@type'  => 'ImageObject',
					'url'    => $full_size['url'],
					'height' => $full_size['height'],
					'width'  => $full_size['width'],
				);
			}
		}

		$object = get_queried_object();
		if ( $object && is_a( $object, 'WP_Post' ) ) {
			$schema['url'] = get_permalink( $object->ID );
		}

		$schema['recipeIngredient'] = array();
		foreach ( self::parse_recipe_component_list( $recipe_json['ingredients'] ) as $ingredient ) {
			if ( 'heading' === $ingredient['type'] ) {
				continue;
			}
			$schema['recipeIngredient'][] = $ingredient['value'];
		}
		$schema['recipeInstructions'] = array();
		$heading_position             = null;
		foreach ( self::parse_recipe_component_list( $recipe_json['instructions'] ) as $instruction ) {
			if ( 'heading' === $instruction['type'] ) {
				$heading_position               = count( $schema['recipeInstructions'] );
				$schema['recipeInstructions'][] = array(
					'@type'           => 'HowToSection',
					'name'            => $instruction['value'],
					'itemListElement' => array(),
				);
				continue;
			}
			$item = array(
				'@type' => 'HowToStep',
				'text'  => $instruction['value'],
			);
			if ( ! is_null( $heading_position ) ) {
				$schema['recipeInstructions'][ $heading_position ]['itemListElement'][] = $item;
			} else {
				$schema['recipeInstructions'][] = $item;
			}
		}

		foreach ( Recipe::get_cooking_attributes() as $name => $meta ) {
			if ( ! empty( $recipe_json[ $name ] ) ) {
				$value = $recipe_json[ $name ];
				if ( false !== stripos( $name, '_time' ) ) {
					$value = self::get_duration_for_time( $value );
				}
				if ( $value && in_array( $name, array( 'cuisine', 'category', 'method' ), true ) ) {
					$bits  = explode( ',', $value );
					$value = trim( $bits[0] );
				}
				$schema[ $meta['property'] ] = $value;
			}
		}

		$total_reviews = $recipe->get_total_reviews();
		if ( $total_reviews ) {
			$schema['aggregateRating'] = array(
				'@type'       => 'AggregateRating',
				'reviewCount' => (string) $total_reviews,
				'ratingValue' => (string) round( (float) $recipe->get_average_rating(), 1 ),
			);
		}

		$nutrition = array();
		if ( $recipe->get_nutrifox_response() ) {
			foreach ( Recipe::get_nutrition_attributes() as $name => $meta ) {
				$value = $recipe->get_formatted_nutrifox_value( $name );
				if ( false === $value ) {
					continue;
				}
				$nutrition[ $meta['property'] ] = $value;
			}
		} else {
			foreach ( Recipe::get_nutrition_attributes() as $name => $meta ) {
				// '0' is a valid value.
				if ( '' === $recipe_json[ $name ] ) {
					continue;
				}
				$nutrition[ $meta['property'] ] = $recipe_json[ $name ];
			}
		}

		if ( ! empty( $nutrition ) ) {
			$nutrition['@type']  = 'nutritionInformation';
			$schema['nutrition'] = $nutrition;
		}

		$video_url   = $recipe->get_video_url();
		$oembed_data = $recipe->get_video_url_response();
		if ( $video_url && $oembed_data ) {
			$schema['video'] = self::get_google_schema_video_from_oembed_data( $oembed_data );
		}

		/**
		 * Permit modification of the Google Schema data generated for a recipe.
		 *
		 * @param array  $schema Existing Google Schema data.
		 * @param object $recipe Tasty Recipes recipe object.
		 */
		$schema = apply_filters( 'tasty_recipes_google_schema', $schema, $recipe );
		return $schema;
	}

	/**
	 * Gets Google Schema video format from oEmbed structured data.
	 *
	 * @param array $oembed_data oEmbed structured data.
	 * @return array
	 */
	public static function get_google_schema_video_from_oembed_data( $oembed_data ) {
		$video_schema = array(
			'@context' => 'http://schema.org',
			'@type'    => 'VideoObject',
		);
		if ( isset( $oembed_data->title ) ) {
			$video_schema['name'] = $oembed_data->title;
		}
		if ( isset( $oembed_data->description ) ) {
			$video_schema['description'] = $oembed_data->description;
		}
		if ( ! empty( $oembed_data->duration ) ) {
			$duration_seconds = $oembed_data->duration;
			$duration         = self::get_duration_for_time( "{$duration_seconds} seconds" );

			$video_schema['duration'] = $duration;
		}
		if ( isset( $oembed_data->embed_url ) ) {
			$video_schema['embedUrl'] = $oembed_data->embed_url;
		} elseif ( isset( $oembed_data->embedUrl ) ) {
			$video_schema['embedUrl'] = $oembed_data->embedUrl;
		} elseif ( isset( $oembed_data->html ) ) {
			if ( preg_match( '#src="([^"]+)"#', $oembed_data->html, $matches ) ) {
				$video_schema['embedUrl'] = $matches[1];
			}
		}

		if ( isset( $oembed_data->content_url ) ) {
			$video_schema['contentUrl'] = $oembed_data->content_url;
		} elseif ( isset( $oembed_data->contentUrl ) ) {
			$video_schema['contentUrl'] = $oembed_data->contentUrl;
		}
		if ( isset( $oembed_data->thumbnail_url ) ) {
			$video_schema['thumbnailUrl'] = array( $oembed_data->thumbnail_url );
		}
		if ( isset( $oembed_data->upload_date ) ) {
			$video_schema['uploadDate'] = date( 'Y-m-d', strtotime( $oembed_data->upload_date ) );
		} elseif ( isset( $oembed_data->uploadDate ) ) {
			$video_schema['uploadDate'] = date( 'Y-m-d', strtotime( $oembed_data->uploadDate ) );
		}
		return $video_schema;
	}

	/**
	 * Enriches a Google Schema with post author if the author doesn't exist.
	 *
	 * @param array  $schema Existing Google Schema data.
	 * @param object $recipe Recipe object.
	 * @param object $post   Post object.
	 * @return array
	 */
	public static function enrich_google_schema_with_post_author( $schema, $recipe, $post ) {

		if ( ! isset( $schema['author'] ) ) {
			$user = get_user_by( 'id', $post->post_author );
			if ( $user ) {
				$schema['author'] = array(
					'@type' => 'Person',
					'name'  => $user->display_name,
				);
			}
		}

		if ( isset( $schema['author'] ) ) {
			$shortcodes = Tasty_Recipes::get_recipe_ids_from_content( $post->post_content, array( 'full-result' => true ) );
			$shortcodes = wp_filter_object_list( $shortcodes, array( 'id' => $recipe->get_id() ) );
			if ( ! empty( $shortcodes[0]['author_link'] ) ) {
				$schema['author']['url'] = $shortcodes[0]['author_link'];
			}
		}

		return $schema;
	}

	/**
	 * Enrich a Google Schema with video from post content if video doesn't exist.
	 *
	 * @param array  $schema Existing Google Schema data.
	 * @param object $recipe Recipe object.
	 * @param object $post   Post object.
	 * @return array
	 */
	public static function enrich_google_schema_with_post_video( $schema, $recipe, $post ) {

		if ( isset( $schema['video'] ) ) {
			return $schema;
		}

		// Transform any AdThrive shortcodes into a special format
		// that only Tasty Recipes will process.
		$processed_content = str_replace( '[adthrive-in-post-video-player ', '[tr-adthrive-in-post-video-player ', $post->post_content );

		// Process any AdThrive, YouTube, or Vimeo videos within the post content.
		$processed_content = apply_filters( 'the_content', $processed_content );

		if ( false === stripos( $processed_content, '<iframe ' )
			&& false === stripos( $processed_content, '<script ' )
			&& false === stripos( $processed_content, 'adthrive-in-post-video-player' ) ) {
			return $schema;
		}

		if ( preg_match_all( '#(<iframe[^>]*src=["\']([^"\']+)["\'][^>]*>[^<]*</iframe>|<script[^>]*src=["\']([^"\']+)["\'][^>]*>[^<]*</script>|\[tr-adthrive-in-post-video-player\s([^\]]+)\])#', $processed_content, $matches, PREG_OFFSET_CAPTURE ) ) {
			$diff = 0;
			foreach ( $matches[1] as $index => $match ) {
				$match[1] = $match[1] + $diff;
				$before   = substr( $processed_content, 0, $match[1] );
				// Ignore content inbetween
				// <!-- tasty-recipes-ignore -->These apples<!-- /tasty-recipes-ignore -->.
				preg_match_all( '#<!--\s?\/?tasty-recipes-ignore\s?-->#i', $before, $before_matches );
				if ( ! empty( $before_matches[0] ) ) {
					$last_before = array_pop( $before_matches[0] );
					// If the last marker wasn't a closing marker, then we're
					// inside the ignore part.
					if ( false === stripos( $last_before, '/tasty-recipes-ignore' ) ) {
						continue;
					}
				}
				if ( false !== stripos( $match[0], 'tr-adthrive-in-post-video-player' ) ) {
					$shortcode_inner = $matches[4][ $index ][0];
					$shortcode_inner = str_replace( array( '&#8221;', '&#8220;', '&#8243;' ), '"', $shortcode_inner );
					// WP 4.0 doesn't correctly handle dashes in shortcode attributes.
					$shortcode_inner = str_replace(
						array(
							'video-id',
							'upload-date',
						),
						array(
							'video_id',
							'upload_date',
						),
						$shortcode_inner
					);
					$atts            = shortcode_parse_atts( $shortcode_inner );
					if ( ! empty( $atts['video_id'] ) ) {
						$oembed_data     = Tasty_Recipes::get_template_part(
							'video/adthrive-oembed-response',
							array(
								'video_id'    => $atts['video_id'],
								'title'       => isset( $atts['name'] ) ? html_entity_decode( $atts['name'], ENT_COMPAT, 'UTF-8' ) : '',
								'description' => isset( $atts['description'] ) ? html_entity_decode( $atts['description'], ENT_COMPAT, 'UTF-8' ) : '',
								'upload_date' => isset( $atts['upload_date'] ) ? $atts['upload_date'] : '',
							)
						);
						$schema['video'] = self::get_google_schema_video_from_oembed_data( json_decode( $oembed_data ) );
						return $schema;
					}
				} else {
					$src       = ! empty( $matches[3][ $index ][0] ) ? $matches[3][ $index ][0] : $matches[2][ $index ][0];
					$video_url = false;
					// We can only reverse the 'src' for Mediavine, YouTube and Vimeo.
					if ( preg_match( '#//video\.mediavine\.com/videos/([^\.]+).js#', $src, $video_matches ) ) {
						$video_url = 'https://dashboard.mediavine.com/videos/' . $video_matches[1] . '/edit/';
					} elseif ( preg_match( '#https?://www\.youtube\.com/embed/([^?]+)#', $src, $video_matches ) ) {
						$video_url = 'https://www.youtube.com/watch?v=' . $video_matches[1];
					} elseif ( preg_match( '#https?://player\.vimeo\.com/video/([^?]+)#', $src, $video_matches ) ) {
						$video_url = 'https://vimeo.com/' . $video_matches[1];
					}
					if ( ! $video_url ) {
						continue;
					}
					$cache_key = '_tr_oembed_' . md5( $video_url );
					$cache     = get_post_meta( $post->ID, $cache_key, true );
					if ( ! $cache ) {
						if ( ! function_exists( '_wp_oembed_get_object' ) ) {
							require_once( ABSPATH . WPINC . '/class-oembed.php' );
						}
						$wp_oembed = _wp_oembed_get_object();
						$provider  = $wp_oembed->get_provider( $video_url );
						if ( ! $provider ) {
							update_post_meta( $post->ID, $cache_key, '{{unknown}}' );
							continue;
						}
						$response_data = $wp_oembed->fetch( $provider, $video_url );
						if ( false === $response_data ) {
							update_post_meta( $post->ID, $cache_key, '{{unknown}}' );
							continue;
						}
						update_post_meta( $post->ID, $cache_key, $response_data );
						Content_Model::enrich_youtube_oembed( $post->ID, $video_url, $cache_key );
						$cache = get_post_meta( $post->ID, $cache_key, true );
					}
					$schema['video'] = self::get_google_schema_video_from_oembed_data( $cache );
					return $schema;
				}
			}
		}

		return $schema;
	}

	/**
	 * Enrich a Google Schema with reviews from post content if reviews don't exist.
	 *
	 * @param array  $schema   Existing Google Schema data.
	 * @param object $recipe   Recipe object.
	 * @param array  $comments Any comments to enrich from.
	 * @return array
	 */
	public static function enrich_google_schema_with_comments( $schema, $recipe, $comments ) {

		if ( isset( $schema['review'] ) || empty( $comments ) ) {
			return $schema;
		}
		/**
		 * Optionally set a limit on the number of reviews included in the schema markup.
		 *
		 * @param int $number Can be any number
		 */
		$limit = apply_filters( 'tasty_recipes_limit_schema_reviews', null );

		$reviews = array();
		foreach ( $comments as $comment ) {
			$rating = get_comment_meta( $comment->comment_ID, Ratings::COMMENT_META_KEY, true );
			if ( ! $rating ) {
				continue;
			}
			$reviews[] = array(
				'@type'         => 'Review',
				'reviewRating'  => array(
					'@type'       => 'Rating',
					'ratingValue' => (string) $rating,
				),
				'author'        => array(
					'@type' => 'Person',
					'name'  => $comment->comment_author,
				),
				'datePublished' => date( 'Y-m-d', strtotime( $comment->comment_date ) ),
				'reviewBody'    => $comment->comment_content,
			);
			if ( ! is_null( $limit ) && count( $reviews ) >= $limit ) {
				break;
			}
		}

		if ( ! empty( $reviews ) ) {
			$schema['review'] = $reviews;
		}

		return $schema;
	}

	/**
	 * Get the comments generated by WP_Query.
	 *
	 * Duplicates some of the comments_template() function.
	 *
	 * @return array
	 */
	public static function get_wp_query_comments() {
		global $post, $user_ID;

		/*
		 * Comment author information fetched from the comment cookies.
		 */
		$commenter = wp_get_current_commenter();

		/*
		 * The email address of the current comment author escaped for use in attributes.
		 * Escaped by sanitize_comment_cookies().
		 */
		$comment_author_email = $commenter['comment_author_email'];

		$comment_args = array(
			'orderby'                   => 'comment_date_gmt',
			'order'                     => 'ASC',
			'status'                    => 'approve',
			'post_id'                   => $post->ID,
			'no_found_rows'             => false,
			'update_comment_meta_cache' => false, // We lazy-load comment meta for performance.
		);

		if ( get_option( 'thread_comments' ) ) {
			$comment_args['hierarchical'] = 'threaded';
		} else {
			$comment_args['hierarchical'] = false;
		}

		if ( $user_ID ) {
			$comment_args['include_unapproved'] = array( $user_ID );
		} elseif ( ! empty( $comment_author_email ) ) {
			$comment_args['include_unapproved'] = array( $comment_author_email );
		}

		$per_page = 0;
		if ( get_option( 'page_comments' ) ) {
			$per_page = (int) get_query_var( 'comments_per_page' );
			if ( 0 === $per_page ) {
				$per_page = (int) get_option( 'comments_per_page' );
			}

			$comment_args['number'] = $per_page;
			$page                   = (int) get_query_var( 'cpage' );

			if ( $page ) {
				$comment_args['offset'] = ( $page - 1 ) * $per_page;
			} elseif ( 'oldest' === get_option( 'default_comments_page' ) ) {
				$comment_args['offset'] = 0;
			} else {
				// If fetching the first page of 'newest', we need a top-level comment count.
				$top_level_query = new \WP_Comment_Query();
				$top_level_args  = array(
					'count'   => true,
					'orderby' => false,
					'post_id' => $post->ID,
					'status'  => 'approve',
				);

				if ( $comment_args['hierarchical'] ) {
					$top_level_args['parent'] = 0;
				}

				if ( isset( $comment_args['include_unapproved'] ) ) {
					$top_level_args['include_unapproved'] = $comment_args['include_unapproved'];
				}

				$top_level_count = $top_level_query->query( $top_level_args );

				$comment_args['offset'] = ( ceil( $top_level_count / $per_page ) - 1 ) * $per_page;
			}
		}

		/**
		 * Filters the arguments used to query comments in comments_template().
		 *
		 * @since 4.5.0
		 *
		 * @see WP_Comment_Query::__construct()
		 *
		 * @param array $comment_args {
		 *     Array of WP_Comment_Query arguments.
		 *
		 *     @type string|array $orderby                   Field(s) to order by.
		 *     @type string       $order                     Order of results. Accepts 'ASC' or 'DESC'.
		 *     @type string       $status                    Comment status.
		 *     @type array        $include_unapproved        Array of IDs or email addresses whose unapproved comments
		 *                                                   will be included in results.
		 *     @type int          $post_id                   ID of the post.
		 *     @type bool         $no_found_rows             Whether to refrain from querying for found rows.
		 *     @type bool         $update_comment_meta_cache Whether to prime cache for comment meta.
		 *     @type bool|string  $hierarchical              Whether to query for comments hierarchically.
		 *     @type int          $offset                    Comment offset.
		 *     @type int          $number                    Number of comments to fetch.
		 * }
		 */
		$comment_args  = apply_filters( 'comments_template_query_args', $comment_args );
		$comment_query = new \WP_Comment_Query( $comment_args );
		$_comments     = $comment_query->comments;

		// Trees must be flattened before they're passed to the walker.
		if ( $comment_args['hierarchical'] ) {
			$comments_flat = array();
			foreach ( $_comments as $_comment ) {
				$comments_flat[]  = $_comment;
				$comment_children = $_comment->get_children(
					array(
						'format'  => 'flat',
						'status'  => $comment_args['status'],
						'orderby' => $comment_args['orderby'],
					)
				);

				foreach ( $comment_children as $comment_child ) {
					$comments_flat[] = $comment_child;
				}
			}
		} else {
			$comments_flat = $_comments;
		}

		/**
		 * Filters the comments array.
		 *
		 * @since 2.1.0
		 *
		 * @param array $comments Array of comments supplied to the comments template.
		 * @param int   $post_ID  Post ID.
		 */
		return apply_filters( 'comments_array', $comments_flat, $post->ID );
	}

	/**
	 * Get a list from a string of ingredients or instructions
	 *
	 * @param string $component Existing text to parse.
	 * @return array
	 */
	public static function parse_recipe_component_list( $component ) {
		$component_list = array();
		if ( preg_match_all( self::LI_MATCH_REGEX, $component, $matches, PREG_OFFSET_CAPTURE ) ) {
			$prev_offset = 0;
			foreach ( $matches[1] as $match ) {
				$maybe_heading = mb_strcut( $component, $prev_offset, ( $match[1] - $prev_offset ), 'UTF-8' );
				$maybe_heading = str_replace( array( '<ul>', '</ul>', '<ol>', '</ol>', '<li>', '</li>' ), '', $maybe_heading );
				$heading_bits  = explode( PHP_EOL, $maybe_heading );
				array_reverse( $heading_bits );
				foreach ( $heading_bits as $bit ) {
					$heading = self::get_heading( $bit );
					if ( $heading ) {
						$component_list[] = array(
							'type'  => 'heading',
							'value' => $heading,
						);
					}
				}
				$component_list[] = array(
					'type'  => 'item',
					'value' => $match[0],
				);
				$prev_offset      = $match[1] + mb_strlen( $match[0], 'UTF-8' );
			}
		} else {
			// No list items detected, so fall back to stripping out things that look like headings.
			$bits = explode( PHP_EOL, $component );
			foreach ( $bits as $bit ) {

				$test_bit = trim( strip_tags( $bit ) );
				if ( empty( $test_bit ) ) {
					continue;
				}
				$heading = self::get_heading( $bit );
				if ( $heading ) {
					$component_list[] = array(
						'type'  => 'heading',
						'value' => $heading,
					);
					continue;
				}
				$component_list[] = array(
					'type'  => 'item',
					'value' => $bit,
				);
			}
		}
		foreach ( $component_list as $i => $component_item ) {
			$component_list[ $i ]['value'] = trim( strip_tags( $component_item['value'] ) );
		}
		return $component_list;
	}

	/**
	 * Format times so strtotime() can use them properly.
	 *
	 * @param string $time Existing time string.
	 * @param mixed  $now  What now should be considered as.
	 * @return string
	 */
	public static function strtotime( $time, $now = null ) {
		if ( null === $now ) {
			$now = time();
		}
		// Parse string to remove any info in parentheses.
		$time = preg_replace( '/\([^\)]+\)/', '', $time );

		// Parse string to handle 00:10 time format.
		$time = preg_replace( '/(\d+):(\d+)/', '$1 hours $2 minutes', $time );

		return strtotime( $time, $now );
	}

	/**
	 * Calculate the number of hours and minutes from a unix timestmp
	 *
	 * @param integer $time Time to compare.
	 * @return array
	 */
	public static function convert_seconds_to_minutes_hours( $time ) {
		$hours   = (int) floor( $time / HOUR_IN_SECONDS );
		$minutes = (int) floor( ( $time - ( $hours * HOUR_IN_SECONDS ) ) / MINUTE_IN_SECONDS );
		$seconds = $time - ( $hours * HOUR_IN_SECONDS ) - ( $minutes * MINUTE_IN_SECONDS );
		return array(
			'hours'   => $hours,
			'minutes' => $minutes,
			'seconds' => $seconds,
		);
	}


	/**
	 * Get an ISO 8601 duration based on a human-readable time.
	 *
	 * @param string $time Time to get the duration of.
	 * @return string
	 */
	public static function get_duration_for_time( $time ) {
		// Assume a number is minutes.
		if ( is_numeric( $time ) ) {
			$time = "{$time} minutes";
		}
		$now              = time();
		$time             = self::strtotime( $time, $now );
		$time             = $time - $now;
		$calculated_times = self::convert_seconds_to_minutes_hours( $time );
		$duration         = 'PT';
		if ( ! empty( $calculated_times['hours'] ) ) {
			$duration .= $calculated_times['hours'] . 'H';
		}
		if ( ! empty( $calculated_times['minutes'] ) ) {
			$duration .= $calculated_times['minutes'] . 'M';
		}
		if ( ! empty( $calculated_times['seconds'] ) ) {
			$duration .= $calculated_times['seconds'] . 'S';
		}
		return 'PT' !== $duration ? $duration : '';
	}

	/**
	 * Get a human-readable time based on an ISO 8601 duration.
	 *
	 * @param string $duration Duration to process.
	 * @return string
	 */
	public static function get_time_for_duration( $duration ) {
		try {
			$date = new \DateInterval( $duration );
		} catch ( \Exception $e ) {
			return '';
		}
		$total = ( $date->h * HOUR_IN_SECONDS ) + ( $date->i * MINUTE_IN_SECONDS );
		return self::format_time_for_human( $total );
	}

	/**
	 * Format a timestamp for a human.
	 *
	 * @param integer $timestamp Timestamp to format.
	 * @return string
	 */
	public static function format_time_for_human( $timestamp ) {
		$bits = self::convert_seconds_to_minutes_hours( $timestamp );
		if ( 0 === $timestamp % HOUR_IN_SECONDS ) {
			if ( HOUR_IN_SECONDS === $timestamp ) {
				// translators: Individual hour.
				$formatted = sprintf( __( '%s hour', 'tasty-recipes' ), $bits['hours'] );
			} else {
				// translators: Multiple hours.
				$formatted = sprintf( __( '%s hours', 'tasty-recipes' ), $bits['hours'] );
			}
		} elseif ( $timestamp >= HOUR_IN_SECONDS * 2 ) {
			// translators: Multiple hours and minutes.
			$formatted = sprintf( __( '%1$s hours %2$s minutes', 'tasty-recipes' ), $bits['hours'], $bits['minutes'] );
		} elseif ( $timestamp >= HOUR_IN_SECONDS && ( $timestamp < HOUR_IN_SECONDS * 2 ) ) {
			// translators: Individual hour but multiple minutes.
			$formatted = sprintf( __( '%1$s hour %2$s minutes', 'tasty-recipes' ), $bits['hours'], $bits['minutes'] );
		} elseif ( $timestamp <= MINUTE_IN_SECONDS ) {
			// translators: Individual minute.
			$formatted = sprintf( __( '%s minute', 'tasty-recipes' ), $bits['minutes'] );
		} elseif ( $timestamp < HOUR_IN_SECONDS ) {
			// translators: Multiple minutes.
			$formatted = sprintf( __( '%s minutes', 'tasty-recipes' ), $bits['minutes'] );
		}
		return $formatted;
	}

	/**
	 * Check whether a given string could be a heading.
	 *
	 * @param string $string String to process.
	 * @return bool
	 */
	public static function get_heading( $string ) {
		$string = trim( $string );
		// For The Red Beans:.
		if ( ':' === substr( $string, -1, 1 ) ) {
			return substr( $string, 0, -1 );
		}
		// <strong>For The Red Beans</strong>
		if ( '<strong>' === substr( $string, 0, 8 ) && '</strong>' === substr( $string, -9, 9 ) ) {
			return substr( $string, 8, -9 );
		}
		// <h3>For The Red Beans</h3>
		if ( preg_match( '#^<h[1-6]>(.+)<\/h[1-6]>$#', $string, $matches ) ) {
			return $matches[1];
		}
		return false;
	}

	/**
	 * Get an image size for the recipe, falling back to the post if needed.
	 *
	 * @param Recipe $recipe Recipe object to inspect.
	 * @param string $size   Specific image size to get.
	 * @return array|false
	 */
	public static function get_recipe_image_size_with_fallback( $recipe, $size ) {
		$image_size = $recipe->get_image_size( $size );
		if ( $image_size ) {
			return $image_size;
		}
		$current_post = get_post();
		if ( ! $current_post ) {
			return false;
		}
		$thumbnail_id = get_post_thumbnail_id( $current_post->ID );
		if ( ! $thumbnail_id ) {
			preg_match( '#wp-image-([\d]+)#', $current_post->post_content, $matches );
			if ( ! empty( $matches[1] ) ) {
				$thumbnail_id = $matches[1];
			}
		}
		if ( ! $thumbnail_id ) {
			return false;
		}
		$image_src = wp_get_attachment_image_src( $thumbnail_id, $size );
		if ( $image_src ) {
			return array(
				'url'    => $image_src[0],
				'width'  => $image_src[1],
				'height' => $image_src[2],
			);
		}
		return false;
	}

}
