<?php
/**
 * Registers shortcodes used by the plugin.
 *
 * @package Tasty_Recipes
 */

namespace Tasty_Recipes;

use Tasty_Recipes;
use Tasty_Recipes\Content_Model;
use Tasty_Recipes\Objects\Recipe;

/**
 * Registers shortcodes used by the plugin.
 */
class Shortcodes {

	/**
	 * Recipe shortcode name.
	 *
	 * @var string
	 */
	const RECIPE_SHORTCODE = 'tasty-recipe';

	/**
	 * Registers shortcodes with their callbacks.
	 */
	public static function action_init_register_shortcode() {
		add_shortcode( self::RECIPE_SHORTCODE, array( __CLASS__, 'render_tasty_recipe_shortcode' ) );
	}

	/**
	 * Add "Jump to Recipe" and "Print Recipe" buttons when the post has a recipe.
	 *
	 * @param string $content Existing post content.
	 * @return string
	 */
	public static function filter_the_content_late( $content ) {

		$post = get_post();
		if ( ! $post ) {
			return $content;
		}

		$recipe_ids = Tasty_Recipes::get_recipe_ids_for_post( $post->ID );

		$should_prepend_jump_to = get_option( Tasty_Recipes::QUICK_LINKS_OPTION, '' );

		if ( empty( $recipe_ids ) ) {
			$should_prepend_jump_to = false;
		}

		if ( false !== stripos( $content, 'Jump to Recipe' ) ) {
			$should_prepend_jump_to = false;
		}

		if ( ! is_singular() ) {
			$should_prepend_jump_to = false;
		}

		/**
		 * Filter to allow for more granular control over whether 'Jump to'
		 * should appear.
		 *
		 * @param boolean $should_prepend_jump_to Whether or not to prepend.
		 * @param integer $post_id                ID for the recipe's post.
		 * @param array   $recipe_ids             Recipe IDs within the post.
		 */
		$should_prepend_jump_to = apply_filters( 'tasty_recipes_should_prepend_jump_to', $should_prepend_jump_to, $post->ID, $recipe_ids );

		if ( $should_prepend_jump_to ) {
			$recipe_id = array_shift( $recipe_ids );
			$open_new  = '';

			/**
			 * Filter to control whether the print link opens in a new window.
			 *
			 * @param boolean
			 */
			if ( apply_filters( 'tasty_recipes_print_link_open_new', false ) ) {
				$open_new = ' target="_blank"';
			}

			$links = array();
			if ( in_array( $should_prepend_jump_to, array( 'jump', 'both' ), true ) ) {
				$links[] = '<a class="tasty-recipes-jump-link" href="#tasty-recipes-' . $recipe_id . '">' . __( 'Jump to Recipe', 'tasty-recipes' ) . '</a>';
			}
			if ( in_array( $should_prepend_jump_to, array( 'both' ), true ) ) {
				$links[] = '<span>&middot;</span>';
			}
			if ( in_array( $should_prepend_jump_to, array( 'print', 'both' ), true ) ) {
				$links[] = '<a class="tasty-recipes-print-link" href="' . esc_url( tasty_recipes_get_print_url( $post->ID, $recipe_id ) ) . '"' . $open_new . '>' . __( 'Print Recipe', 'tasty-recipes' ) . '</a>';
			}
			$links = implode( '', $links );

			$html    = <<<EOT
<style>
.tasty-recipes-quick-links { text-align:center; }
.tasty-recipes-quick-links a { padding: 0.5rem; }
</style>
<div class="tasty-recipes-quick-links">
	{$links}
</div>
EOT;
			$content = $html . PHP_EOL . PHP_EOL . $content;
		}

		return $content;
	}

	/**
	 * Renders the Tasty Recipes shortcode.
	 *
	 * @param array  $atts    Shortcode attributes.
	 * @param string $content Shortcode inner content.
	 */
	public static function render_tasty_recipe_shortcode( $atts, $content = '' ) {

		if ( empty( $atts['id'] ) ) {
			return '';
		}

		$recipe = Recipe::get_by_id( (int) $atts['id'] );
		if ( ! $recipe ) {
			return '';
		}

		// Ensures the shortcode preview has access to the full recipe data.
		Tasty_Recipes::get_instance()->recipe_json = $recipe_json = $recipe->to_json();

		// There are some dependencies on the parent post.
		$current_post = get_post();

		$wrapper_classes = array(
			'tasty-recipes',
			'tasty-recipes-' . $recipe->get_id(),
		);
		if ( tasty_recipes_is_print() ) {
			$wrapper_classes[] = 'tasty-recipes-print';
		} else {
			$wrapper_classes[] = 'tasty-recipes-display';
		}

		if ( ! empty( $recipe_json['image_sizes']['thumbnail'] ) ) {
			$wrapper_classes[] = 'tasty-recipes-has-image';
		} else {
			$wrapper_classes[] = 'tasty-recipes-no-image';
		}

		// Added by block editor 'Additional Classes' feature.
		if ( ! empty( $atts['className'] ) ) {
			$wrapper_classes = array_merge( $wrapper_classes, explode( ' ', $atts['className'] ) );
		}

		/**
		 * Modify template used in rendering recipe.
		 *
		 * @param string $template
		 */
		$template = apply_filters( 'tasty_recipes_recipe_template', 'tasty-recipes' );
		if ( ! in_array( $template, array( 'tasty-recipes', 'easyrecipe' ), true ) ) {
			return '';
		}

		$custom_design = false;
		if ( file_exists( get_stylesheet_directory() . '/tasty-recipes.php' ) ) {
			$template = get_stylesheet_directory() . '/tasty-recipes.php';
		} else {
			$custom_design = get_option( Tasty_Recipes::TEMPLATE_OPTION, '' );
			$custom_path   = dirname( TASTY_RECIPES_PLUGIN_FILE ) . '/templates/designs/' . $custom_design . '/tasty-recipes.php';
			if ( $custom_design && file_exists( $custom_path ) ) {
				$template = $custom_path;
			} else {
				$template = 'recipe/' . $template;
			}
		}

		if ( 'recipe/easyrecipe' === $template ) {
			$wrapper_classes[] = 'easyrecipe';
		}

		$shareasale = get_option( Tasty_Recipes::SHAREASALE_OPTION );
		if ( $shareasale ) {
			$wrapper_classes[] = 'tasty-recipes-has-plug';
		}

		$before_recipe = '';
		$error         = get_post_meta( $recipe->get_id(), 'nutrifox_error', true );
		if ( $error && current_user_can( 'edit_posts' ) && is_admin() ) {
			$before_recipe .= '<div style="border:4px solid #dc3232;padding:10px 12px;margin-top:10px;margin-bottom:10px;"><p>Nutrifox API integration failed.</p>';
			$before_recipe .= '<pre>' . wp_kses_post( $error->get_error_message() ) . '</pre>';
			$before_recipe .= '<p>Try saving the recipe again. Contact Tasty Recipes support if the error persists.</p>';
			$before_recipe .= '</div>' . PHP_EOL;
		}
		if ( $current_post ) {
			$before_recipe .= '<a class="button tasty-recipes-print-button tasty-recipes-no-print"';
			if ( tasty_recipes_is_print() ) {
				$before_recipe .= ' onclick="window.print();event.preventDefault();"';
			}
			$before_recipe .= ' href="' . esc_url( tasty_recipes_get_print_url( $current_post->ID, $recipe->get_id() ) ) . '">' . esc_html__( 'Print', 'tasty-recipes' ) . '</a>';
		}

		/**
		 * Modify output rendered before the recipe.
		 *
		 * @param string $before_recipe  Prepared output to display.
		 * @param Recipe $recipe
		 */
		$before_recipe = apply_filters( 'tasty_recipes_before_recipe', $before_recipe, $recipe );

		// Begin the recipe output.
		$ret  = $before_recipe;
		$ret .= '<div id="tasty-recipes-' . $recipe->get_id() . '" class="' . esc_attr( implode( ' ', $wrapper_classes ) ) . '">' . PHP_EOL;

		$embed_data = $recipe->get_video_url_response();

		$orig_ingredients = $recipe_json['ingredients'];
		$ingredients      = self::process_ingredients_annotate_with_spans( $orig_ingredients );
		if ( ! is_feed() && $ingredients !== $orig_ingredients ) {
			$recipe_scalable  = '<button class="tasty-recipes-scale-button tasty-recipes-scale-button-active" data-amount="1">' . __( '1x', 'tasty-recipes' ) . '</button>';
			$recipe_scalable .= '<button class="tasty-recipes-scale-button" data-amount="2">' . __( '2x', 'tasty-recipes' ) . '</button>';
			$recipe_scalable .= '<button class="tasty-recipes-scale-button" data-amount="3">' . __( '3x', 'tasty-recipes' ) . '</button>';
		} else {
			$recipe_scalable = false;
		}

		/**
		 * Allow ingredient scaling to be disabled.
		 *
		 * @param mixed $recipe_scalable Existing scalable data.
		 */
		$recipe_scalable = apply_filters( 'tasty_recipes_scalable', $recipe_scalable );

		$recipe_author_name = '';
		if ( ! empty( $recipe_json['author_name'] ) ) {
			$link = ! empty( $atts['author_link'] ) ? $atts['author_link'] : get_option( Tasty_Recipes::DEFAULT_AUTHOR_LINK_OPTION, '' );
			if ( $link ) {
				$recipe_author_name = '<a class="tasty-recipes-author-name" href="' . esc_url( $link ) . '">' . $recipe_json['author_name'] . '</a>';
			} else {
				$recipe_author_name = '<span class="tasty-recipes-author-name">' . $recipe_json['author_name'] . '</span>';
			}
		}

		$template_vars = array(
			'recipe'                  => $recipe,
			'recipe_styles'           => '',
			'recipe_scripts'          => '',
			'recipe_json'             => $recipe_json,
			'recipe_title'            => $recipe_json['title_rendered'],
			'recipe_image'            => ! empty( $recipe_json['image_sizes']['thumbnail'] ) ? $recipe_json['image_sizes']['thumbnail']['html'] : '',
			'recipe_rating_icons'     => '',
			'recipe_rating_label'     => '',
			'recipe_author_name'      => $recipe_author_name,
			'recipe_details'          => array(),
			'recipe_description'      => self::do_caption_shortcode( $recipe_json['description_rendered'] ),
			'recipe_ingredients'      => apply_filters( 'tasty_recipes_the_content', self::do_caption_shortcode( $ingredients ) ),
			'recipe_scalable'         => $recipe_scalable,
			'recipe_instructions'     => apply_filters( 'tasty_recipes_the_content', self::do_caption_shortcode( $recipe_json['instructions'] ) ),
			'recipe_keywords'         => $recipe_json['keywords'],
			'recipe_notes'            => $recipe_json['notes_rendered'],
			'recipe_nutrifox_id'      => $recipe_json['nutrifox_id'],
			'recipe_nutrifox_embed'   => '',
			'recipe_video_embed'      => '',
			'recipe_nutrition'        => array(),
			'recipe_hidden_nutrition' => array(),
			'instagram_handle'        => get_option( Tasty_Recipes::INSTAGRAM_HANDLE_OPTION ),
			'instagram_hashtag'       => get_option( Tasty_Recipes::INSTAGRAM_HASHTAG_OPTION ),
		);

		/**
		 * Enable responsive iframes by default, but permit disabling on a site.
		 *
		 * @param boolean $responsive_iframes Whether or not to enable responsive iframes.
		 */
		$responsive_iframes = apply_filters( 'tasty_recipes_enable_responsive_iframes', true );

		$padding_percentage = null;
		if ( ! empty( $embed_data->html ) ) {
			$recipe_video_embed = $embed_data->html;
			// Responsive iframes using CSS when width and height are in pixels.
			// See https://blog.theodo.fr/2018/01/responsive-iframes-css-trick/.
			$height_regex = '#<iframe[^>]+(height="([\d]+)")[^>]+?>#';
			$width_regex  = '#<iframe[^>]+(width="([\d]+)")[^>]+?>#';
			if ( $responsive_iframes
				&& preg_match( $height_regex, $recipe_video_embed, $height_matches )
				&& preg_match( $width_regex, $recipe_video_embed, $width_matches ) ) {
				$height             = (int) $height_matches[2];
				$width              = (int) $width_matches[2];
				$recipe_video_embed = str_replace(
					array(
						$height_matches[1],
						$width_matches[1],
					),
					'',
					$recipe_video_embed
				);
				$padding_percentage = round( ( ( $height / $width ) * 100 ), 2 );
				$recipe_video_embed = '<div class="tasty-recipe-responsive-iframe-container">' . $recipe_video_embed . '</div>';
				// First try to inject the 'fitvidsignore' class into the existing classes.
				$recipe_video_embed = preg_replace( '#(<iframe[^>]+class=")([^"]+)#', '$1fitvidsignore $2', $recipe_video_embed, -1, $count );
				// If no replacements were found, then the <iframe> needs a class.
				if ( ! $count ) {
					$recipe_video_embed = str_replace( '<iframe ', '<iframe class="fitvidsignore" ', $recipe_video_embed );
				}
			}
			$template_vars['recipe_video_embed'] = $recipe_video_embed;
		} elseif ( ! empty( $embed_data->provider_url )
			&& 'www.adthrive.com' === parse_url( $embed_data->provider_url, PHP_URL_HOST ) ) {
			// If the AdThrive plugin is active, assume the <div> will be
			// handled correctly on the frontend.
			if ( shortcode_exists( 'adthrive-in-post-video-player' ) ) {
				// Show the thumbnail as the preview in the backend.
				if ( is_admin() ) {
					$template_vars['recipe_video_embed'] = sprintf( '<img src="%s">', esc_url( $embed_data->thumbnail_url ) );
				} else {
					$template_vars['recipe_video_embed'] = sprintf( '<div class="adthrive-video-player in-post" data-video-id="%s"></div>', $embed_data->video_id );
				}
				// Fallback is to display the shortcode.
			} else {
				$template_vars['recipe_video_embed'] = $recipe->get_video_url();
			}
		}

		$styles = file_get_contents( dirname( dirname( __FILE__ ) ) . '/assets/css/recipe.css' );
		if ( 'recipe/easyrecipe' === $template ) {
			$easyrecipe_css = file_get_contents( dirname( dirname( __FILE__ ) ) . '/assets/css/easyrecipe.css' );
			$easyrecipe_css = str_replace( 'assets/images/easyrecipe', plugins_url( 'assets/images/easyrecipe', dirname( __FILE__ ) ), $easyrecipe_css );
			$styles        .= PHP_EOL . PHP_EOL . $easyrecipe_css;
		}
		if ( $custom_design ) {
			$custom_path = dirname( TASTY_RECIPES_PLUGIN_FILE ) . '/templates/designs/' . $custom_design . '/tasty-recipes.css';
			if ( file_exists( $custom_path ) ) {
				$styles .= PHP_EOL . PHP_EOL . file_get_contents( $custom_path );
			}
		}
		/**
		 * Allow third-parties to more easily inject their own styles.
		 */
		$custom_styles_path = apply_filters( 'tasty_recipes_custom_css', get_stylesheet_directory() . '/tasty-recipes.css' );
		if ( file_exists( $custom_styles_path ) ) {
			$styles .= PHP_EOL . PHP_EOL . file_get_contents( $custom_styles_path );
		}

		if ( $responsive_iframes && null !== $padding_percentage ) {
			$styles .= PHP_EOL . '.tasty-recipe-responsive-iframe-container { position: relative; overflow: hidden; padding-top: ' . $padding_percentage . '%; }';
			$styles .= PHP_EOL . '.tasty-recipe-responsive-iframe-container iframe { position: absolute; top: 0; left: 0; width: 100%; height: 100%; border: 0; }';
		}
		$template_vars['recipe_styles'] = '<style type="text/css">' . PHP_EOL . $styles . PHP_EOL . '</style>' . PHP_EOL;

		if ( $recipe_scalable ) {
			$template_vars['recipe_scripts'] = '<script type="text/javascript">' . PHP_EOL . file_get_contents( dirname( dirname( __FILE__ ) ) . '/assets/js/scale-buttons.js' ) . PHP_EOL . '</script>';
		}

		$total_reviews = $recipe->get_total_reviews();
		if ( $total_reviews ) {
			$average_rating                        = round( (float) $recipe->get_average_rating(), 1 );
			$template_vars['recipe_rating_label']  = '<span class="rating-label">';
			$template_vars['recipe_rating_label'] .= sprintf(
				// translators: Ratings from number of reviews.
				__( '%1$s from %2$s reviews', 'tasty-recipes' ),
				'<span class="average">' . $average_rating . '</span>',
				'<span class="count">' . (int) $total_reviews . '</span>'
			);
			$template_vars['recipe_rating_label'] .= '</span>';
			$template_vars['recipe_rating_icons']  = Ratings::get_rendered_rating( $average_rating );
		}

		if ( ! empty( $template_vars['recipe_author_name'] ) ) {
			$template_vars['recipe_details']['author'] = array(
				'label' => __( 'Author', 'tasty-recipes' ),
				'value' => $template_vars['recipe_author_name'],
				'class' => 'author',
			);
		}

		foreach ( Recipe::get_cooking_attributes() as $attribute => $meta ) {

			if ( empty( $recipe_json[ $attribute ] ) ) {
				continue;
			}
			$value = $recipe_json[ $attribute ];
			if ( 'yield' === $attribute && $recipe_scalable ) {
				$old_value = $value;
				$value     = Unit_Amount_Parser::annotate_string_with_spans( $value );
				if ( $value !== $old_value ) {
					$value .= ' <span class="tasty-recipes-yield-scale"><span data-amount="1">1</span>x</span>';
				}
			}

			$value = '<span class="' . esc_attr( 'tasty-recipes-' . str_replace( '_', '-', $attribute ) ) . '">' . $value . '</span>';
			$template_vars['recipe_details'][ $attribute ] = array(
				'label' => $meta['label'],
				'value' => $value,
				'class' => str_replace( '_', '-', $attribute ),
			);
		}

		$nutrifox = $recipe->get_nutrifox_response();
		foreach ( Recipe::get_nutrition_attributes() as $attribute => $meta ) {
			// '0' is a valid value.
			if ( '' === $recipe_json[ $attribute ] ) {
				// See if the data exists in Nutrifox now.
				if ( $nutrifox
					&& isset( $meta['nutrifox_key'] )
					&& ( ! isset( $nutrifox['nutrients'][ $meta['nutrifox_key'] ]['visible'] )
						|| true === $nutrifox['nutrients'][ $meta['nutrifox_key'] ]['visible'] ) ) {
					$nutrifox_value = $recipe->get_formatted_nutrifox_value( $attribute );
					$template_vars['recipe_hidden_nutrition'][ $attribute ] = array(
						'value' => '<span class="' . esc_attr( 'tasty-recipes-' . str_replace( '_', '-', $attribute ) ) . '">' . $nutrifox_value . '</span>',
					);
				}
				continue;
			}
			$template_vars['recipe_nutrition'][ $attribute ] = array(
				'label' => $meta['label'],
				'value' => '<span class="' . esc_attr( 'tasty-recipes-' . str_replace( '_', '-', $attribute ) ) . '">' . $recipe_json[ $attribute ] . '</span>',
			);
		}

		if ( ! empty( $recipe_json['nutrifox_id'] ) ) {
			$nutrifox_id            = (int) $recipe_json['nutrifox_id'];
			$nutrifox_iframe_url    = sprintf( 'https://nutrifox.com/embed/label/%d', $nutrifox_id );
			$nutrifox_resize_script = file_get_contents( dirname( dirname( __FILE__ ) ) . '/assets/js/nutrifox-resize.js' );

			$template_vars['recipe_nutrifox_embed'] = <<<EOT
<script type="text/javascript">
{$nutrifox_resize_script}
</script>
<iframe id="nutrifox-label-{$nutrifox_id}" src="{$nutrifox_iframe_url}" style="width:100%;border-width:0;"></iframe>
EOT;

		}

		/**
		 * Allow third-parties to modify the template variables prior to rendering.
		 *
		 * @param array $template_vars Template variables to be used.
		 * @param object $recipe Recipe object.
		 */
		$template_vars = apply_filters( 'tasty_recipes_recipe_template_vars', $template_vars, $recipe );
		$ret          .= Tasty_Recipes::get_template_part( $template, $template_vars );
		$ret          .= '</div>';

		if ( $shareasale ) {
			$ret .= '<div class="tasty-recipes-plug">';
			$ret .= esc_html( 'Recipe Card powered by', 'tasty-recipes' );
			$ret .= '<a href="' . esc_url( sprintf( 'https://shareasale.com/r.cfm?b=973044&u=%s&m=69860&urllink=&afftrack=trattr', $shareasale ) ) . '" target="_blank" rel="nofollow"><img nopin="nopin" src="' . esc_url( plugins_url( 'assets/images/tasty-recipes-neutral.svg', dirname( __FILE__ ) ) ) . '" scale="0" height="20px"></a>';
			$ret .= '</div>';
		}
		return $ret;
	}

	/**
	 * Get the shortcode for a given recipe.
	 *
	 * @param Recipe $recipe Recipe instance.
	 */
	public static function get_shortcode_for_recipe( Recipe $recipe ) {
		return '[' . self::RECIPE_SHORTCODE . ' id="' . $recipe->get_id() . '"]';
	}

	/**
	 * Process ingredients to annotate <li> with structured data.
	 *
	 * @param string $ingredients Existing ingredients string.
	 * @return string
	 */
	public static function process_ingredients_annotate_with_spans( $ingredients ) {
		// Prioritize list items.
		$ingredients = preg_replace_callback(
			'#(<li[^\>]*>)(.*)(<\/li>)#U',
			function( $m ) {
				if ( Unit_Amount_Parser::string_has_non_numeric_amount( $m[2] ) ) {
					$m[1] = str_replace( '<li', '<li data-has-non-numeric-amount', $m[1] );
				}
				return $m[1] . Unit_Amount_Parser::annotate_string_with_spans( $m[2] ) . $m[3];
			},
			$ingredients,
			-1,
			$count
		);
		if ( $count ) {
			return $ingredients;
		}
		// Fall back when there weren't list items.
		$bits = explode( PHP_EOL, $ingredients );
		foreach ( $bits as $i => $bit ) {
			if ( Distribution_Metadata::get_heading( $bit ) ) {
				continue;
			}
			$start = '';
			$end   = '';
			// String starts with some HTML element.
			if ( preg_match( '#^(<p[^>]*>)(.+)(</p>)$#', $bit, $matches ) ) {
				$start = $matches[1];
				$end   = $matches[3];
				$bit   = $matches[2];
			}
			if ( Unit_Amount_Parser::string_has_non_numeric_amount( $bit ) ) {
				$bits[ $i ] = $start . '<span data-has-non-numeric-amount>' . $bit . '</span>' . $end;
				continue;
			}
			$bits[ $i ] = $start . Unit_Amount_Parser::annotate_string_with_spans( $bit ) . $end;
		}
		return implode( PHP_EOL, $bits );
	}


	/**
	 * Do the caption shortcode when applying content filters.
	 *
	 * @param string $content Content to render.
	 * @return string
	 */
	private static function do_caption_shortcode( $content ) {
		global $shortcode_tags;
		if ( ! has_shortcode( $content, 'caption' )
			&& ! isset( $shortcode_tags['caption'] ) ) {
			return $content;
		}
		$backup_tags    = $shortcode_tags;
		$shortcode_tags = array(
			'caption' => $backup_tags['caption'],
		);
		$content        = do_shortcode( $content );
		$shortcode_tags = $backup_tags;
		return $content;
	}

}
