<?php
/**
 * Base converter class.
 *
 * @package Tasty_Recipes
 */

namespace Tasty_Recipes\Converters;

use Tasty_Recipes\Block_Editor;
use Tasty_Recipes\Objects\Recipe;
use Tasty_Recipes\Ratings;
use Tasty_Recipes\Shortcodes;

/**
 * Base converter class.
 */
abstract class Converter {

	/**
	 * Matching string for existing recipes.
	 *
	 * @var string
	 */
	protected static $match_string = null;

	/**
	 * Get the total number of posts with recipes in their content.
	 *
	 * @return int
	 */
	public static function get_count() {
		global $wpdb;
		if ( is_null( static::$match_string ) ) {
			return 0;
		}
		$query = self::get_select_query( 'COUNT(ID)' );
		// @codingStandardsIgnoreStart
		return $wpdb->get_var( $query );
		// @codingStandardsIgnoreEnd
	}

	/**
	 * Get post ids for posts with recipes in their content.
	 *
	 * @param integer $per_page Number of posts to fetch. Defaults to 10.
	 * @return array
	 */
	public static function get_post_ids( $per_page = 10 ) {
		global $wpdb;
		if ( is_null( static::$match_string ) ) {
			return array();
		}
		$query = self::get_select_query( 'ID' ) . ' LIMIT 0,' . (int) $per_page;
		// @codingStandardsIgnoreStart
		$post_ids = $wpdb->get_col( $query );
		// @codingStandardsIgnoreEnd
		return array_map( 'intval', $post_ids );
	}

	/**
	 * Get a SELECT query for posts matching the $match_string.
	 *
	 * @param string $select_var Selection to perform.
	 * @return string
	 */
	protected static function get_select_query( $select_var ) {
		global $wpdb;

		$match_strings = is_array( static::$match_string ) ? static::$match_string : array( static::$match_string );
		$like_query    = array();
		foreach ( $match_strings as $match_string ) {
			$like_query[] = $wpdb->prepare( 'post_content LIKE %s', '%' . $match_string . '%' );
		}
		$like_query = implode( ' OR ', $like_query );
		if ( count( $match_strings ) > 1 ) {
			$like_query = ' ( ' . $like_query . ' ) ';
		}
		return "SELECT {$select_var} FROM {$wpdb->posts} WHERE post_type != 'revision' AND {$like_query}";
	}

	/**
	 * Convert the recipe content within a given post.
	 *
	 * @param integer $post_id ID for the post with the recipe.
	 * @param string  $type    Whether to create a shortcode or a block.
	 * @return Recipe|false
	 */
	public static function convert_post( $post_id, $type = 'shortcode' ) {
		global $wpdb;

		if ( ! in_array( $type, array( 'shortcode', 'block' ), true ) ) {
			return false;
		}

		$content = $wpdb->get_var( $wpdb->prepare( "SELECT post_content FROM $wpdb->posts WHERE ID=%d", $post_id ) );
		// Correct Windows-style line endings.
		$content  = str_replace( "\r\n", "\n", $content );
		$content  = str_replace( "\r", "\n", $content );
		$existing = static::get_existing_to_convert( $content );
		if ( ! $existing ) {
			return false;
		}
		$recipe = static::create_recipe_from_existing( $existing, $post_id );
		if ( ! $recipe ) {
			return false;
		}
		Ratings::update_recipe_rating( $recipe, $post_id );
		if ( 'shortcode' === $type ) {
			$shortcode = PHP_EOL . Shortcodes::get_shortcode_for_recipe( $recipe ) . PHP_EOL;
			$content   = str_replace( $existing, $shortcode, $content );
		} elseif ( 'block' === $type ) {
			$block   = PHP_EOL . Block_Editor::get_block_for_recipe( $recipe ) . PHP_EOL;
			$content = str_replace( $existing, $block, $content );
		}
		wp_update_post(
			array(
				'ID'           => $post_id,
				'post_content' => $content,
			)
		);
		return $recipe;
	}

	/**
	 * Convert recipe content to Tasty Recipes format
	 *
	 * @param string  $existing Existing content that may have recipe content.
	 * @param integer $post_id  ID for the post with the recipe.
	 * @return Recipe
	 */
	public static function create_recipe_from_existing( $existing, $post_id ) {
		$existing = $existing;
		$post_id  = $post_id;
		return false;
	}

	/**
	 * Apply a filter to converted data before saving to the database.
	 *
	 * @param array  $converted_data Data to be converted.
	 * @param Recipe $recipe         Created recipe object.
	 * @return Recipe
	 */
	protected static function save_converted_data_to_recipe( $converted_data, $recipe ) {
		/**
		 * Filter converted data before it's applied to the recipe.
		 *
		 * @param array $converted_data
		 * @param string $type
		 */
		$called_class   = get_called_class();
		$bits           = explode( '\\', $called_class );
		$type           = array_pop( $bits );
		$type           = strtolower( $type );
		$converted_data = apply_filters( 'tasty_recipes_convert_recipe', $converted_data, $type );
		foreach ( $converted_data as $key => $value ) {
			$setter = "set_{$key}";
			$recipe->$setter( $value );
		}
		return $recipe;
	}

	/**
	 * Transform lines into a list, and ! section headings into <h4>.
	 *
	 * Used by ZipList and Meal Planner Pro.
	 *
	 * @param string $content    Content to be transformed.
	 * @param string $list_style Either an ordered or unordered list.
	 * @return string
	 */
	protected static function process_lines_into_lists_and_headings( $content, $list_style ) {
		$list_opening = '<' . $list_style . '>';
		$list_closing = '</' . $list_style . '>';
		$is_open      = false;
		$bits         = explode( PHP_EOL, $content );
		$new_bits     = array();
		foreach ( $bits as $i => $line ) {
			$pre  = '';
			$line = trim( $line );
			if ( 0 === stripos( $line, '!' ) ) {
				if ( $is_open ) {
					$pre     = $list_closing . PHP_EOL;
					$is_open = false;
				}
				$line       = substr( $line, 1 );
				$new_bits[] = $pre . '<h4>' . static::process_markdownish_into_html( $line ) . '</h4>';
			} elseif ( ! empty( $line ) ) {
				if ( ! $is_open ) {
					$pre     = $list_opening . PHP_EOL;
					$is_open = true;
				}
				$new_bits[] = $pre . '<li>' . static::process_markdownish_into_html( $line ) . '</li>';
			}
		}
		$content = implode( PHP_EOL, $new_bits );
		if ( $is_open ) {
			$content .= PHP_EOL . $list_closing;
		}
		return $content;
	}

	/**
	 * Process MPP, Yummly, and ZipList markdown-ish markup into HTML.
	 *
	 * @param string $item Content to be transformed.
	 * @return string
	 */
	protected static function process_markdownish_into_html( $item ) {
		$output   = $item;
		$link_ptr = '#\[(.*?)\| *(.*?)( (.*?))?\]#';
		preg_match_all( $link_ptr, $item, $matches );
		if ( isset( $matches[0] ) ) {
			$orig         = $matches[0];
			$substitution = preg_replace( $link_ptr, '<a href="$2"$3>$1</a>', str_replace( '"', '', $orig ) );
			$output       = str_replace( $orig, $substitution, $item );
		}

		// Must be an image.
		if ( '%http' === substr( $output, 0, 5 ) ) {
			$output = '<img src="' . esc_url( substr( $output, 1 ) ) . '">';
		}
		$output = preg_replace( '/(^|\s)\*([^\s\*][^\*]*[^\s\*]|[^\s\*])\*(\W|$)/', '$1<strong>$2</strong>$3', $output );
		$output = preg_replace( '/(^|\s)_([^\s_][^_]*[^\s_]|[^\s_])_(\W|$)/', '$1<em>$2</em>$3', $output );
		return $output;
	}

	/**
	 * Get the image id for a given media file.
	 *
	 * If the image doesn't exist in the media library, it's imported.
	 *
	 * @param string $file Path for the file.
	 * @return integer
	 */
	protected static function get_image_id_from_file( $file ) {
		global $wpdb;

		$image_id = $wpdb->get_var( $wpdb->prepare( "SELECT ID FROM $wpdb->posts WHERE post_type='attachment' and guid=%s", $file ) );
		if ( ! $image_id ) {
			$image_id = static::media_sideload_image( $file );
		}
		return $image_id;
	}

	/**
	 * Sideload an image URL into WordPress
	 *
	 * @param string $file Path for the file.
	 * @return integer
	 */
	public static function media_sideload_image( $file ) {
		/**
		 * Fires before an image is imported into WordPress.
		 *
		 * Can be used to prevent images from being imported, or to apply
		 * a custom algorithm to matching images.
		 *
		 * @param null   $retval Return value.
		 * @param string $file   Image file to import.
		 */
		$retval = apply_filters( 'tasty_recipes_pre_import_image', null, $file );
		if ( null !== $retval ) {
			return $retval;
		}

		preg_match( '/[^\?]+\.(jpe?g|jpe|gif|png)\b/i', $file, $matches );
		if ( ! $matches ) {
			do_action( 'tasty_recipes_imported_image', 0 );
			return 0;
		}

		$file_array             = array();
		$file_array['name']     = basename( $matches[0] );
		$file_array['tmp_name'] = download_url( $file );
		if ( is_wp_error( $file_array['tmp_name'] ) ) {
			do_action( 'tasty_recipes_imported_image', 0 );
			return 0;
		}

		$id = media_handle_sideload( $file_array, 0 );
		if ( is_wp_error( $id ) ) {
			unlink( $file_array['tmp_name'] );
			do_action( 'tasty_recipes_imported_image', 0 );
			return 0;
		}
		do_action( 'tasty_recipes_imported_image', $id );
		return $id;
	}

	/**
	 * Format numerical times (eg. 00:10).
	 *
	 * @param string $value Time string to format.
	 * @return string
	 */
	public static function format_times( $value ) {
		$value     = preg_replace( '/(\d+):(\d+)/', '$1 hours $2 minutes', $value );
		$incorrect = array(
			'/00 hours /',
			'/ 00 minutes/',
			'/01 hours/',
			'/1 hours/',
			'/0(\d)/',
		);
		$correct   = array(
			'',
			'',
			'1 hour',
			'1 hour',
			'$1',
		);
		$value     = preg_replace( $incorrect, $correct, $value );
		return $value;
	}

}
