<?php
/**
 * Plugin Name:     Tasty Recipes
 * Plugin URI:      https://www.wptasty.com/tasty-recipes
 * Description:     Tasty Recipes is the easiest way to publish recipes on your WordPress blog.
 * Author:          WP Tasty
 * Author URI:      https://www.wptasty.com
 * Text Domain:     tasty-recipes
 * Domain Path:     /languages
 * Version:         2.5.2
 *
 * @package         Tasty_Recipes
 */

define( 'TASTY_RECIPES_PLUGIN_VERSION', '2.5.2' );
define( 'TASTY_RECIPES_PLUGIN_FILE', __FILE__ );

/**
 * Base controller class for the plugin.
 */
class Tasty_Recipes {

	/**
	 * Store of recipe JSON data for current view.
	 *
	 * Used to share state between Shortcode and Admin classes.
	 *
	 * @var array
	 */
	public $recipe_json = array();

	/**
	 * Singleton instance for this class.
	 *
	 * @var object
	 */
	private static $instance;

	/**
	 * Option name for the default author link.
	 *
	 * @var string
	 */
	const DEFAULT_AUTHOR_LINK_OPTION = 'tasty_recipes_default_author_link';

	/**
	 * Option name for the Instagram handle.
	 *
	 * @var string
	 */
	const INSTAGRAM_HANDLE_OPTION = 'tasty_recipes_instagram_handle';

	/**
	 * Option name for the Instagram tag.
	 *
	 * @var string
	 */
	const INSTAGRAM_HASHTAG_OPTION = 'tasty_recipes_instagram_tag';

	/**
	 * Option name for the license key.
	 *
	 * @var string
	 */
	const LICENSE_KEY_OPTION = 'tasty_recipes_license_key';

	/**
	 * Option name for the ShareASale affiliate ID.
	 *
	 * @var string
	 */
	const SHAREASALE_OPTION = 'tasty_recipes_shareasale';

	/**
	 * Option name for plugin activation state.
	 *
	 * @var string
	 */
	const PLUGIN_ACTIVATION_OPTION = 'tasty_recipes_do_activation_redirect';

	/**
	 * Option name for the template.
	 *
	 * @var string
	 */
	const TEMPLATE_OPTION = 'tasty_recipes_template';

	/**
	 * Option name for the quick links.
	 *
	 * @var string
	 */
	const QUICK_LINKS_OPTION = 'tasty_recipes_quick_links';

	/**
	 * Instantiates and gets the singleton instance for the class.
	 *
	 * @return object
	 */
	public static function get_instance() {
		if ( ! isset( self::$instance ) ) {
			self::$instance = new Tasty_Recipes;
			self::$instance->require_files();
			self::$instance->setup_actions();
			self::$instance->setup_filters();
		}
		return self::$instance;
	}

	/**
	 * Loads required plugin files and registers autoloader.
	 */
	private static function require_files() {

		require_once dirname( __FILE__ ) . '/functions.php';

		load_plugin_textdomain( 'tasty-recipes', false, basename( dirname( __FILE__ ) ) . '/languages' );

		/**
		 * Register the class autoloader
		 */
		spl_autoload_register(
			function( $class ) {
					$class = ltrim( $class, '\\' );
				if ( 0 !== stripos( $class, 'Tasty_Recipes\\' ) ) {
					return;
				}

				$parts = explode( '\\', $class );
				array_shift( $parts ); // Don't need "Tasty_Recipes".
				$last    = array_pop( $parts ); // File should be 'class-[...].php'.
				$last    = 'class-' . $last . '.php';
				$parts[] = $last;
				$file    = dirname( __FILE__ ) . '/inc/' . str_replace( '_', '-', strtolower( implode( '/', $parts ) ) );
				if ( file_exists( $file ) ) {
					require $file;
				}

					// Might be a trait.
				$file = str_replace( '/class-', '/trait-', $file );
				if ( file_exists( $file ) ) {
					require $file;
				}
			}
		);
	}

	/**
	 * Registry of actions used in the plugin.
	 */
	private function setup_actions() {
		// Bootstrap.
		add_action( 'init', array( 'Tasty_Recipes\Block_Editor', 'action_init_register' ) );
		add_action( 'init', array( 'Tasty_Recipes\Content_Model', 'action_init_register_cron_events' ) );
		add_action( 'tasty_recipes_process_thumbnails', array( 'Tasty_Recipes\Content_Model', 'action_tasty_recipes_process_thumbnails' ) );
		add_action( 'tasty_recipes_enrich_youtube_embeds', array( 'Tasty_Recipes\Content_Model', 'action_tasty_recipes_enrich_youtube_embeds' ) );
		add_action( 'init', array( 'Tasty_Recipes\Content_Model', 'action_init_register_post_types' ) );
		add_action( 'init', array( 'Tasty_Recipes\Content_Model', 'action_init_register_rewrite_rules' ) );
		add_action( 'init', array( 'Tasty_Recipes\Content_Model', 'action_init_register_oembed_providers' ) );
		add_action( 'init', array( 'Tasty_Recipes\Shortcodes', 'action_init_register_shortcode' ) );
		// Frontend.
		add_action( 'wp_print_styles', array( 'Tasty_Recipes\Assets', 'action_wp_print_styles' ) );
		add_action( 'body_class', array( 'Tasty_Recipes\Content_Model', 'filter_body_class' ) );
		add_action( 'wp_head', array( 'Tasty_Recipes\Distribution_Metadata', 'action_wp_head_google_schema' ) );
		add_action( 'wp_head', array( 'Tasty_Recipes\Distribution_Metadata', 'action_wp_head_noindex' ) );
		add_action( 'wp_head', array( 'Tasty_Recipes\Ratings', 'action_wp_head' ) );
		add_filter( 'wpseo_schema_graph_pieces', array( 'Tasty_Recipes\Distribution_Metadata', 'filter_wpseo_schema_graph_pieces' ), 10, 2 );
		foreach ( array( 'wp_insert_comment', 'wp_update_comment', 'wp_set_comment_status' ) as $hook ) {
			add_action( $hook, array( 'Tasty_Recipes\Ratings', 'action_modify_comment_update_recipe_ratings' ) );
		}
		// Admin.
		add_action( 'admin_init', array( 'Tasty_Recipes\Admin', 'action_admin_init' ) );
		add_action( 'http_request_args', array( 'Tasty_Recipes\Admin', 'filter_http_request_args' ), 10, 2 );
		add_action( 'admin_menu', array( 'Tasty_Recipes\Admin', 'action_admin_menu' ) );
		add_action( 'admin_notices', array( 'Tasty_Recipes\Admin', 'action_admin_notices_license_key' ) );
		add_action( 'admin_enqueue_scripts', array( 'Tasty_Recipes\Assets', 'action_admin_enqueue_scripts' ) );
		add_action( 'wp_enqueue_editor', array( 'Tasty_Recipes\Assets', 'action_wp_enqueue_editor' ) );
		add_action( 'enqueue_block_editor_assets', array( 'Tasty_Recipes\Assets', 'action_enqueue_block_editor_assets' ) );
		add_action( 'admin_notices', array( 'Tasty_Recipes\Editor', 'action_admin_notices' ) );
		add_action( 'after_wp_tiny_mce', array( 'Tasty_Recipes\Editor', 'action_after_wp_tiny_mce' ) );
		add_action( 'add_option_' . self::LICENSE_KEY_OPTION, array( 'Tasty_Recipes\Admin', 'action_update_option_register_license' ) );
		add_action( 'update_option_' . self::LICENSE_KEY_OPTION, array( 'Tasty_Recipes\Admin', 'action_update_option_register_license' ) );
		add_action( 'update_option_' . self::LICENSE_KEY_OPTION, array( 'Tasty_Recipes\Admin', 'action_update_option_clear_transient' ) );
		add_action( 'media_buttons', array( 'Tasty_Recipes\Editor', 'action_media_buttons' ) );
		add_action( 'wp_ajax_tasty_recipes_remove_license_key', array( 'Tasty_Recipes\Admin', 'handle_wp_ajax_remove_license_key' ) );
		add_action( 'wp_ajax_tasty_recipes_get_count', array( 'Tasty_Recipes\Admin', 'handle_wp_ajax_get_count' ) );
		add_action( 'wp_ajax_tasty_recipes_convert', array( 'Tasty_Recipes\Admin', 'handle_wp_ajax_convert' ) );
		add_action( 'wp_ajax_tasty_recipes_convert_recipe', array( 'Tasty_Recipes\Editor', 'handle_wp_ajax_convert_recipe' ) );
		add_action( 'wp_ajax_tasty_recipes_parse_shortcode', array( 'Tasty_Recipes\Editor', 'handle_wp_ajax_parse_shortcode' ) );
		add_action( 'wp_ajax_tasty_recipes_modify_recipe', array( 'Tasty_Recipes\Editor', 'handle_wp_ajax_modify_recipe' ) );
	}

	/**
	 * Registry of filters used in the plugin.
	 */
	private function setup_filters() {

		// Bootstrap.
		add_filter( 'rewrite_rules_array', array( 'Tasty_Recipes\Content_Model', 'filter_rewrite_rules_array' ) );

		add_filter( 'the_content', array( 'Tasty_Recipes\Shortcodes', 'filter_the_content_late' ), 100 );

		// WordPress' standard text formatting filters.
		add_filter( 'tasty_recipes_the_title', 'wptexturize' );
		add_filter( 'tasty_recipes_the_title', 'convert_chars' );
		add_filter( 'tasty_recipes_the_title', 'trim' );
		add_filter( 'tasty_recipes_the_content', 'wptexturize' );
		add_filter( 'tasty_recipes_the_content', 'convert_smilies', 20 );
		add_filter( 'tasty_recipes_the_content', 'wpautop' );
		add_filter( 'tasty_recipes_the_content', 'shortcode_unautop' );
		add_filter( 'tasty_recipes_the_content', 'prepend_attachment' );
		// Only available >= WordPress 4.4.
		if ( function_exists( 'wp_make_content_images_responsive' ) ) {
			add_filter( 'tasty_recipes_the_content', 'wp_make_content_images_responsive' );
		}

		// Plugin-specific filters.
		add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), array( 'Tasty_Recipes\Admin', 'filter_plugin_action_links' ) );
		add_filter( 'teeny_mce_buttons', array( 'Tasty_Recipes\Assets', 'filter_teeny_mce_buttons' ), 10, 2 );
		add_filter( 'teeny_mce_before_init', array( 'Tasty_Recipes\Assets', 'filter_teeny_mce_before_init' ), 10, 2 );
		add_filter( 'update_post_metadata', array( 'Tasty_Recipes\Content_Model', 'filter_update_post_metadata_nutrifox_id' ), 10, 4 );
		add_filter( 'update_post_metadata', array( 'Tasty_Recipes\Content_Model', 'filter_update_post_metadata_video_url' ), 10, 4 );
		add_filter( 'update_post_metadata', array( 'Tasty_Recipes\Content_Model', 'filter_update_post_metadata_thumbnail_id' ), 10, 4 );
		add_filter( 'template_include', array( 'Tasty_Recipes\Content_Model', 'filter_template_include' ), 1000 );
		add_filter( 'preprocess_comment', array( 'Tasty_Recipes\Ratings', 'filter_preprocess_comment' ) );
		add_filter( 'comment_form_field_comment', array( 'Tasty_Recipes\Ratings', 'filter_comment_form_field_comment' ) );
		add_filter( 'comment_text', array( 'Tasty_Recipes\Ratings', 'filter_comment_text' ), 10, 2 );
	}

	/**
	 * Actions to perform when activating the plugin.
	 */
	public static function plugin_activation() {
		self::require_files();
		\Tasty_Recipes\Content_Model::action_init_register_rewrite_rules();
		flush_rewrite_rules();
		update_option( self::PLUGIN_ACTIVATION_OPTION, true );
	}

	/**
	 * Determine whether there's a recipe present in the post.
	 *
	 * @param integer $post_id ID for the post to inspect.
	 * @return boolean
	 */
	public static function has_recipe( $post_id ) {
		$post = get_post( $post_id );
		if ( ! $post ) {
			return false;
		}
		if ( false !== stripos( $post->post_content, '[' . Tasty_Recipes\Shortcodes::RECIPE_SHORTCODE ) ) {
			return true;
		}
		if ( false !== stripos( $post->post_content, '<!-- wp:' . Tasty_Recipes\Block_Editor::RECIPE_BLOCK_TYPE . ' ' ) ) {
			return true;
		}
		return false;
	}

	/**
	 * Get the recipe ids embedded within a given post.
	 *
	 * @param integer $post_id ID for the post to parse.
	 * @param array   $options Any options to configure the bheavior.
	 * @return array
	 */
	public static function get_recipe_ids_for_post( $post_id, $options = array() ) {
		$post = get_post( $post_id );
		if ( ! $post_id || ! $post ) {
			return array();
		}
		return self::get_recipe_ids_from_content( $post->post_content, $options );
	}

	/**
	 * Get the recipe ids embedded within a given string.
	 *
	 * @param string $content Content to search for recipe ids.
	 * @param array  $options Configure return value behavior.
	 * @return array
	 */
	public static function get_recipe_ids_from_content( $content, $options = array() ) {

		$defaults = array(
			'disable-json-ld' => null,
			'full-result'     => false,
		);
		$options  = array_merge( $defaults, $options );

		$recipes = array();
		if ( preg_match_all( '/' . get_shortcode_regex() . '/s', $content, $matches ) ) {
			foreach ( $matches[2] as $i => $shortcode ) {
				if ( Tasty_Recipes\Shortcodes::RECIPE_SHORTCODE !== $shortcode ) {
					continue;
				}
				$atts = shortcode_parse_atts( $matches[3][ $i ] );
				if ( empty( $atts['id'] ) ) {
					continue;
				}

				if ( false === $options['disable-json-ld']
					&& in_array( 'disable-json-ld', $atts, true ) ) {
					continue;
				}

				if ( ! empty( $options['full-result'] ) ) {
					$recipes[] = $atts;
				} else {
					$recipes[] = (int) $atts['id'];
				}
			}
		}
		if ( function_exists( 'parse_blocks' ) ) {
			foreach ( parse_blocks( $content ) as $block ) {
				if ( is_object( $block ) && ! empty( $block->blockName )
					&& Tasty_Recipes\Block_Editor::RECIPE_BLOCK_TYPE === $block->blockName ) {
					if ( ! empty( $block->attrs->id ) && ( ! isset( $block->attrs->disableJSON ) || false === (bool) $block->attrs->disableJSON ) ) {
						if ( ! empty( $options['full-result'] ) ) {
							$recipes[] = (array) $block->attrs;
						} else {
							$recipes[] = (int) $block->attrs->id;
						}
					}
				}
				if ( is_array( $block ) && ! empty( $block['blockName'] )
					&& Tasty_Recipes\Block_Editor::RECIPE_BLOCK_TYPE === $block['blockName'] ) {
					if ( ! empty( $block['attrs']['id'] ) && ( ! isset( $block['attrs']['disableJSON'] ) || false === (bool) $block['attrs']['disableJSON'] ) ) {
						if ( ! empty( $options['full-result'] ) ) {
							$recipes[] = (array) $block['attrs'];
						} else {
							$recipes[] = (int) $block['attrs']['id'];
						}
					}
				}
			}
		}
		return $recipes;
	}

	/**
	 * Get a dictionary of converters supported by Tasty Recipes.
	 *
	 * @return array
	 */
	public static function get_converters() {
		return array(
			'cookbook'       => array(
				'class' => 'Tasty_Recipes\Converters\Cookbook',
				'label' => 'Cookbook',
			),
			'easyrecipe'     => array(
				'class' => 'Tasty_Recipes\Converters\EasyRecipe',
				'label' => 'EasyRecipe',
			),
			'mealplannerpro' => array(
				'class' => 'Tasty_Recipes\Converters\MealPlannerPro',
				'label' => 'Meal Planner Pro',
			),
			'srp'            => array(
				'class' => 'Tasty_Recipes\Converters\Simple_Recipe_Pro',
				'label' => 'Simple Recipe Pro',
			),
			'wpcom'          => array(
				'class' => 'Tasty_Recipes\Converters\WPCom',
				'label' => 'WordPress.com',
			),
			'wprm'           => array(
				'class' => 'Tasty_Recipes\Converters\WP_Recipe_Maker',
				'label' => 'WP Recipe Maker',
			),
			'wpur'           => array(
				'class' => 'Tasty_Recipes\Converters\WP_Ultimate_Recipe',
				'label' => 'WP Ultimate Recipe',
			),
			'yummly'         => array(
				'class' => 'Tasty_Recipes\Converters\Yummly',
				'label' => 'Yummly',
			),
			'yumprint'       => array(
				'class' => 'Tasty_Recipes\Converters\YumPrint',
				'label' => 'YumPrint Recipe Card',
			),
			'ziplist'        => array(
				'class' => 'Tasty_Recipes\Converters\ZipList',
				'label' => 'ZipList (or Zip Recipes)',
			),
		);
	}

	/**
	 * Get the recipes embedded within a given post.
	 *
	 * @param integer $post_id ID for the post to inspect.
	 * @param array   $options Any options to pass through to the parser.
	 * @return array
	 */
	public static function get_recipes_for_post( $post_id, $options = array() ) {
		$recipes = array();
		foreach ( self::get_recipe_ids_for_post( $post_id, $options ) as $id ) {
			$recipe = Tasty_Recipes\Objects\Recipe::get_by_id( $id );
			if ( $recipe ) {
				$recipes[] = $recipe;
			}
		}
		return $recipes;
	}

	/**
	 * Get a fully-qualified path to a template.
	 *
	 * @param string $template Template name.
	 * @return string
	 */
	public static function get_template_path( $template ) {
		$full_path = dirname( __FILE__ ) . '/templates/' . $template . '.php';
		return apply_filters( 'tasty_recipes_template_path', $full_path, $template );
	}

	/**
	 * Get a rendered template.
	 *
	 * @param string $template Fully-qualified template path.
	 * @param array  $vars     Variables to pass into the template.
	 * @return string
	 */
	public static function get_template_part( $template, $vars = array() ) {
		$full_path = self::get_template_path( $template );
		// Provided template may already be a full path.
		if ( ! file_exists( $full_path ) ) {
			$full_path = $template;
		}
		if ( ! file_exists( $full_path ) ) {
			return '';
		}

		ob_start();
		// @codingStandardsIgnoreStart
		if ( ! empty( $vars ) ) {
			extract( $vars );
		}
		// @codingStandardsIgnoreEnd
		include $full_path;
		return ob_get_clean();
	}


}

/**
 * Access the Tasty Recipes instance.
 *
 * @return Tasty_Recipes
 */
// @codingStandardsIgnoreStart
function Tasty_Recipes() {
// @codingStandardsIgnoreEnd
	return Tasty_Recipes::get_instance();
}
add_action( 'plugins_loaded', 'Tasty_Recipes' );

/**
 * Register the plugin activation hook
 */
register_activation_hook( __FILE__, array( 'Tasty_Recipes', 'plugin_activation' ) );
