<?php
/**
 * External Links Controller for WordPress. Exclusively on Envato Market: http://codecanyon.net/user/42theme/portfolio?ref=42Theme
 * @encoding     UTF-8
 * @version      1.0.3
 * @copyright    Copyright (C) 2020 42Theme ( https://42theme.com ). All rights reserved.
 * @license      Envato License https://codecanyon.net/licenses/standard?ref=42theme
 * @author       Alexander Khmelnitskiy ( ak@42theme.com )
 * @support      support@42theme.com
 **/

namespace T42\Junction;

/**
 * SINGLETON: Class used to implement Activation tab on plugin settings page.
 *
 * @since 1.0.0
 * @author Alexandr Khmelnytsky (info@alexander.khmelnitskiy.ua)
 */
final class PluginActivation {

	/**
	 * The one true PluginActivation.
	 *
	 * @var PluginActivation
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new PluginActivation instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		add_action( 'update_option_envato_purchase_code_' . EnvatoItem::get_instance()->get_id(), [$this, 'reset_temporary_activation'], 10 );

	}

	/**
	 * Reset temporary activation on every pid change.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function reset_temporary_activation() {

		delete_transient ( 'activated_' . EnvatoItem::get_instance()->get_id() );

		/** Reset updates cache. */
		PluginUpdater::get_instance()->reset_cache();

	}

	/**
	 * Return Activation Status.
	 *
	 * @param bool|string $pid - Purchase Code. Optional.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return boolean True if activated.
	 **/
	public function is_activated( $pid = false ) {

	    if ( $pid !== false ) {
		    $purchase_code = $pid;
        } else {

		    /** Get fresh PID from form. */
		    if ( isset( $_POST['envato_purchase_code_' . EnvatoItem::get_instance()->get_id()] ) ) {
			    $purchase_code = filter_input( INPUT_POST, 'envato_purchase_code_' . EnvatoItem::get_instance()->get_id() );

			    /** Or get PID from option. */
		    } else {
			    $purchase_code = get_option( 'envato_purchase_code_' . EnvatoItem::get_instance()->get_id() );
		    }

        }

		/** If we do not have $purchase_code then nothing to check. */
		if ( ! $purchase_code ) {
			return false; // No key. Not activated.
        }

		/** Clean and validate purchase code. */
		/** Remove spaces. */
		$purchase_code = trim( $purchase_code );

		/** Make sure the code is valid before sending it to Envato. */
		if ( ! preg_match( "/^(\w{8})-((\w{4})-){3}(\w{12})$/", $purchase_code ) ) {
			return false; // Wrong key format. Not activated.
		}

	    /** Check temporary activation */
        $local_activation = $this->local_validation(); // 0 - if no local value, go to download.

		if ( $local_activation === 0 ) {

		    /** Need Remote validation. */
			$remote_activation = $this->remote_validation( $purchase_code );

			if ( $remote_activation ) {

                $this->temporary_activation( $remote_activation );
                return filter_var( $remote_activation, FILTER_VALIDATE_BOOLEAN );

            } else {

				/** Not activated. */
				$this->temporary_activation( false );
                return false;
            }

        } else {

			/** Use local activation. */
			$this->temporary_activation( $local_activation );
			return filter_var( $local_activation, FILTER_VALIDATE_BOOLEAN );

        }

	}

	/**
	 * Set temporary activation.
	 *
	 * @param bool $activate - Temporary Activate/Deactivate.
	 * @param int $hours - Timeout for temporary activation.
	 *
	 * @since 1.0.0
	 * @access public
	 */
	public function temporary_activation( $activate, $hours = 12 ) {

	    if ( filter_var( $activate, FILTER_VALIDATE_BOOLEAN ) ) {
		    set_transient( 'activated_' . EnvatoItem::get_instance()->get_id(), '1', $hours * HOUR_IN_SECONDS );
        } else {
		    set_transient( 'activated_' . EnvatoItem::get_instance()->get_id(), '0', $hours * HOUR_IN_SECONDS );
        }

    }

	/**
	 * Validate PID on local server.
	 *
	 * @return int|true|false - status of temporary activation status.
     *         0 - if we dont have local activation status and need remove validation.
	 * @since 1.0.0
	 * @access public
	 */
	public function local_validation() {

		/** Get temporary activation status. */
		$tmp_activation_status = get_transient( 'activated_' . EnvatoItem::get_instance()->get_id() );

		/** Send query to server if we dont have temporary option. */
		if ( false === $tmp_activation_status ) {
			return 0;
        } else {
			return filter_var( $tmp_activation_status, FILTER_VALIDATE_BOOLEAN );
        }

    }

	/**
	 * Validate PID on our server.
	 *
	 * @param $purchase_code - Envato Purchase Code.
	 *
	 * @return array|mixed|object
	 * @since 1.0.0
	 * @access public
	 */
	public function remote_validation( $purchase_code ) {

		$curl = curl_init();

		/** Prepare URL. */
		$url = 'https://updates.42theme.com/wp-content/plugins/t42-purchase-validator/src/Validate.php?';
		$url .= 'action=validate&'; // Action.
		$url .= 'plugin=t42-junction&'; // Plugin Name.
		$url .= 'domain=' . parse_url( site_url(), PHP_URL_HOST ) . '&'; // Domain Name.
		$url .= 'version=1.0.3&'; // Plugin version.
		$url .= 'pid=' . $purchase_code . '&'; // Purchase Code.
		$url .= 'admin_e=' . base64_encode( get_option( 'admin_email' ) );

		curl_setopt( $curl, CURLOPT_URL, $url );
		curl_setopt( $curl, CURLOPT_RETURNTRANSFER, true );
		curl_setopt( $curl, CURLOPT_HEADER, false );
		$json = curl_exec( $curl );

		/**
		 * Handle connection errors.
		 * Show users an appropriate message asking to try again later.
		 **/
		if ( curl_errno( $curl ) > 0 ) {
			echo esc_html( 'Error connecting to: ' . $url . PHP_EOL . 'Please check your security plugins and add this url to white list.' );
			return false;
		}

		/**
		 * If we reach this point, we have a proper response.
		 * Get the response code to check if the content was found.
		 **/
		$responseCode = curl_getinfo( $curl, CURLINFO_HTTP_CODE );

		/**
		 * Anything other than HTTP 200 indicates a request error.
		 * In this case, we again ask the user to try again later.
		 **/
		if ( $responseCode !== 200 ) {
			echo esc_html('Failed to get content due to an error: HTTP ' . $responseCode . PHP_EOL . 'URL: ' . $url );
			return false;
		}

		curl_close( $curl );

		$res = json_decode( $json, true );

		if ( true === $res ) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * Render Purchase Code field.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_pid() {

		/** Get PID. */
		$purchase_code = Settings::get_instance()->options['envato_purchase_code_' . EnvatoItem::get_instance()->get_id()];

        ?>
        <div class="t42-activation-box">

            <?php
            /** Render input. */
            SettingsUI::get_instance()->render_activate_input(
                $purchase_code,
                '',
                esc_html__( 'Enter your CodeCanyon purchase code. Allowed only one Purchase Code per website.', 't42-junction' ),
                [
                    'name' => 't42_junction_settings[envato_purchase_code_' . EnvatoItem::get_instance()->get_id() . ']',
                    'id' => 't42_envato_purchase_code',
                    'placeholder' => 'Purchase Code'
                ]
            );
            ?>

        </div>

		<?php
	}

	/**
	 * Render FAQ block.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_FAQ() {
	    ?>
        <div class="t42-activation-faq">

            <div class="t42-accordion" data-t42-accordion="showfirst: false">

                <h3><?php esc_html_e( 'Activation FAQ\'S', 't42-junction' ); ?></h3>

                <div class="t42-accordion-title">
                    <i class="t42-icon-question-circle-o"></i>
                    <span class="t42-list-item__text"><?php esc_html_e( 'Where is my Purchase Code?', 't42-junction' ); ?></span>
                </div>
                <div class="t42-accordion-content">
                    <p><?php esc_html_e( 'The purchase code is a unique combination of characters that confirms that you bought the plugin. You can find your purchase code in ', 't42-junction' ); ?>
                        <a href="https://1.envato.market/cc-downloads" target="_blank"><?php esc_html_e( 'your account', 't42-junction' );?></a>
                        <?php esc_html_e( ' on the CodeCanyon. Learn more about ', 't42-junction' ); ?>
                        <a href="https://help.market.envato.com/hc/en-us/articles/202822600-Where-Is-My-Purchase-Code-" target="_blank"><?php esc_html_e( 'How to find your purchase code', 't42-junction' );?></a>
                        <?php esc_html_e( ' .', 't42-junction');?>
                    </p>
                </div>

                <div class="t42-accordion-title">
                    <i class="t42-icon-question-circle-o"></i>
                    <span class="t42-list-item__text"><?php esc_html_e( 'Can I use one Purchase Code on multiple sites?', 't42-junction' ); ?></span>
                </div>
                <div class="t42-accordion-content">
                    <p>
                        <?php esc_html_e( 'No, this is prohibited by license terms. You can use the purchase code on only one website at a time. Learn more about ', 't42-junction' ); ?>
                        <a href="https://1.envato.market/KYbje" target="_blank"><?php esc_html_e( 'Envato License', 't42-junction' );?></a>
                        <?php esc_html_e( ' terms. ', 't42-junction' ); ?>
                    </p>
                </div>

                <div class="t42-accordion-title">
                    <i class="t42-icon-question-circle-o"></i>
                    <span class="t42-list-item__text"><?php esc_html_e( 'What are the benefits of plugin activation?', 't42-junction' ); ?></span>
                </div>
                <div class="t42-accordion-content">
                    <p>
                        <?php esc_html_e( 'Activation of the plugin allows you to use all the functionality of the plugin on your site. In addition, in some cases, activating the plugin allows you to access additional features and capabilities of the plugin. Also, using an authored version of the plugin, you can be sure that you will not violate the license.', 't42-junction' ); ?>
                    </p>
                </div>

                <div class="t42-accordion-title">
                    <i class="t42-icon-question-circle-o"></i>
                    <span class="t42-list-item__text"><?php esc_html_e( 'What should I do if my Purchase Code does not work?', 't42-junction' ); ?></span>
                </div>
                <div class="t42-accordion-content">
                    <p>
                        <?php esc_html_e( 'There are several reasons why the purchase code may not work on your site. Learn more why your ', 't42-junction' ); ?>
                        <a href="https://help.market.envato.com/hc/en-us/articles/204451834-My-Purchase-Code-is-Not-Working" target="_blank"><?php esc_html_e( 'Purchase Code is Not Working', 't42-junction' );?></a>
                        <?php esc_html_e( ' .', 't42-junction');?>
                    </p>
                </div>

            </div>
        </div>
        <?php
    }

	/**
	 * Main PluginActivation Instance.
	 *
	 * Insures that only one instance of PluginActivation exists in memory at any one time.
	 *
	 * @static
	 * @return PluginActivation
	 * @since 1.0.0
	 **/
	public static function get_instance() {
		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof PluginActivation ) ) {
			self::$instance = new PluginActivation;
		}

		return self::$instance;
	}

} // End Class PluginActivation.
