<?php
/**
 * External Links Controller for WordPress. Exclusively on Envato Market: http://codecanyon.net/user/42theme/portfolio?ref=42Theme
 * @encoding     UTF-8
 * @version      1.0.3
 * @copyright    Copyright (C) 2020 42Theme ( https://42theme.com ). All rights reserved.
 * @license      Envato License https://codecanyon.net/licenses/standard?ref=42theme
 * @author       Alexander Khmelnitskiy ( ak@42theme.com )
 * @support      support@42theme.com
 **/

namespace T42\Junction;

use T42\Junction as Junction;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class used to implement base plugin features.
 *
 * @since 1.0.0
 * @author Alexander Khmelnitskiy ( ak@42theme.com )
 **/
final class PluginHelper {

	/**
	 * The one true PluginHelper.
	 *
	 * @var PluginHelper
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new PluginHelper instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Add plugin links. */
		add_filter( 'plugin_action_links_' . Junction::$basename, [$this, 'add_links'] );

		/** Add plugin meta. */
		add_filter( 'plugin_row_meta', [$this, 'add_row_meta'], 10, 2 );

		/** Load JS and CSS for Backend Area. */
		$this->enqueue_backend();

	}

	/**
	 * Load JS and CSS for Backend Area.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	function enqueue_backend() {

		/** Add admin styles. */
		add_action( 'admin_enqueue_scripts', [ $this, 'add_admin_styles' ] );

		/** Add admin javascript. */
		add_action( 'admin_enqueue_scripts', [ $this, 'add_admin_scripts' ] );

	}

	/**
	 * Add CSS for admin area.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function add_admin_styles() {

		$screen = get_current_screen();

		/** Add styles only on WP Plugins page. */
		if ( $screen->base == 'plugins' ) {
			wp_enqueue_style( 't42-junction-plugins', Junction::$url . 'css/plugins' . Junction::$suffix . '.css', [], Junction::$version );
		}

	}

	/**
	 * Add JS for admin area.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function add_admin_scripts() {

		$screen = get_current_screen();

		/** Add scripts only on WP Plugins page. */
		if ( $screen->base == 'plugins' ) {
			wp_enqueue_script( 't42-junction-plugins', Junction::$url . 'js/plugins' . Junction::$suffix . '.js', [ 'jquery' ], Junction::$version, true );
		}
	}

	/**
	 * Add additional links on plugin page.
	 *
	 * @param array $links Current links: Deactivate | Edit
	 * @return array
	 * @since 1.0.0
	 * @access public
	 **/
	public function add_links( $links ) {

		array_unshift( $links, '<a title="' . esc_html__( 'Documentation', 't42-junction' ) . '" href="' . esc_url( EnvatoItem::$documentation_url ) . '" target="_blank">' . esc_html__( 'Documentation', 't42-junction' ) . '</a>' );
		array_unshift( $links, '<a title="' . esc_html__( 'Settings', 't42-junction' ) . '" href="' . admin_url( 'options-general.php?page=t42_junction_settings' ) . '">' . esc_html__( 'Settings', 't42-junction' ) . '</a>' );
		array_push( $links, '<a href="https://42theme.com/" target="_blank" class="t42-junction-site"><img src="data:image/svg+xml;base64,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" alt="' . esc_html__( 'Documentation', 't42-junction' ) . '"> 42Theme</a>' );

		return $links;
	}

	/**
	 * Add "Rate us" link on plugin page.
	 *
	 * @param array $links The array having default links for the plugin. Deactivate | Edit
	 * @param string $file The name of the plugin file.
	 * @return array
	 * @since 1.0.0
	 * @access public
	 **/
	public function add_row_meta( $links, $file ) {

		if ( Junction::$basename !== $file ) {
			return $links;
		}

		$links[] = esc_html__( 'Rate us:', 't42-junction' )
		           . "<span class='t42-junction-rating-stars'>"
		           . "     <a href='https://codecanyon.net/downloads?ref=42Theme' target='_blank'>"
		           . "         <span class='dashicons dashicons-star-filled'></span>"
		           . "     </a>"
		           . "     <a href='https://codecanyon.net/downloads?ref=42Theme' target='_blank'>"
		           . "         <span class='dashicons dashicons-star-filled'></span>"
		           . "     </a>"
		           . "     <a href='https://codecanyon.net/downloads?ref=42Theme' target='_blank'>"
		           . "         <span class='dashicons dashicons-star-filled'></span>"
		           . "     </a>"
		           . "     <a href='https://codecanyon.net/downloads?ref=42Theme' target='_blank'>"
		           . "         <span class='dashicons dashicons-star-filled'></span>"
		           . "     </a>"
		           . "     <a href='https://codecanyon.net/downloads?ref=42Theme' target='_blank'>"
		           . "         <span class='dashicons dashicons-star-filled'></span>"
		           . "     </a>"
		           . "<span>";

		return $links;

	}

	/**
	 * Main PluginHelper Instance.
	 *
	 * Insures that only one instance of PluginHelper exists in memory at any one time.
	 *
	 * @static
	 * @return PluginHelper
	 * @since 1.0.0
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof PluginHelper ) ) {
			self::$instance = new PluginHelper();
		}

		return self::$instance;

	}

} // End Class PluginHelper.

