<?php
/**
 * External Links Controller for WordPress. Exclusively on Envato Market: http://codecanyon.net/user/42theme/portfolio?ref=42Theme
 * @encoding     UTF-8
 * @version      1.0.3
 * @copyright    Copyright (C) 2020 42Theme ( https://42theme.com ). All rights reserved.
 * @license      Envato License https://codecanyon.net/licenses/standard?ref=42theme
 * @author       Alexander Khmelnitskiy ( ak@42theme.com )
 * @support      support@42theme.com
 **/

namespace T42\Junction;

use T42\Junction as Junction;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class used to render settings fields.
 *
 * @since 1.0.0
 * @author Alexander Khmelnitskiy ( ak@42theme.com )
 **/
final class Settings {

	/**
	 * Junction Plugin settings.
	 *
	 * @var array()
	 * @since 1.0.0
	 **/
	public $options = [];

	/**
	 * The one true Settings.
	 *
	 * @var Settings
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new Settings instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Get plugin settings. */
		$this->options = $this->get_options();

	}

	/**
	 * Render head part for Settings Page.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_head() {
		?>
        <div class="t42-header">
            <div>
                <h1>
                    <img class="t42-plugin-logo" src="<?php echo esc_attr( Junction::$url ) . 'images/sign-logo.svg'; ?>" alt="<?php esc_html_e( Junction::$name ); ?>">
					<?php esc_html_e( Junction::$name ); ?>
                    <sup><?php echo esc_html__( 'ver.', 't42-junction' ) . esc_html( Junction::$version ); ?></sup>
                </h1>

                <?php if ( ! PluginActivation::get_instance()->is_activated() ) : ?>
                    <div class="t42-activation-status">
                        <i class="t42-icon-warning"></i>
                        <?php esc_html_e( 'Your copy of Junction plugin is not activated.', 't42-junction' ); ?>
                        <a href="#" class="t42-button t42-activate-btn"><?php esc_html_e( 'Activate Now', 't42-junction' ); ?></a>
                    </div>
                <?php endif; ?>

            </div>
        </div>
		<?php
	}

	/**
	 * Render tabs part for Settings Page.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_tabs() {
	    ?>
        <div class="t42-tab-box" >
            <div>
                <ul class="t42-tab" data-t42-tab="{connect:'#t42-settings-tab', animation: 'fade', swiping: false}">
                    <li><a href=""><?php esc_html_e( 'General', 't42-junction' ); ?></a></li>
                    <li><a href=""><?php esc_html_e( 'Assignments', 't42-junction' ); ?></a></li>
                    <li><a href=""><?php esc_html_e( 'Activation', 't42-junction' ); ?></a></li>
                    <li><a href=""><?php esc_html_e( 'Status', 't42-junction' ); ?></a></li>
                    <li><a href=""><?php esc_html_e( 'Uninstall', 't42-junction' ); ?></a></li>
                </ul>

                <ul id="t42-settings-tab" class="t42-switcher t42-margin">
                    <li class="t42-tab-general">
						<?php settings_fields( 'JunctionOptionsGroup' ); ?>
						<?php do_settings_sections( 'JunctionOptionsGroup' ); ?>

                        <div class="t42-panel t42-panel-box">
                            <h2><?php esc_html_e( 'General Settings', 't42-junction' ); ?></h2>

                            <div class="t42-alert">
	                            <?php esc_html_e( 'Note. The plugin process only external links. All settings related to outbound links only. If any attributes are set for the link, the plugin will not change them. Thus, you always have complete control over your site.', 't42-junction' ); ?>
                            </div>

                            <?php $SF = SettingsFields::get_instance(); ?>
	                        <?php $SF->target_blank(); ?>
	                        <?php $SF->noopener(); ?>
	                        <?php $SF->noreferrer(); ?>
	                        <?php $SF->nofollow(); ?>
	                        <?php $SF->dofollow(); ?>
	                        <?php $SF->skip(); ?>

                        </div>
                    </li>
                    <li class="t42-tab-assignments">
                        <div class="t42-panel t42-panel-box">
                            <h2><?php esc_html_e( 'Assignments Settings', 't42-junction' ); ?></h2>
                            <?php TabAssignments::get_instance()->render_assignments(); ?>
                        </div>
                    </li>
                    <li class="t42-tab-activation">
                        <div class="t42-panel t42-panel-box">
                            <h2><?php esc_html_e( 'Plugin Activation', 't42-junction' ); ?></h2>
                            <?php
                                /** Plugin Activation Tab. */
                                PluginActivation::get_instance()->render_pid();

                                /** Render FAQ accordion. */
                                PluginActivation::get_instance()->render_FAQ();
                            ?>
                        </div>
                    </li>
                    <li class="t42-tab-status">
                        <div class="t42-panel t42-panel-box">
                            <h2><?php esc_html_e( 'Status Reports', 't42-junction' ); ?></h2>
                            <?php
                            /** Get TabStatus instance. */
                            $status = TabStatus::get_instance();

                            /** Render "System Requirements". */
                            $status->render_system_requirements();

                            /** Render Privacy Notice. */
                            $status->render_privacy_notice();

                            ?>
                        </div>
                    </li>
                    <li class="t42-tab-uninstall">
                        <div class="t42-panel t42-panel-box">
                            <h2><?php esc_html_e( 'Uninstall Settings', 't42-junction' ); ?></h2>
                            <?php SettingsFields::get_instance()->field_delete_plugin(); ?>
                        </div>
                    </li>
                </ul>
            </div>
        </div>
        <?php
    }

	/**
	 * Render footer part for Settings Page.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_foot() {
	    ?>
        <footer class="t42-bottom-fixed">
            <div class="t42-width-1-1">

                <div class="t42-flex t42-flex-middle t42-flex-space-between">
                    <div>
                        <button type="submit" name="submit" id="submit" class="t42-button t42-button-primary"><?php esc_html_e( 'Save Changes', 't42-junction' ); ?></button>
                    </div>
                    <div>
                        <ul class="t42-subnav t42-subnav-line">
                            <li><a href="<?php echo esc_url( EnvatoItem::$documentation_url ); ?>" target="_blank"><?php esc_html_e( 'Documentation', 't42-junction' ); ?></a></li>
                            <li><a href="<?php echo esc_url( EnvatoItem::$changelog_url ); ?>" target="_blank"><?php esc_html_e( 'Changelog', 't42-junction' ); ?></a></li>
                            <li><a href="https://codecanyon.net/user/42theme/portfolio?ref=42Theme" target="_blank"><?php esc_html_e( 'More Plugins', 't42-junction' ); ?></a></li>
                        </ul>
                    </div>
                </div>

            </div>
        </footer>
        <?php
	}

	/**
	 * Plugin Settings Page.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_options_page() {

		/** Security check. */
		if ( ! current_user_can( 'manage_options' ) ) {
			return;
		}

		$activated = 'not-activated';
		if ( PluginActivation::get_instance()->is_activated() ) {
			$activated = 'activated';
        }

		?>
        <!--suppress HtmlUnknownTarget -->
        <form class="t42-form t42-form-horizontal" action='options.php' method='post'>
            <div class="t42-settings-container <?php esc_attr_e( $activated ); ?>">

                <div class="t42-settings-content">

                    <?php $this->render_head(); ?>

                    <?php $this->render_tabs(); ?>

                </div>

                <?php $this->render_foot(); ?>

            </div>
		</form>
		<?php
	}

	/**
	 * Get saved plugin settings or default values.
	 *
	 * @since 1.0.0
	 * @return array
	 **/
	public function get_options() {

		/** Options. */
		$options = get_option( 't42_junction_settings' );

		/** Default values. */
		$defaults = [
			'skip'                                      => '',
			'assignments'                               => "{|matchingMethod|:1,|WPContent|:0,|WPContentVal|:||,|homePage|:0,|menuItems|:0,|menuItemsVal|:||,|dateTime|:0,|dateTimeStart|:||,|dateTimeEnd|:||,|userRoles|:0,|userRolesVal|:||,|URL|:0,|URLVal|:||,|devices|:0,|devicesVal|:||,|PHP|:0,|PHPVal|:||}", // Assignments
            'delete_plugin'                             => 'plugin', // Uninstall mode, only plugin.
			'envato_purchase_code_' . EnvatoItem::get_instance()->get_id() => '',
		];

		/** Default values for checkboxes. */
		if ( $options === false ) {
			$defaults['target_blank'] = 'true'; // Add target="_blank".
			$defaults['noopener'] = 'true'; // Add rel="noopener".
			$defaults['noreferrer'] = 'false'; // Add rel="noreferrer".
			$defaults['nofollow'] = 'true'; // Add rel="nofollow".
			$defaults['dofollow'] = 'false'; // Add rel="dofollow".

        /** Set all to FALSE, if options have one, then wp_parse_args override false value. */
		} else {
			$defaults['target_blank'] = 'false';
			$defaults['noopener']  = 'false'; // Add rel="noopener".
			$defaults['noreferrer'] = 'false'; // Add rel="noreferrer".
			$defaults['nofollow'] = 'false'; // Add rel="nofollow".
			$defaults['dofollow'] = 'false'; // Add rel="dofollow".
		}

		/**
		 * The Purchase Code entered by the user must be stored in the wp_options table as a string
		 * using the following syntax:
		 *    option_name: envato_purchase_code_<item_id>
         *    option_value: <purchase_code>
         **/
        if ( isset( $options['envato_purchase_code_' . EnvatoItem::get_instance()->get_id()] ) ) {
            update_option( 'envato_purchase_code_' . EnvatoItem::get_instance()->get_id(), $options['envato_purchase_code_' . EnvatoItem::get_instance()->get_id()] );
        }

		return wp_parse_args( $options, $defaults );
	}

	/**
	 * Generate Settings Page.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function settings_init() {

		$group_name = 'JunctionOptionsGroup';
		$section_id = 't42_junction_settings_page_general_section';
		$option_name = 't42_junction_settings';

		/** Create settings section. */
		register_setting( $group_name, $option_name );
		add_settings_section( $section_id, '', null, $group_name );

	}

	/**
	 * Add plugin settings page.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function add_settings_page() {
		add_action( 'admin_menu', [$this, 'add_admin_menu'] );
		add_action( 'admin_init', [$this, 'settings_init'] );
	}

	/**
	 * Add admin menu for plugin settings.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return void
	 **/
	public function add_admin_menu() {

		add_menu_page(
			esc_html__( 'Junction Settings', 't42-junction' ),
			esc_html__( 'Junction', 't42-junction' ),
			'manage_options',
			't42_junction_settings',
			[ $this, 'render_options_page' ],
			'data:image/svg+xml;base64,' . base64_encode( file_get_contents( Junction::$path . 'images/logo-menu.svg' ) ),
			'58.2852'// Always change digits after "." for different plugins.
		);
	}

	/**
	 * Main Settings Instance.
	 *
	 * Insures that only one instance of Settings exists in memory at any one time.
	 *
	 * @static
	 * @return Settings
	 * @since 1.0.0
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Settings ) ) {
			self::$instance = new Settings();
		}

		return self::$instance;

	}

} // End Class Settings.

