<?php
/**
 * External Links Controller for WordPress. Exclusively on Envato Market: http://codecanyon.net/user/42theme/portfolio?ref=42Theme
 * @encoding     UTF-8
 * @version      1.0.3
 * @copyright    Copyright (C) 2020 42Theme ( https://42theme.com ). All rights reserved.
 * @license      Envato License https://codecanyon.net/licenses/standard?ref=42theme
 * @author       Alexander Khmelnitskiy ( ak@42theme.com )
 * @support      support@42theme.com
 **/

namespace T42\Junction;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/**
 * SINGLETON: Class used to render fields.
 *
 * @since 1.0.0
 * @author Alexander Khmelnitskiy ( ak@42theme.com )
 **/
final class SettingsUI {

	/**
	 * The one true SettingsUI.
	 *
	 * @var SettingsUI
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Render type="number" input field.
	 *
	 * @param integer $value - Input value.
	 * @param string $label - Field label.
	 * @param string $description
	 * @param array $attributes
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_input( $value, $label = '', $description = '',  $attributes = [] ) {

		/** Prepare html attributes. */
		$id   = isset( $attributes['id'] ) ? $attributes['id'] : 't42-select-' . md5( uniqid( rand(), true ) );
		$name = isset( $attributes['name'] ) ? $attributes['name'] : '';
		$class = isset( $attributes['class'] ) ? $attributes['class'] : '';
		$placeholder = isset( $attributes['placeholder'] ) ? $attributes['placeholder'] : '';

		?>
        <div class="t42-form-row">
			<?php if ( $label ) : ?>
                <label class="t42-form-label" <?php echo ( $id ) ? 'for="' . esc_attr( $id ) . '"' : ''; ?>><?php esc_html_e( $label ); ?></label>
			<?php endif; ?>

            <div class="t42-form-controls">
                <!--suppress HtmlFormInputWithoutLabel -->
                <div class="t42-form-icon">
                    <i class="t42-icon-key"></i>
                    <!--suppress HtmlFormInputWithoutLabel -->
                    <input
                            type='text'
						<?php echo ( $name ) ? 'name="' . esc_attr( $name ) . '"' : ''; ?>
						<?php echo ( $class ) ? 'class="' . esc_attr( $class ) . '"' : ''; ?>
						<?php echo ( $id ) ? 'id="' . esc_attr( $id ) . '"' : ''; ?>
						<?php echo ( $placeholder ) ? 'placeholder="' . esc_attr( $placeholder ) . '"' : ''; ?>
                            value='<?php esc_attr_e( $value ); ?>'
                    >
                </div>

				<?php if ( $description ) : ?>
                    <span class="t42-form-help-inline"><i class="t42-icon-question-circle-o"></i> <?php esc_html_e( $description ); ?></span>
				<?php endif; ?>
            </div>
        </div>
		<?php
	}

	/**
	 * Render type="number" input field.
	 *
	 * @param integer $value - Input value.
	 * @param string $label - Field label.
	 * @param string $description
	 * @param array $attributes
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_activate_input( $value, $label = '', $description = '',  $attributes = [] ) {

		/** Prepare html attributes. */
		$id   = isset( $attributes['id'] ) ? $attributes['id'] : 't42-select-' . md5( uniqid( rand(), true ) );
		$name = isset( $attributes['name'] ) ? $attributes['name'] : '';
		$class = isset( $attributes['class'] ) ? $attributes['class'] : '';
		$placeholder = isset( $attributes['placeholder'] ) ? $attributes['placeholder'] : '';

		?>
        <div class="t42-form-row">
			<?php if ( $label ) : ?>
                <label class="t42-form-label" <?php echo ( $id ) ? 'for="' . esc_attr( $id ) . '"' : ''; ?>><?php esc_html_e( $label ); ?></label>
			<?php endif; ?>

            <div class="t42-form-controls">
                <!--suppress HtmlFormInputWithoutLabel -->
                <div class="t42-form-icon">
                    <i class="t42-icon-key"></i>
                    <!--suppress HtmlFormInputWithoutLabel -->
                    <input
                            type='text'
						<?php echo ( $name ) ? 'name="' . esc_attr( $name ) . '"' : ''; ?>
						<?php echo ( $class ) ? 'class="' . esc_attr( $class ) . '"' : ''; ?>
						<?php echo ( $id ) ? 'id="' . esc_attr( $id ) . '"' : ''; ?>
						<?php echo ( $placeholder ) ? 'placeholder="' . esc_attr( $placeholder ) . '"' : ''; ?>
                            value='<?php esc_attr_e( $value ); ?>'
                    >
                </div>

                <button type="submit" name="activate" class="t42-activate-license-btn t42-button" ><i class="uk-icon-key"></i> Activate</button>

				<?php if ( $description ) : ?>
                    <span class="t42-form-help-inline"><?php esc_html_e( $description ); ?></span>
				<?php endif; ?>
            </div>
        </div>
		<?php
	}

	/**
	 * Render type="number" input field.
	 *
	 * @param string $name - Settings field name.
	 * @param integer $value - Input value.
	 * @param string | bool $class - CSS Classes for input field.
	 * @param string $label - Field label.
	 * @param string $desc - Field description.
	 * @param integer | bool $min - Input minimum value.
	 * @param integer | bool $max - Input maximum value.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_number( $name, $value, $class = false, $label = '',  $desc = '', $min = false, $max = false  ) {

		$id = 't42-number-' . md5( uniqid( rand(), true ) ); // Uniq id.
		$class = ( $class !== false ) ? $class : '';
		$min = ( $min !== false ) ? $min : '';
		$max = ( $max !== false ) ? $max : '';

		?>
        <div class="t42-form-row">
			<?php if ( $label ) : ?>
                <label class="t42-form-label" <?php echo ( $id ) ? 'for="' . esc_attr( $id ) . '"' : ''; ?>><?php esc_html_e( $label ); ?></label>
			<?php endif; ?>

            <div class="t42-form-controls">
                <!--suppress HtmlFormInputWithoutLabel -->
                <input type='number'
                       required="required"
					<?php echo ( $class ) ? 'class="' . esc_attr( $class ) . '"' : ''; ?>
					<?php echo ( $id ) ? 'id="' . esc_attr( $id ) . '"' : ''; ?>
					<?php echo ( $label ) ? 'placeholder="' . esc_attr( $label ) . '"' : ''; ?>
                       name='<?php esc_attr_e( $name ); ?>'
					<?php echo ( $min ) ? 'min="' . esc_attr( $min ) . '"' : ''; ?>
					<?php echo ( $max ) ? 'max="' . esc_attr( $max ) . '"' : ''; ?>
                       value='<?php esc_attr_e( $value ); ?>'>

				<?php if ( $desc ) : ?>
                    <span class="t42-form-help-inline"><i class="t42-icon-question-circle-o"></i> <?php esc_html_e( $desc ); ?></span>
				<?php endif; ?>
            </div>
        </div>
		<?php
	}

	/**
	 * Render type="checkbox" input field.
	 *
	 * @param string $name - Settings field name.
	 * @param string $label - Field label.
	 * @param bool $checked - Is checkbox checked?
	 * @param string | bool $class - CSS Classes for input field.
	 * @param string $desc - Field description.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_checkbox( $name, $label, $checked, $class = false,  $desc = '' ) {

		$id = 't42-checkbox-' . md5( uniqid( rand(), true ) ); // Uniq id.
		$class = ( $class !== false ) ? $class : '';

		?>
        <div class="t42-form-row">
			<?php if ( $label ) : ?>
                <label class="t42-form-label"><?php esc_html_e( $label ); ?></label>
			<?php endif; ?>

            <div class="t42-form-controls">

                <div class="t42-display-inline-block">
                    <label class="pure-material-switch">
                        <input type="checkbox"
							<?php echo ( $id ) ? 'id="' . esc_attr( $id ) . '"' : ''; ?>
							<?php echo ( $class ) ? 'class="' . esc_attr( $class ) . '"' : ''; ?>
							<?php esc_attr_e( $checked ); ?>
                               name='<?php esc_attr_e( $name ); ?>'
                               value="true"
                        >
                        <span></span>
                    </label>
                </div>

				<?php if ( $desc ) : ?>
                    <span class="t42-form-help-inline t42-large"><i class="t42-icon-question-circle-o"></i> <?php esc_html_e( $desc ); ?></span>
				<?php endif; ?>
            </div>
        </div>
		<?php
	}

	/**
	 * Render select field.
	 *
	 * @param array $options - Options for select.
	 * @param string $selected - Value of selected option.
	 * @param string $label - Field label.
	 * @param string $description - Short field description.
	 * @param array $attributes - Additional attributes for select: id, name, class. Optional.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function render_select( $options, $selected, $label, $description = '',  $attributes = [] ) {

		/** Prepare html attributes. */
		$id   = isset( $attributes['id'] ) ? $attributes['id'] : 't42-select-' . md5( uniqid( rand(), true ) );
		$name = isset( $attributes['name'] ) ? $attributes['name'] : '';
		$class = isset( $attributes['class'] ) ? $attributes['class'] : '';


		?>
        <div class="t42-form-row">
			<?php if ( $label ) : ?>
                <label class="t42-form-label" <?php echo ( $id ) ? 'for="' . esc_attr( $id ) . '"' : ''; ?>><?php esc_html_e( $label ); ?></label>
			<?php endif; ?>

            <div class="t42-form-controls">
                <!--suppress HtmlFormInputWithoutLabel -->
                <select
					<?php echo ( $id ) ? 'id="' . esc_attr( $id ) . '"' : ''; ?>
					<?php echo ( $name ) ? 'name="' . esc_attr( $name ) . '"' : ''; ?>
					<?php echo ( $class ) ? 'class="' . esc_attr( $class ) . '"' : ''; ?>>

					<?php
					foreach ( $options as $key => $value ) {
						$isSelected = ( $key == $selected ) ? 'selected=""' : '';
						echo "<option " . esc_attr( $isSelected ) . " value='" . esc_attr( $key ) . "'>" . esc_html( $value ) . "</option>";
					}
					?>
                </select>

				<?php if ( $description ) : ?>
                    <span class="t42-form-help-inline"><i class="t42-icon-question-circle-o"></i> <?php esc_html_e( $description ); ?></span>
				<?php endif; ?>
            </div>
        </div>
		<?php
	}

	/**
	 * Main SettingsUI Instance.
	 *
	 * Insures that only one instance of SettingsUI exists in memory at any one time.
	 *
	 * @static
	 * @return SettingsUI
	 * @since 1.0.0
	 **/
	public static function get_instance() {

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof SettingsUI ) ) {
			self::$instance = new SettingsUI();
		}

		return self::$instance;

	}

} // End Class SettingsUI.

