<?php
/**
 * Plugin Name: Junction
 * Plugin URI: https://junction.wordpress.42theme.com/
 * Description: External Links Controller for WordPress. Fixes unsafe links to Cross-origin destinations, ensures links open in new tabs, improves SEO and Site Reputation.
 * Version: 1.0.3
 * Author: 42Theme
 * Author URI: https://codecanyon.net/user/42theme/portfolio/?ref=42theme
 **/

/**
 * External Links Controller for WordPress. Exclusively on Envato Market: http://codecanyon.net/user/42theme/portfolio?ref=42Theme
 * @encoding     UTF-8
 * @version      1.0.3
 * @copyright    Copyright (C) 2020 42Theme ( https://42theme.com ). All rights reserved.
 * @license      Envato License https://codecanyon.net/licenses/standard?ref=42theme
 * @author       Alexander Khmelnitskiy ( ak@42theme.com )
 * @support      support@42theme.com
 **/

namespace T42;

use T42\Junction\EnvatoItem;
use T42\Junction\MessageHelper;
use T42\Junction\PluginActivation;
use T42\Junction\PluginHelper;
use T42\Junction\PluginUpdater;
use T42\Junction\Settings;
use T42\Junction\Shortcodes;
use T42\Junction\simple_html_dom;
use T42\Junction\TabAssignments;
use T42\Junction\Helper;

/** Exit if accessed directly. */
if ( ! defined( 'ABSPATH' ) ) {
	header( 'Status: 403 Forbidden' );
	header( 'HTTP/1.1 403 Forbidden' );
	exit;
}

/** Include plugin autoloader for additional classes. */
require __DIR__ . '/src/autoload.php';

/**
 * SINGLETON: Core class used to implement a Junction plugin.
 *
 * This is used to define core logic of plugin.
 *
 * @since 1.0.0
 **/
final class Junction {

	/**
     * Plugin version.
	 *
	 * @string $version
	 * @since 1.0.0
	 **/
	public static $version = '';

	/**
	 * Plugin name.
	 *
	 * @string $version
	 * @since 1.0.0
	 **/
	public static $name = '';

	/**
	 * Use minified libraries if SCRIPT_DEBUG is turned off.
	 *
	 * @since 1.0.0
	 **/
	public static $suffix = '';

	/**
	 * URL (with trailing slash) to plugin folder.
	 *
	 * @var string
	 * @since 1.0.0
	 **/
	public static $url = '';

	/**
	 * PATH to plugin folder.
	 *
	 * @var string
	 * @since 1.0.0
	 **/
	public static $path = '';

	/**
	 * Plugin base name.
	 *
	 * @var string
	 * @since 1.0.0
	 **/
	public static $basename = '';

	/**
	 * The one true Junction.
	 *
	 * @var Junction
	 * @since 1.0.0
	 **/
	private static $instance;

	/**
	 * Sets up a new plugin instance.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	private function __construct() {

		/** Initialize main variables. */
	    $this->initialization();

	    /** Define admin hooks. */
		$this->admin_hooks();

		/** Define public hooks. */
		$this->public_hooks();

		/** Define hooks that runs on both the front-end as well as the dashboard. */
		$this->both_hooks();

	}

	/**
	 * Define hooks that runs on both the front-end as well as the dashboard.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 **/
	private function both_hooks() {

		/** Load translation. */
		add_action( 'plugins_loaded', [$this, 'load_textdomain'] );

	}

	/**
	 * Return plugin version.
	 *
	 * @return string
	 * @since 1.0.0
	 * @access public
	 **/
	public function get_version() {
		return self::$version;
	}

	/**
	 * Initialize main variables.
	 *
	 * @since 1.0.0
	 * @access public
     * @return void
	 **/
	public function initialization() {

		/** Plugin version. */
		if ( ! function_exists('get_plugin_data') ) {
			require_once( ABSPATH . 'wp-admin/includes/plugin.php' );
		}

		$plugin_data = get_plugin_data( __FILE__ );
		self::$version = $plugin_data['Version'];

		/** Plugin name. */
		self::$name = $plugin_data['Name'];

		/** Gets the plugin URL (with trailing slash). */
		self::$url = plugin_dir_url( __FILE__ );

		/** Gets the plugin PATH. */
		self::$path = plugin_dir_path( __FILE__ );

		/** Use minified libraries if SCRIPT_DEBUG is turned off. */
		self::$suffix = ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG ) ? '' : '.min';

		/** Set plugin basename. */
		self::$basename = plugin_basename( __FILE__ );

	}

	/**
	 * Register all of the hooks related to the admin area functionality.
	 *
	 * @since 1.0.0
	 * @access private
     * @return void
	 **/
    private function admin_hooks() {

	    /** Run PluginHelper. */
	    PluginHelper::get_instance();

	    /** Add plugin setting page. */
        Settings::get_instance()->add_settings_page();

	    /** Add admin styles. */
	    add_action( 'admin_enqueue_scripts', [$this, 'admin_styles' ] );

	    /** Add admin javascript. */
	    add_action( 'admin_enqueue_scripts', [$this, 'admin_scripts' ] );

	    /** Remove "Thank you for creating with WordPress" and WP version only from plugin settings page. */
	    add_action( 'admin_enqueue_scripts', [$this, 'remove_wp_copyrights'] );

	    /** Remove all "third-party" notices from plugin settings page. */
	    add_action( 'in_admin_header', [$this, 'remove_all_notices'], 1000);

	    /** Plugin Assignments tab. */
	    TabAssignments::get_instance();

	    /** Plugin update mechanism enable only if plugin have Envato ID. */
	    $plugin_id = EnvatoItem::get_instance()->get_id();

	    if ( (int)$plugin_id > 0 ) {
		    PluginUpdater::get_instance();
	    }

	    /** Add "Settings Saved" message. */
	    $this->add_message_settings_saved();

	    /** Add "Activation" message. */
	    add_action( 'update_option_t42_junction_settings', [$this, 'add_message_activation'], 10, 2 );
    }

	/**
	 * Fires after the value of envato_purchase_code_{item_id} option has been successfully updated.
	 *
	 * @param $old_value mixed - The old option value.
	 * @param $value mixed - The new option value.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 **/
    public function add_message_activation( $old_value, $value ) {

    	/** Work only if purchase code was changed. */
    	if (
    		$old_value['envato_purchase_code_' . EnvatoItem::get_instance()->get_id()] ===
		    $value['envato_purchase_code_' . EnvatoItem::get_instance()->get_id()] ) {
    		return;
	    }

	    PluginActivation::get_instance()->reset_temporary_activation();

	    if ( PluginActivation::get_instance()->is_activated( $value['envato_purchase_code_' . EnvatoItem::get_instance()->get_id()] ) ) {

		    /** Add "Activation success" message. */
		    MessageHelper::get_instance()->add_message([
			    'title' => esc_html__( 'Activation Success.', 't42-junction' ),
			    'message' => esc_html__( 'Thank you for activating.', 't42-junction' ),
			    'icon' => 'ico-check',
			    'color' => 'green',
			    'timeout' => 7500
		    ]);

	    } else {

		    /** Add "Activation failed" message. */
		    MessageHelper::get_instance()->add_message([
			    'title' => esc_html__( 'Activation failed.', 't42-junction' ),
			    'message' => esc_html__( 'Invalid purchase code.', 't42-junction' ),
			    'icon' => 'ico-ban',
			    'color' => 'red',
			    'timeout' => 7500
		    ]);

	    }

    }

	/**
	 * Add "Settings Saved" message.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 **/
    public function add_message_settings_saved() {

    	/** Work only if settings was updated. */
	    if ( ! isset( $_GET['settings-updated'] ) ) { return; }

	    $settings_updated = filter_input( INPUT_GET, 'settings-updated' );
	    $settings_updated = filter_var( $settings_updated, FILTER_VALIDATE_BOOLEAN );

	    /** Some error occurred on settings save. */
	    if ( ! $settings_updated ) { return; }

	    /** Add "Settings saved" message. */
	    MessageHelper::get_instance()->add_message([
		    'title' => esc_html__( 'Settings saved.', 't42-junction' ),
		    'message' => esc_html__( 'Plugin settings saved successfully.', 't42-junction' ),
		    'icon' => 'ico-info',
		    'color' => 'blue',
		    'timeout' => 7000
	    ]);

    }

	/**
	 * Register all of the hooks related to the public-facing functionality.
	 *
	 * @since 1.0.0
	 * @access private
     * @return void
	 **/
	private function public_hooks() {

		/** Start buffering to get full HTML document. */
		add_action( 'template_redirect', [$this, 'buffer_start'], 0 );

		/** End buffering to get full HTML document. */
		add_action( 'plugins_loaded', [$this, 'buffer_end'], PHP_INT_MAX );

		/** Add shortcodes. */
		Shortcodes::get_instance();
    }

	/**
	 * Performs all manipulations with the links.
	 *
	 * @param string $buffer - Full HTML of page.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return string|string[]
	 */
	public function process( $buffer ) {

		/** Checks if plugin should work on this page. */
		if ( ! TabAssignments::get_instance()->display() ) { return $buffer; }

		/** Checks if page have [disable_junction] shortcode. */
		if ( strpos( $buffer, '[disable_junction]' ) !== false ) {

			/** Remove Shortcode and exit. */
			$buffer = str_replace('[disable_junction]', '', $buffer );

			return $buffer;
		}

		/** Get Plugin Settings. */
		$options = Settings::get_instance()->options;

		/** Create a DOM object. */
		if ( ! class_exists( 'simple_html_dom' ) ) {
			require __DIR__ . '/src/simple_html_dom.php';
		}
		$html = new simple_html_dom();

		/** Load HTML from a string. */
		$html->load( $buffer );

		/** Foreach 'a' on page. */
		foreach ( $html->find( 'a[href*="//"]' ) as $a ) {

			/** Skip Internal links. */
			if ( $this->is_internal( $a->href ) ) { continue; }

			/** Skip links from 'Skip Links' params. */
			if ( $this->is_in_skip_links( $a->href, $options['skip'] ) ) { continue; }

			/** Link Has attribute junction-skip="" skip it. */
			if ( isset( $a->attr['junction-skip'] ) ) { continue; }

			/** Link Has class junction-skip skip it. */
			if ( strpos( $a->class, 'junction-skip' ) !== false ) { continue; }

			/** Add target="_blank" */
			if ( filter_var( $options['target_blank'], FILTER_VALIDATE_BOOLEAN ) ) {

				/** Set target to '_blank' only if link don't have target="" */
				if ( ! isset( $a->attr['target'] ) ) {
					$a->attr['target'] = '_blank';
				}

			}

			/** Set rel only if link don't have rel="" */
			if ( ( ! isset( $a->attr['rel'] ) ) OR ( empty( trim( $a->attr['rel'] ) ) )  ){

				$rel = '';

				/** Add rel="noopener". */
				if ( filter_var( $options['noopener'], FILTER_VALIDATE_BOOLEAN ) ) {
					$rel .= ' noopener ';
				}

				/** Add rel="noreferrer". */
				if ( filter_var( $options['noreferrer'], FILTER_VALIDATE_BOOLEAN ) ) {
					$rel .= ' noreferrer ';
				}

				/** Add rel="nofollow". */
				if ( filter_var( $options['nofollow'], FILTER_VALIDATE_BOOLEAN ) ) {
					$rel .= ' nofollow ';
				}

				/** Add rel="dofollow". */
				if ( filter_var( $options['dofollow'], FILTER_VALIDATE_BOOLEAN ) ) {
					$rel .= ' dofollow ';
				}

				$a->attr['rel'] = trim( preg_replace( '!\s+!', ' ', $rel ) );

			}

		}

		/** Return result. */
		return $html->save();
	}

	/**
	 * Return true href in 'Skip Links' param.
	 *
	 * @param string $href - URL to check.
	 * @param string $skip - Parts of the URLs to match
	 *
	 * @since  1.0.0
	 * @access public
	 * @return bool
	 **/
	private function is_in_skip_links( $href, $skip ) {

		/** Slit textarea to lines. */
		$skip_urls = preg_split('/\r\n|[\r\n]/', $skip );

		/** Remove empty array elements. */
		$skip_urls = array_filter( $skip_urls );

		foreach ( $skip_urls as $url ) {

			$url = trim( $url ); // Some clearing.

			/** If href contain url - skip link. */
			if ( strpos( $href, $url ) !== false ) {
				return true;
			}

		}

		return false;

	}

	/**
	 * Return true if this is internal link.
	 *
	 * @param string $href - URL to check.
	 *
	 * @since  1.0.0
	 * @access public
	 * @return bool
	 **/
	private function is_internal( $href ) {

		/** Prepare href. */
		$parsed_url = parse_url( $href );
		$href = $parsed_url['host'] . $parsed_url['path'];

		/** Prepare Site url. */
		$parsed_site = parse_url( get_site_url() );
		$site = $parsed_site['host'] . $parsed_site['path'];

		/** If href start with site url is is internal link. */
		if ( substr( $href, 0, strlen( $site ) ) === $site ) {
			return true;
		} else {
			return false;
		}

	}

	/**
	 * End buffering to get full HTML document.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function buffer_end() {

		if ( ob_get_length() ) {
			ob_end_flush();
		}

	}

	/**
	 * Start buffering to get full HTML document.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
    public function buffer_start() {

	    ob_start( [$this, 'process'] );

	}

	/**
	 * Remove all other notices.
	 *
	 * @since 1.0.0
	 * @access public
	 **/
	public function remove_all_notices() {

		/** Work only on plugin settings page. */
		$screen = get_current_screen();
		if ( $screen->base != "toplevel_page_t42_junction_settings" ) { return; }

		/** Remove other notices. */
		remove_all_actions( 'admin_notices' );
		remove_all_actions( 'all_admin_notices' );

	}

	/**
	 * Remove "Thank you for creating with WordPress" and WP version only from plugin settings page.
	 *
	 * @since 1.0.0
	 * @access private
	 * @return void
	 **/
	public function remove_wp_copyrights() {

		/** Remove "Thank you for creating with WordPress" and WP version from plugin settings page. */
		$screen = get_current_screen(); // Get current screen.

		/** Junction Settings Page. */
		if ( $screen->base == 'toplevel_page_t42_junction_settings' ) {
			add_filter( 'admin_footer_text', '__return_empty_string', 11 );
			add_filter( 'update_footer', '__return_empty_string', 11 );
		}

	}

	/**
	 * Load the plugin text domain for translation.
	 *
	 * @since 1.0.0
	 * @access public
     * @return void
	 **/
	public function load_textdomain() {

		load_plugin_textdomain( 't42-junction', false, basename( dirname( __FILE__ ) ) . '/languages/' );

	}

	/**
	 * Add CSS for admin area.
	 *
	 * @param string $hook - The current admin page.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function admin_styles( $hook ) {

		/** Load styles for plugin settings page. */
		if ( 'toplevel_page_t42_junction_settings' === $hook ) {

			wp_enqueue_style( 't42-iziToast', self::$url . 'css/iziToast' . self::$suffix . '.css', [], self::$version );
			wp_enqueue_style( 't42-junction-uikit', self::$url . 'css/uikit' . self::$suffix . '.css', [], self::$version );
			wp_enqueue_style( 't42-junction-admin', self::$url . 'css/admin' . self::$suffix . '.css', [], self::$version );

		} elseif ( 'plugin-install.php' === $hook ) {

			/** Styles only for our plugin. */
			if ( isset( $_GET['plugin'] ) AND $_GET['plugin'] === 't42-junction' ) {
				wp_enqueue_style( 't42-junction-plugin-install', self::$url . 'css/plugin-install' . self::$suffix . '.css', [], self::$version );
			}

		}

	}

	/**
	 * Add JS for admin area.
	 *
	 * @param string $hook - The current admin page.
	 *
	 * @return void
	 * @since 1.0.0
	 **/
	public function admin_scripts( $hook ) {

		/** Scripts only for plugin settings page. */
		if ( $hook === 'toplevel_page_t42_junction_settings' ) {
			wp_enqueue_script( 't42-iziToast', self::$url . 'js/iziToast' . self::$suffix . '.js', [], self::$version, true );
			wp_enqueue_script( 't42-junction-uikit', self::$url . 'js/uikit' . self::$suffix . '.js', ['jquery'], self::$version, true );
			wp_enqueue_script( 't42-junction-uikit-accordion', self::$url . 'js/accordion.uikit' . self::$suffix . '.js', ['jquery', 't42-junction-uikit'], self::$version, true );
			wp_enqueue_script( 't42-junction-admin', self::$url . 'js/admin' . self::$suffix . '.js', ['jquery'], self::$version, true );

			/** Pass admin messages to JavaScript. */
			$this->pass_messages_to_js();
		}

	}

	/**
	 * Pass admin messages to JavaScript.
	 *
	 * @static
	 * @since 1.0.0
	 **/
	private function pass_messages_to_js() {

		/** Pass admin messages to JavaScript. */
		$messages = MessageHelper::get_instance()->get_messages();
		if ( count( $messages ) ) {
			wp_localize_script( 't42-junction-admin', 't42Junction', [
				'messages' => $messages
			] );
		}

		/** Clear old messages. */
		MessageHelper::get_instance()->clear_messages();

	}

	/**
	 * Run when the plugin is activated.
	 *
	 * @static
	 * @since 1.0.0
	 **/
	public static function on_activation() {

		/** Security checks. */
		if ( ! current_user_can( 'activate_plugins' ) ) {
			return;
		}

		$plugin = isset( $_REQUEST['plugin'] ) ? $_REQUEST['plugin'] : '';
		check_admin_referer( "activate-plugin_{$plugin}" );

		/** Send install Action to our host. */
		Helper::get_instance()->send_action( 'install', 't42-junction', self::$version );

	}

	/**
	 * Main Junction Instance.
	 *
	 * Insures that only one instance of Junction exists in memory at any one time.
	 *
	 * @static
     * @return Junction | null
	 * @since 1.0.0
	 **/
	public static function get_instance() {

        if ( isset( $_REQUEST['wc-ajax'] ) ) { return null; }

		if ( ! isset( self::$instance ) && ! ( self::$instance instanceof Junction ) ) {

			self::$instance = new Junction;

		}

		return self::$instance;
	}

} // End Class Junction.

/** Run when the plugin is activated. */
register_activation_hook( __FILE__, ['T42\Junction', 'on_activation'] );

/** Run Junction class. */
Junction::get_instance();
