<?php
namespace Barn2\Plugin\Posts_Table_Pro\Admin;

use Posts_Table_Args,
    Barn2\Plugin\Posts_Table_Pro\Util\Options,
    Barn2\PTP_Lib\Registerable,
    Barn2\PTP_Lib\Service,
    Barn2\PTP_Lib\Conditional,
    Barn2\PTP_Lib\Plugin\Licensed_Plugin,
    Barn2\PTP_Lib\Admin\Settings_API_Helper,
    Barn2\PTP_Lib\Util;

/**
 * This class handles our plugin settings page in the admin.
 *
 * @package   Barn2\posts-table-pro
 * @author    Barn2 Plugins <support@barn2.co.uk>
 * @license   GPL-3.0
 * @copyright Barn2 Media Ltd
 */
class Settings_Page implements Service, Registerable, Conditional {

    const MENU_SLUG    = 'posts_table';
    const OPTION_GROUP = 'posts_table_pro';

    private $plugin;

    public function __construct( Licensed_Plugin $plugin ) {
        $this->plugin = $plugin;
    }

    public function is_required() {
        return Util::is_admin();
    }

    public function register() {
        add_action( 'admin_menu', [ $this, 'add_settings_page' ] );
        add_action( 'admin_init', [ $this, 'register_settings' ] );
    }

    public function add_settings_page() {
        add_options_page(
            __( 'Posts Table Pro Settings', 'posts-table-pro' ), __( 'Posts Table Pro', 'posts-table-pro' ), 'manage_options', self::MENU_SLUG, array(
            $this, 'render_settings_page' )
        );
    }

    public function render_settings_page() {
        ?>
        <div class="wrap">
            <h1><?php _e( 'Posts Table Pro Settings', 'posts-table-pro' ); ?></h1>
            <form action="options.php" method="post">
                <?php
                // Output the hidden form fields (_wpnonce, etc)
                settings_fields( self::OPTION_GROUP );

                // Output the sections and their settings
                do_settings_sections( self::MENU_SLUG );
                ?>
                <p class="submit">
                    <input name="Submit" type="submit" name="submit" class="button button-primary" value="<?php esc_attr_e( 'Save Changes', 'posts-table-pro' ); ?>" />
                </p>
            </form>
        </div>
        <?php
    }

    public function register_settings() {
        $default_args = Posts_Table_Args::$default_args;

        register_setting( self::OPTION_GROUP, Options::SHORTCODE_OPTION_KEY, [
            'type'              => 'string', // array type not supported, so just use string
            'description'       => 'Posts Table Pro shortcode defaults',
            'sanitize_callback' => [ $this, 'sanitize_shortcode_settings' ]
        ] );

        register_setting( self::OPTION_GROUP, Options::MISC_OPTION_KEY, [
            'type'              => 'string', // array type not supported, so just use string
            'description'       => 'Posts Table Pro miscellaneous settings',
            'sanitize_callback' => [ $this, 'sanitize_misc_settings' ]
        ] );

        // Licence Key section.
        Settings_API_Helper::add_settings_section(
            'ptp_license_key', self::MENU_SLUG, '', [ $this, 'support_links' ],
            [
                $this->plugin->get_license_setting()->get_license_key_setting(),
                $this->plugin->get_license_setting()->get_license_override_setting()
            ]
        );

        // Table Content section.
        Settings_API_Helper::add_settings_section(
            'ptp_shortcode_defaults', self::MENU_SLUG, __( 'Table content', 'posts-table-pro' ), [ $this, 'table_content_description' ], [
            [
                'id'      => Options::SHORTCODE_OPTION_KEY . '[post_type]',
                'title'   => __( 'Post type', 'posts-table-pro' ),
                'type'    => 'select',
                'desc'    => sprintf(
                    __( 'Displays all published posts by default. To restrict posts by category, tag, custom field, etc. add the %1$srelevant option%2$s to your shortcode.', 'posts-table-pro' ),
                    Util::format_link_open( Util::barn2_url( 'kb/ptp-include-exclude/' ) ),
                    '</a>'
                ),
                'options' => $this->get_registered_post_types(),
                'default' => $default_args['post_type'],
            ],
            [
                'id'      => Options::SHORTCODE_OPTION_KEY . '[columns]',
                'title'   => __( 'Columns', 'posts-table-pro' ),
                'type'    => 'text',
                'desc'    => __( 'Enter the columns for your posts table.', 'posts-table-pro' ) . $this->read_more( 'kb/posts-table-columns/' ),
                'default' => $default_args['columns'],
            ],
            [
                'id'      => Options::SHORTCODE_OPTION_KEY . '[image_size]',
                'title'   => __( 'Image size', 'posts-table-pro' ),
                'type'    => 'text',
                'desc'    => __( "Enter     WxH in pixels (e.g. 80x80).", 'posts-table-pro' ) . $this->read_more( 'kb/ptp-column-widths/#image-size' ),
                'default' => $default_args['image_size'],
            ],
            [
                'id'      => Options::SHORTCODE_OPTION_KEY . '[lightbox]',
                'title'   => __( 'Image lightbox', 'posts-table-pro' ),
                'type'    => 'checkbox',
                'label'   => __( 'Display images in a lightbox when opened', 'posts-table-pro' ),
                'default' => $default_args['lightbox'],
            ],
            [
                'title'   => __( 'Shortcodes', 'posts-table-pro' ),
                'type'    => 'checkbox',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[shortcodes]',
                'label'   => __( 'Display shortcodes, HTML and other formatting inside the table content', 'posts-table-pro' ),
                'default' => $default_args['shortcodes']
            ],
            [
                'id'                => Options::SHORTCODE_OPTION_KEY . '[content_length]',
                'title'             => __( 'Content length', 'posts-table-pro' ),
                'type'              => 'number',
                'class'             => 'small-text',
                'suffix'            => __( 'words', 'posts-table-pro' ),
                'desc'              => __( 'Enter -1 to show the full content.', 'posts-table-pro' ),
                'default'           => $default_args['content_length'],
                'custom_attributes' => [
                    'min' => -1
                ]
            ],
            [
                'id'                => Options::SHORTCODE_OPTION_KEY . '[excerpt_length]',
                'title'             => __( 'Excerpt length', 'posts-table-pro' ),
                'type'              => 'number',
                'class'             => 'small-text',
                'suffix'            => __( 'words', 'posts-table-pro' ),
                'desc'              => __( 'Enter -1 to show the full excerpt.', 'posts-table-pro' ),
                'default'           => $default_args['excerpt_length'],
                'custom_attributes' => [
                    'min' => -1
                ]
            ],
            [
                'id'      => Options::SHORTCODE_OPTION_KEY . '[links]',
                'title'   => __( 'Links', 'posts-table-pro' ),
                'type'    => 'text',
                'desc'    => __( 'Link table data to the post, page, category, tag, or term.', 'posts-table-pro' ) . $this->read_more( 'kb/links-posts-table/' ),
                'default' => $default_args['links'],
            ],
            [
                'id'      => Options::SHORTCODE_OPTION_KEY . '[button_text]',
                'title'   => __( 'Button text', 'posts-table-pro' ),
                'type'    => 'text',
                'desc'    => __( 'The button text when using the \'button\' column in the table.', 'posts-table-pro' ) . $this->read_more( 'kb/posts-table-button-column' ),
                'default' => $default_args['button_text'],
            ]
        ] );

        // Loading Posts section.
        Settings_API_Helper::add_settings_section(
            'ptp_post_loading', self::MENU_SLUG, __( 'Loading', 'posts-table-pro' ), '__return_false', array(
            array(
                'title'             => __( 'Lazy load', 'posts-table-pro' ),
                'type'              => 'checkbox',
                'id'                => Options::SHORTCODE_OPTION_KEY . '[lazy_load]',
                'label'             => __( 'Load the posts one page at a time', 'posts-table-pro' ),
                'desc'              => __( 'Enable this if you have many posts or experience slow page load times.', 'posts-table-pro' ) . '<br/>' .
                __( 'Warning: Lazy load limits the searching and sorting features in the table. Only use it if you definitely need it.', 'posts-table-pro' ) .
                $this->read_more( 'kb/posts-table-lazy-load/' ),
                'default'           => $default_args['lazy_load'],
                'class'             => 'toggle-parent',
                'custom_attributes' => array(
                    'data-child-class' => 'post-limit',
                    'data-toggle-val'  => 0
                )
            ),
            array(
                'title'             => __( 'Post limit', 'posts-table-pro' ),
                'type'              => 'number',
                'id'                => Options::SHORTCODE_OPTION_KEY . '[post_limit]',
                'desc'              => __( 'The maximum number of posts to display in one table. Enter -1 to show all posts.', 'posts-table-pro' ),
                'default'           => $default_args['post_limit'],
                'class'             => 'small-text post-limit',
                'custom_attributes' => array(
                    'min' => -1
                )
            ),
            array(
                'title'             => __( 'Posts per page', 'posts-table-pro' ),
                'type'              => 'number',
                'id'                => Options::SHORTCODE_OPTION_KEY . '[rows_per_page]',
                'desc'              => __( 'The number of posts per page of results. Enter -1 to display all posts on one page.', 'posts-table-pro' ),
                'default'           => $default_args['rows_per_page'],
                'custom_attributes' => array(
                    'min' => -1
                )
            ),
            array(
                'title'             => __( 'Caching', 'posts-table-pro' ),
                'type'              => 'checkbox',
                'id'                => Options::SHORTCODE_OPTION_KEY . '[cache]',
                'label'             => __( 'Cache table contents to improve load time', 'posts-table-pro' ),
                'default'           => $default_args['cache'],
                'class'             => 'toggle-parent',
                'custom_attributes' => array(
                    'data-child-class' => 'expires-after'
                )
            ),
            array(
                'title'             => __( 'Cache expires after', 'posts-table-pro' ),
                'type'              => 'number',
                'id'                => Options::MISC_OPTION_KEY . '[cache_expiry]',
                'suffix'            => __( 'hours', 'posts-table-pro' ),
                'desc'              => __( 'Your table data will be refreshed after this length of time.', 'posts-table-pro' ),
                'default'           => 6,
                'class'             => 'expires-after',
                'custom_attributes' => array(
                    'min' => 1,
                    'max' => 9999
                )
            )
        ) );

        // Sorting section.
        Settings_API_Helper::add_settings_section(
            'ptp_sorting', self::MENU_SLUG, __( 'Sorting', 'posts-table-pro' ), '__return_false', array(
            array(
                'title'             => __( 'Sort by', 'posts-table-pro' ),
                'type'              => 'select',
                'id'                => Options::SHORTCODE_OPTION_KEY . '[sort_by]',
                'options'           => array(
                    'title'         => __( 'Title', 'posts-table-pro' ),
                    'id'            => __( 'ID', 'posts-table-pro' ),
                    'date'          => __( 'Date published', 'posts-table-pro' ),
                    'modified'      => __( 'Date modified', 'posts-table-pro' ),
                    'menu_order'    => __( 'Page order (menu order)', 'posts-table-pro' ),
                    'name'          => __( 'Post slug', 'posts-table-pro' ),
                    'author'        => __( 'Author', 'posts-table-pro' ),
                    'comment_count' => __( 'Number of comments', 'posts-table-pro' ),
                    'rand'          => __( 'Random', 'posts-table-pro' ),
                    'custom'        => __( 'Other', 'posts-table-pro' )
                ),
                'desc'              => __( 'The initial sort order applied to the table.', 'posts-table-pro' ) . $this->read_more( 'kb/posts-table-sort-options/' ),
                'default'           => $default_args['sort_by'],
                'class'             => 'toggle-parent',
                'custom_attributes' => array(
                    'data-child-class' => 'custom-sort',
                    'data-toggle-val'  => 'custom'
                )
            ),
            array(
                'title' => __( 'Sort column', 'posts-table-pro' ),
                'type'  => 'text',
                'id'    => Options::SHORTCODE_OPTION_KEY . '[sort_by_custom]',
                'class' => 'regular-text custom-sort',
                'desc'  => __( 'Enter any column in your table. Note: only available if lazy load is disabled.', 'posts-table-pro' )
            ),
            array(
                'title'   => __( 'Sort direction', 'posts-table-pro' ),
                'type'    => 'select',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[sort_order]',
                'options' => array(
                    ''     => __( 'Automatic', 'posts-table-pro' ),
                    'asc'  => __( 'Ascending (A to Z, oldest to newest)', 'posts-table-pro' ),
                    'desc' => __( 'Descending (Z to A, newest to oldest)', 'posts-table-pro' )
                ),
                'default' => $default_args['sort_order']
            )
        ) );

        // Table Controls section.
        Settings_API_Helper::add_settings_section(
            'ptp_table_controls', self::MENU_SLUG, __( 'Table controls', 'posts-table-pro' ), '__return_false', array(
            array(
                'title'             => __( 'Search filters', 'posts-table-pro' ),
                'type'              => 'select',
                'id'                => Options::SHORTCODE_OPTION_KEY . '[filters]',
                'options'           => array(
                    'false'  => __( 'Disabled', 'posts-table-pro' ),
                    'true'   => __( 'Show based on columns in table', 'posts-table-pro' ),
                    'custom' => __( 'Custom', 'posts-table-pro' )
                ),
                'desc'              => __( 'Show dropdown menus to filter by category, tag, or custom taxonomy.', 'posts-table-pro' ) . $this->read_more( 'kb/posts-table-filters/' ),
                'default'           => $default_args['filters'],
                'class'             => 'toggle-parent',
                'custom_attributes' => array(
                    'data-child-class' => 'custom-search-filter',
                    'data-toggle-val'  => 'custom'
                )
            ),
            array(
                'title' => __( 'Custom filters', 'posts-table-pro' ),
                'type'  => 'text',
                'id'    => Options::SHORTCODE_OPTION_KEY . '[filters_custom]',
                'desc'  => __( 'Enter the filters as a comma-separated list.', 'posts-table-pro' ),
                'class' => 'regular-text custom-search-filter'
            ),
            array(
                'title'   => __( 'Page length', 'posts-table-pro' ),
                'type'    => 'select',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[page_length]',
                'options' => array(
                    'top'    => __( 'Above table', 'posts-table-pro' ),
                    'bottom' => __( 'Below table', 'posts-table-pro' ),
                    'both'   => __( 'Above and below table', 'posts-table-pro' ),
                    'false'  => __( 'Hidden', 'posts-table-pro' )
                ),
                'desc'    => __( "The position of the 'Show [x] entries' dropdown list.", 'posts-table-pro' ),
                'default' => $default_args['page_length']
            ),
            array(
                'title'   => __( 'Search box', 'posts-table-pro' ),
                'type'    => 'select',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[search_box]',
                'options' => array(
                    'top'    => __( 'Above table', 'posts-table-pro' ),
                    'bottom' => __( 'Below table', 'posts-table-pro' ),
                    'both'   => __( 'Above and below table', 'posts-table-pro' ),
                    'false'  => __( 'Hidden', 'posts-table-pro' )
                ),
                'default' => $default_args['search_box']
            ),
            array(
                'title'   => __( 'Totals', 'posts-table-pro' ),
                'type'    => 'select',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[totals]',
                'options' => array(
                    'top'    => __( 'Above table', 'posts-table-pro' ),
                    'bottom' => __( 'Below table', 'posts-table-pro' ),
                    'both'   => __( 'Above and below table', 'posts-table-pro' ),
                    'false'  => __( 'Hidden', 'posts-table-pro' )
                ),
                'desc'    => __( "The position of the post totals, e.g. 'Showing 1 to 5 of 10 entries'.", 'posts-table-pro' ),
                'default' => $default_args['totals']
            ),
            array(
                'title'   => __( 'Pagination buttons', 'posts-table-pro' ),
                'type'    => 'select',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[pagination]',
                'options' => array(
                    'top'    => __( 'Above table', 'posts-table-pro' ),
                    'bottom' => __( 'Below table', 'posts-table-pro' ),
                    'both'   => __( 'Above and below table', 'posts-table-pro' ),
                    'false'  => __( 'Hidden', 'posts-table-pro' )
                ),
                'desc'    => __( 'The position of the paging buttons which scroll between results.', 'posts-table-pro' ),
                'default' => $default_args['pagination']
            ),
            array(
                'title'   => __( 'Pagination type', 'posts-table-pro' ),
                'type'    => 'select',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[paging_type]',
                'options' => array(
                    'numbers'        => __( 'Numbers only', 'posts-table-pro' ),
                    'simple'         => __( 'Prev|Next', 'posts-table-pro' ),
                    'simple_numbers' => __( 'Prev|Next + Numbers', 'posts-table-pro' ),
                    'full'           => __( 'Prev|Next|First|Last', 'posts-table-pro' ),
                    'full_numbers'   => __( 'Prev|Next|First|Last + Numbers', 'posts-table-pro' )
                ),
                'default' => $default_args['paging_type']
            ),
            array(
                'title'   => __( 'Reset button', 'posts-table-pro' ),
                'type'    => 'checkbox',
                'id'      => Options::SHORTCODE_OPTION_KEY . '[reset_button]',
                'label'   => __( 'Show the reset button above the table', 'posts-table-pro' ),
                'default' => $default_args['reset_button']
            )
        ) );
    }

    public function sanitize_shortcode_settings( $args ) {
        $this->plugin->get_license_setting()->save_posted_license_key();

        // Check for empties
        foreach ( array( 'image_size', 'links' ) as $arg ) {
            if ( empty( $args[$arg] ) ) {
                $args[$arg] = Posts_Table_Args::$default_args[$arg];
            }
        }

        // Sanitize image size
        if ( isset( $args['image_size'] ) ) {
            $args['image_size'] = preg_replace( '/[^\wx\-]/', '', $args['image_size'] );
        }

        // Check ints
        foreach ( array( 'rows_per_page', 'content_length', 'excerpt_length', 'post_limit' ) as $arg ) {
            if ( ! isset( $args[$arg] ) ) {
                continue;
            }
            if ( false === ( $int_val = filter_var( $args[$arg], FILTER_VALIDATE_INT ) ) ) {
                $args[$arg] = Posts_Table_Args::$default_args[$arg];
            }
            // These can be a positive int or -1 only
            if ( 0 === $int_val || $int_val < -1 ) {
                $args[$arg] = -1;
            }
        }

        // Check bools
        foreach ( array( 'lightbox', 'shortcodes', 'lazy_load', 'cache', 'reset_button' ) as $arg ) {
            if ( ! isset( $args[$arg] ) ) {
                $args[$arg] = false;
            }
            $args[$arg] = filter_var( $args[$arg], FILTER_VALIDATE_BOOLEAN );
        }

        return $args;
    }

    public function sanitize_misc_settings( $args ) {
        if ( isset( $args['cache_expiry'] ) ) {
            $args['cache_expiry'] = filter_var( $args['cache_expiry'], FILTER_VALIDATE_INT, array( 'options' => array( 'default' => 6 ) ) );
        }
        return $args;
    }

    public function table_content_description() {
        printf(
            '<p>' .
            /* translators: %1: knowledge base link start, %2: knowledge base link end */
            __( 'All options on this page can be overridden by entering the corresponding option in the [posts_table] shortcode. See the %1$sfull list of shortcode options%2$s for details.', 'posts-table-pro' ) .
            '</p>',
            Util::format_link_open( Util::barn2_url( 'kb/posts-table-options/' ), true ),
            '</a>'
        );
    }

    public function support_links() {
        printf( '<p>%s | %s</p>',
            Util::format_link( $this->plugin->get_documentation_url(), __( 'Documentation', 'posts-table-pro' ), true ),
            Util::format_link( $this->plugin->get_support_url(), __( 'Support', 'posts-table-pro' ), true )
        );
    }

    private function read_more( $path ) {
        return ' ' . Util::barn2_link( $path );
    }

    private function get_registered_post_types() {
        $post_types = get_post_types();

        // Internal WP post types.
        $internal_post_types = array(
            'revision',
            'nav_menu_item',
            'custom_css',
            'customize_changeset',
            'oembed_cache',
            'wp_block'
        );

        // CPTs added by plugins which are not relevant (e.g. internal CPT) or not supported.
        $plugin_post_types_not_supported = array(
            'acf-field',
            'acf-field-group',
            'nf_sub',
            'edd_log',
            'edd_payment',
            'edd_discount',
            'product_variation',
            'shop_order_refund',
            'shop_coupon',
            'tribe-ea-record',
            'deleted_event'
        );

        foreach ( (array) apply_filters( 'posts_table_settings_page_post_types', array_merge( $internal_post_types, $plugin_post_types_not_supported ) ) as $type ) {
            unset( $post_types[$type] );
        }

        return $post_types;
    }

}
