<?php
/**
 * Plugin Name: Age Checker
 * Plugin URI: https://codecanyon.net/item/age-checker-for-wordpress/20626595
 * Description: Age verification plugin for WordPress
 * Version: 1.2.2
 * Author: Themeware
 * Author URI: http://www.themeware.net/
 * License: https://codecanyon.net/licenses/standard
 */

date_default_timezone_set( 'UTC' );
define( "AGC_VERSION", "1.2.2" );
define( "AGC_PLUGIN_DIR", plugin_dir_path( __FILE__ ) );
define( "AGC_PLUGIN_URL", plugins_url( '/', __FILE__ ) );

/**
 * New Settings Added
 *
 * Since 1.2.0
 * terms
 * post_types
 *
 * Since 1.1.5
 * logo
 * logo_width
 *
 * Since 1.1.0
 * restriction
 * pages
 * posts
 *
 * Since 1.0.1
 * exit
 * exit_url
 * form_button_exit
 */

// Default settings
$agc_default = array(
    'active' => false, // Plugin active status 
    'age' => 18, // Minimum age
    'method' => 'birthday', // Verification method
    'date_format' => 'dmy', 
	'exit' => false,
	'exit_url' => null,
    'cookie_life' => 10, // Days
    'search_engine' => false, // If search engine crawlers are shown age verfication screen
    'restriction' => 'all',
    'pages' => array(),
    'posts' => array(),
    'terms' => array(),
    'post_types' => array(),
    'logo' => null,
    'logo_width' => 200,
    'background' => 'color',
    'background_color' => '#ec5b5b',
    'background_image' => null,
    'background_image_overlay' => 0.5,
    'background_video' => null,
    'background_video_overlay' => 0.5,
    'form_title' => 'Confirm Your Age',
    'form_intro' => 'We require users to be %agc_age% years old or over, please enter your birthday.',
    'form_checkbox' => 'I confirm that I am %agc_age% years old or over',
	'form_button' => 'Submit',
	'form_button_exit' => 'Exit',
	'response_birthday_empty' => 'Please enter your birthday.',
	'response_birthday_date' => 'Oops it looks like that date doesn\'t exist.',
	'response_birthday_below' => 'Your age is below the minimum required.',
	'response_checkbox_empty' => 'Please confirm if you are %agc_age% years old or over.'
);

add_action( 'plugins_loaded', 'agc_load' );

function agc_load() {
        global $agc_default;
	$current_options = get_option( 'agc_options' );

	if ( $current_options == false ) {
		update_option( 'agc_options', $agc_default );
	}
	// Merge latest settings to older record in database
	else if( !isset( $current_options['post_types'] ) ) {
		update_option( 'agc_options', array_merge( $agc_default, $current_options ) );
	}
}

add_action( 'admin_enqueue_scripts', 'agc_admin_scripts' );

function agc_admin_scripts() {
	if( isset( $_GET['page'] ) && $_GET['page'] == 'age_checker' ) {
		wp_enqueue_media(); // Enables wordpress media modal
		wp_enqueue_style( 'agc_admin_jqueryui', AGC_PLUGIN_URL.'admin/assets/jquery-ui-1.12.1/jquery-ui.min.css', array(), '1.12.1' );
		wp_enqueue_style( 'agc_admin_spectrum', AGC_PLUGIN_URL.'admin/assets/css/spectrum.css', array(), '1.8.0' );
		wp_enqueue_style( 'agc_admin_style', AGC_PLUGIN_URL.'admin/assets/css/style.css', array(), AGC_VERSION );
		// This changes some of the screen style for compatibility with admin preview
		wp_enqueue_style( 'agc_admin_preview', AGC_PLUGIN_URL.'admin/assets/css/preview.css', array( 'agc_style' ), AGC_VERSION );
		wp_enqueue_style( 'agc_icomoon', AGC_PLUGIN_URL.'inc/assets/css/icomoon.css', array(), AGC_VERSION );
		wp_enqueue_style( 'agc_admin_chosen', AGC_PLUGIN_URL.'admin/assets/chosen/chosen.min.css', array(), '1.8.5' );
		wp_enqueue_style( 'agc_style', AGC_PLUGIN_URL.'inc/assets/css/style.css', array(), AGC_VERSION );
		// Adjustments for older versions of IE
		wp_enqueue_style( 'agc_style_old_ie', AGC_PLUGIN_URL.'inc/assets/css/old_ie.css', array( 'agc_style' ), AGC_VERSION );
		wp_style_add_data( 'agc_style_old_ie', 'conditional', 'lte IE 9' );
		wp_enqueue_script( 'agc_admin_jqueryui', AGC_PLUGIN_URL.'admin/assets/jquery-ui-1.12.1/jquery-ui.min.js', array( 'jquery' ), '1.12.1', true );
		wp_enqueue_script( 'agc_admin_spectrum', AGC_PLUGIN_URL.'admin/assets/js/spectrum.js', array( 'jquery' ), '1.8.0', true );
		wp_enqueue_script( 'agc_admin_chosen', AGC_PLUGIN_URL.'admin/assets/chosen/chosen.jquery.min.js', array( 'jquery' ), '1.8.5', true );
		wp_register_script( 'agc_admin_js', AGC_PLUGIN_URL.'admin/assets/js/js.js', array( 'jquery' ), AGC_VERSION, true );
		wp_localize_script( 'agc_admin_js', 'agc', array( 'url' => AGC_PLUGIN_URL ) );
		wp_enqueue_script( 'agc_admin_js' );
		wp_enqueue_script( 'agc_js', AGC_PLUGIN_URL.'inc/assets/js/js.js', array( 'jquery' ), AGC_VERSION, true );
	}
}

// Files required to initialize verification screen
add_action( 'wp_enqueue_scripts', 'agc_scripts', 0 );

function agc_scripts() {
	$current_page_id = get_queried_object_id();
	$post_terms = wp_get_post_terms( $current_page_id, get_taxonomies() );
	$post_type = get_post_type( $current_page_id );

	$load_scripts = true; 
	$options = get_option( 'agc_options' );
	$selected_pages = !empty( $options['pages'] ) ? $options['pages'] : array();
	$selected_posts = !empty( $options['posts'] ) ? $options['posts'] : array();
	$selected_terms = !empty( $options['terms'] ) ? $options['terms'] : array();
	
	$children = array();
	foreach( $selected_terms as $val ) {
		$term = get_term( $val );
		$children = array_merge( $children, get_term_children($val, $term->taxonomy) );
	} 
	$selected_terms = array_merge( $selected_terms, $children );

	$selected_post_types = !empty( $options['post_types'] ) ? $options['post_types'] : array();

	// Situations where age checker should be disabled
	if( !current_user_can( 'manage_options' ) || !isset( $_GET['agc_preview'] ) ) {
		if( $options['active'] == true ) {
			if( $options['restriction'] == 'exclusion' || $options['restriction'] == 'selection' ) {
				if( $options['restriction'] == 'exclusion' ) {
					$bool = false;
				}
				else if( $options['restriction'] == 'selection' ) {
					$bool = true;
					$load_scripts = false;
				}

				if( in_array( $current_page_id, array_merge( $selected_pages, $selected_posts ) ) ) {
					$load_scripts = $bool;
				}

				if( is_tag() || (function_exists('is_product_tag') && is_product_tag()) ) {
					if( in_array( get_queried_object_id(), $selected_terms ) ) {
						$load_scripts = $bool;
					}
				}
				else if( is_category() || (function_exists('is_product_category') && is_product_category()) ) { 
					if( in_array( get_queried_object_id(), $selected_terms ) ) {
						$load_scripts = $bool;
					}
				}
				else {
					if( !empty( $post_terms ) ) {
						foreach( $post_terms as $term ) {
							if( in_array( $term->term_id, $selected_terms ) ) {
								$load_scripts = $bool;
							}
						}
					}
				}

				if( in_array( $post_type, $selected_post_types ) ) {
					$load_scripts = $bool;
				}
			}

			if( $options['search_engine'] == false ) {
				if( isset( $_SERVER['HTTP_USER_AGENT'] ) && preg_match( '/bot|crawl|slurp|spider/i', $_SERVER['HTTP_USER_AGENT'] ) ) {
					$load_scripts = false;
				}
			}
		}
		else {
			$load_scripts = false;
		}
	}

	if( $load_scripts ) {
		ob_start();
    	include AGC_PLUGIN_DIR."inc/verify_template.php";
    	$screen = ob_get_clean();

		wp_enqueue_style( 'agc_icomoon', AGC_PLUGIN_URL.'inc/assets/css/icomoon.css', array(), AGC_VERSION );
		wp_enqueue_style( 'agc_style', AGC_PLUGIN_URL.'inc/assets/css/style.css', array(), AGC_VERSION );
		// Adjustments for older versions of IE
		wp_enqueue_style( 'agc_style_old_ie', AGC_PLUGIN_URL.'inc/assets/css/old_ie.css', array( 'agc_style' ), AGC_VERSION );
		wp_style_add_data( 'agc_style_old_ie', 'conditional', 'lte IE 9' );
		wp_register_script( 'agc_initialize', AGC_PLUGIN_URL.'inc/assets/js/initialize.js', array( 'jquery' ), AGC_VERSION );
		wp_localize_script( 'agc_initialize', 'agc', array( 'url' => AGC_PLUGIN_URL, 'admin_ajax' => admin_url( 'admin-ajax.php' ), 'age' => $options['age'], 'preview' => isset( $_GET['agc_preview'] ) ? 1 : 0, 'screen' => $screen ) );
		wp_enqueue_script( 'agc_initialize' );
		wp_enqueue_script( 'agc_js', AGC_PLUGIN_URL.'inc/assets/js/js.js', array( 'jquery' ), AGC_VERSION );
	}
}

add_action( 'admin_menu', 'agc_admin', 100 );

function agc_admin() {
	add_menu_page( 'Age Checker', 'Age Checker', 'manage_options', 'age_checker', 'agc_admin_output', 'dashicons-lock' );
}

function agc_admin_output() {
	include AGC_PLUGIN_DIR.'admin/admin_template.php';
}

// If logged in
add_action( 'wp_ajax_agc_edit', 'agc_edit' );

function agc_edit() {
        global $agc_default;
	check_ajax_referer( 'agc_edit', 'nonce' );

        if( current_user_can( 'manage_options' ) ) {
		$settings_required = array();
		$styling_required = array();
		$form_required = array();
		$resp_required = array();
		
		if( $_POST['agc_method'] == 'checkbox' ) {
			$form_required[] = 'agc_form_checkbox';
			$resp_required[] = 'agc_response_checkbox_empty';
		}

		else if( $_POST['agc_method'] == 'birthday' ) {
			$resp_required[] = 'agc_response_birthday_empty';
			$resp_required[] = 'agc_response_birthday_date';
			$resp_required[] = 'agc_response_birthday_below';
		}
		
		if( $_POST['agc_exit'] == true ) {
			$settings_required[] = 'agc_exit_url';
			$form_required[] = 'agc_form_button_exit';
		}

		if( $_POST['agc_logo'] != null ) {
			$styling_required[] = 'agc_logo_width';
		}
		
		if( $_POST['agc_background'] == 'color' ) {
			$styling_required[] = 'agc_background_color';
		}

		else if( $_POST['agc_background'] == 'image' ) {
			$styling_required[] = 'agc_background_image';
		}

		else if( $_POST['agc_background'] == 'video' ) {
			$styling_required[] = 'agc_background_video';
		}


		$required = array(
			'General' => array_merge( 
				array(
				'agc_active',
				'agc_age',
				'agc_method',
				'agc_date_format',
				'agc_cookie_life',
				'agc_search_engine',
				'agc_exit' ), $settings_required
			),
			'Restricted sections' => array( 'agc_restriction' ),
			'Styling' => array_merge(
				array( 'agc_background' ), $styling_required
			),
			'Form text' => array_merge(
				array( 'agc_form_title', 'agc_form_button' ), $form_required
			),
			'Response text' => $resp_required
		);

		$validate = array();

		foreach ( $required as $section => $array ) {
			foreach( $array as $name ) {
				if( !isset( $_POST[$name] ) || ( isset( $_POST[$name] ) && $_POST[$name] == null ) ) {
					if( $name == 'agc_background_image' || $name == 'agc_background_video' ) {
						$validate[$section][$name] = 'Please attach a file';
					}
					else {
						$validate[$section][$name] = 'Please enter a value';
					}
				}
			}
		}

		if( !isset( $validate['General']['agc_age'] ) ) {
			// If numbers haven't been entered
			if( !is_numeric( $_POST['agc_age'] ) ) {
				$validate['General']['agc_age'] = 'Please enter in digits only';
			}
		}

		if( !isset( $validate['General']['agc_cookie_life'] ) ) {
			// If numbers haven't been entered
			if( !is_numeric( $_POST['agc_cookie_life'] ) ) {
				$validate['General']['agc_cookie_life'] = 'Please enter in digits only';
			}
		}

		if( !empty( $_POST['agc_logo'] ) ) {
			$mime = get_post_mime_type( $_POST['agc_logo'] );
			if( !in_array( $mime,  array( 'image/jpeg', 'image/png', 'image/gif' ) ) ) {
				$validate['Styling']['agc_logo'] = 'Please upload a JPG, PNG or GIF image';
			}
		}

		if( !isset( $validate['Styling']['agc_background'] ) ) {
			if( $_POST['agc_background'] == 'image' ) {
				if( !isset( $validate['Styling']['agc_background_image'] ) ) {
					$mime = get_post_mime_type( $_POST['agc_background_image'] );
					if( !in_array( $mime,  array( 'image/jpeg', 'image/png', 'image/gif' ) ) ) {
						$validate['Styling']['agc_background_image'] = 'Please upload a JPG, PNG or GIF image';
					}
				}
			}
			else if( $_POST['agc_background'] == 'video' ) {
				if( !isset( $validate['Styling']['agc_background_video'] ) ) {
					$mime = get_post_mime_type( $_POST['agc_background_video'] );
					if( $mime != 'video/mp4' ) {
						$validate['Styling']['agc_background_video'] = 'Please upload a MP4 video';
					}
				}
			}
		}

		if( !empty( $validate ) ) {
			die( json_encode( array(
				'result' => 'validate',
				'output' => json_encode( $validate )
			) ) );
		}

                foreach( $agc_default as $key => $val ) {
                        $options[$key] = $_POST['agc_'.$key];
                }

                $current_option = get_option( 'agc_options' );
                if( $current_option != $options ) {
                	if( update_option( 'agc_options', $options ) ) {
	                        die( json_encode( array(
	                                "result" => "success",
	                                "output" => ""
	                        ) ) );
	                }
                }
                else {
                	die( json_encode( array(
                                "result" => "success",
                                "output" => ""
                        ) ) );
                }
	}
}

// If logged in
add_action( 'wp_ajax_agc_verify', 'agc_verify' );
// Not logged in
add_action( 'wp_ajax_nopriv_agc_verify', 'agc_verify' );

function agc_verify() {
	$options = stripslashes_deep( get_option( 'agc_options' ) );

	// Checkbox method
	if( $options['method'] == 'checkbox' ) {
		if( !isset( $_POST['agc_check'] ) ) {
			die( json_encode( array(
				"result" => "validate",
				"output" => str_replace( '%agc_age%', $options['age'], $options['response_checkbox_empty'] )
			) ) );
		}
	}

	// Birthday method
	else if ( $options['method'] == 'birthday' ) {
		if( $_POST['agc_day'] == '' || $_POST['agc_month'] == '' || $_POST['agc_year'] == '' ) {
			die( json_encode( array(
				"result" => "validate",
				"output" => str_replace( '%agc_age%', $options['age'], $options['response_birthday_empty'] )
			) ) );
		}
		else if( !checkdate( $_POST['agc_month'], $_POST['agc_day'], $_POST['agc_year'] ) ) {
			die( json_encode( array(
				"result" => "validate",
				"output" => str_replace( '%agc_age%', $options['age'], $options['response_birthday_date'] )
			) ) );
		}
		else {
			$birthday = sprintf( "%04d-%02d-%02d", $_POST['agc_year'], $_POST['agc_month'], $_POST['agc_day'] );
			$today = new DateTime();
			$min = $today->modify( "-{$options['age']} year" )->format( "Y-m-d" );

			// Check if after the minimum age date
			if( $birthday > $min ) {
				die( json_encode( array(
					"result" => "validate",
					"output" => str_replace( '%agc_age%', $options['age'], $options['response_birthday_below'] )
				) ) );
			}
		}
	}
	
	// Only set the cookie if not previewing
	if( !current_user_can( 'manage_options' ) || !isset( $_POST['agc_preview'] ) ) {
		// Cookie name has age appended incase minimum age changes
		setcookie( 'agc_verified_'.$options['age'], true, time() + ( 86400 * $options['cookie_life'] ), '/' );
	}

	die( json_encode( array(
                "result" => "success",
                "output" => ""
        ) ) );
}
