<?php

/**
 * @copyright 	Copyright (c) 2009-2017 Ryan Demmer. All rights reserved
 * @license   	GNU/GPL 2 or later - http://www.gnu.org/licenses/old-licenses/gpl-2.0.html
 * JCE is free software. This version may have been modified pursuant
 * to the GNU General Public License, and as distributed it includes or
 * is derivative of works licensed under the GNU General Public License or
 * other free or open source software licenses
 */
defined('_JEXEC') or die('RESTRICTED');

require_once(WF_EDITOR . '/extensions/filesystem/joomla.php');

class WFServerFileSystem extends WFJoomlaFileSystem
{
    
    public function __construct($config = array())
    {        
        parent::__construct($config);

        $this->setProperties(array(
            'base' => $this->getRootUrl(),
        ));
    }
    
    private function getRootUrl() {
    	$wf = WFEditorPlugin::getInstance();
    	
    	$url = $wf->getParam('filesystem.server.root_url', JURI::root(true));
    	
    	// trim trailing slash
    	$url = rtrim($url, "/");
        
        return $url;
    }
    
    private function getBasePath() {
    	$wf = WFEditorPlugin::getInstance();
        
        return $wf->getParam('filesystem.server.base_dir', JPATH_SITE);
    }

    /**
     * Get the base directory.
     *
     * @return string base dir
     */
    public function getBaseDir()
    {
        $base = $this->getBasePath();
        
        return WFUtility::makePath($base, $this->getRootDir());
    }

    /**
     * Get the full base url.
     *
     * @return string base url
     */
    public function getBaseURL()
    {
        $wf = WFEditorPlugin::getInstance();
        
        $root = $wf->getParam('filesystem.server.root_url', JURI::root(true));

        $root = rtrim($root, "/");
        
        return $root . '/' . $this->getRootDir();
    }

    /**
     * Return the full user directory path. Create if required.
     *
     * @param string	The base path
     *
     * @return Full path to folder
     */
    public function getRootDir()
    {
        static $root;

        if (!isset($root)) {
            $wf = WFEditorPlugin::getInstance();
            $root = parent::getRootDir();

            // Revert to default if empty (overwrite allow root parameter)
            if (empty($root)) {
                $root = 'images';
            }

            // Create the folder
            $full = WFUtility::makePath($this->getBasePath(), $root);

            if (!JFolder::exists($full)) {
                $this->folderCreate($full);
            }

            // Fallback
            $root = JFolder::exists($full) ? $root : 'images';
        }

        JDispatcher::getInstance()->trigger('onWfFileSystemGetRootDir', array(&$root));

        return $root;
    }
    
    public function getFiles($relative, $filter = '') {
    	$files = parent::getFiles($relative, $filter);
    	
    	$x = 0;
    	
    	array_walk($files, function(&$file, $key) {
    		$url = trim($file['url'], "/");
    		$file['url'] = $this->getRootUrl() . '/' . $url;

    		$file['properties'] = $this->getFileDetails($file['id'], $key);
    	});

    	return $files;
    }
    
    /**
     * Get a files properties.
     *
     * @return array Array of properties
     *
     * @param string $file File relative path
     */
    public function getFileDetails($file, $count = 1)
    {
        $data = parent::getFileDetails($file, $count = 1);

        $data['preview'] = $this->getBaseUrl() . '/' . rawurldecode($file);
        
        if (preg_match('#\.(jpg|jpeg|bmp|gif|tiff|png)#i', $file)) {
        	$data['preview'] .= '?' . $data['modified'];
        }
        
        return $data;
    }
    
    public function isMatch($needle, $haystack)
    {                
        return $needle == $this->getRootUrl() . '/' . ltrim($haystack, "/");
    }
}