<?php

/**
 * @copyright    Copyright (C) 2005 - 2017 Ryan Demmer. All rights reserved
 * @author        Ryan Demmer
 * @license      GNU/GPL
 * JCE is free software. This version may have been modified pursuant
 * to the GNU General Public License, and as distributed it includes or
 * is derivative of works licensed under the GNU General Public License or
 * other free or open source software licenses
 */
// no direct access
defined('_JEXEC') or die();

require __DIR__ . '/loader.php';

class WFS3Client
{
    private $key;

    private $secret;
    
    private $region;
    
    private $ssl;

    private $bucket;

    private $endpoint;

    public function __construct($config = array())
    {
        foreach($config as $key => $value) {
            $this->$key = $value;
        }
    }
    
    private function getClient() {
    	// create instance
        $configuration = new Akeeba\Engine\Postproc\Connector\S3v4\Configuration($this->key, $this->secret, 'v4', $this->region);
        $configuration->setSSL((bool) $this->ssl);
    	
    	$client = new Akeeba\Engine\Postproc\Connector\S3v4\Connector($configuration);
    	
    	return $client;
    }

    /**
     * Get object information.
     *
     * @param string $bucket     Bucket name
     * @param string $uri        Object URI
     * @param bool   $returnInfo Return response information
     *
     * @return mixed | false
     */
    public function getObjectInfo($uri, $returnInfo = true)
    {
        $client = $this->getClient();
        
        $request = new Akeeba\Engine\Postproc\Connector\S3v4\Request('HEAD', $this->bucket, $uri, $client->getConfiguration());
        $response = $request->getResponse();

        if (!$response->error->isError() && $response->code !== 200 && $response->code !== 404) {
            $response->error = new Akeeba\Engine\Postproc\Connector\S3v4\Response\Error(
                $response->code,
                "Unexpected HTTP status {$response->code}"
            );
        }

        if ($response->error->isError()) {
            throw new Akeeba\Engine\Postproc\Connector\S3v4\Exception\CannotGetFile(
                sprintf(__METHOD__ . '(): [%s] %s', $response->error->getCode(), $response->error->getMessage()),
                $response->error->getCode()
            );
        }

        return $response->code == 200 ? $returnInfo ? $response->headers : true : false;
    }

    /**
     * Get object or bucket Access Control Policy.
     *
     * @param string $bucket Bucket name
     * @param string $uri    Object URI
     *
     * @return mixed | false
     */
    public function getAccessControlPolicy($uri = '')
    {
        $client = $this->getClient();
        
        $request = new Akeeba\Engine\Postproc\Connector\S3v4\Request('GET', $this->bucket, $uri, $client->getConfiguration());

        $request->setParameter('acl', null);

        $response = $request->getResponse();

        if (!$response->error->isError() && $response->code !== 200) {
            $response->error = new Akeeba\Engine\Postproc\Connector\S3v4\Response\Error(
                $response->code,
                "Unexpected HTTP status {$response->code}"
            );
        }

        if ($response->error->isError()) {
            throw new Akeeba\Engine\Postproc\Connector\S3v4\Exception\CannotGetFile(
                sprintf(__METHOD__ . '(): [%s] %s', $response->error->getCode(), $response->error->getMessage()),
                $response->error->getCode()
            );
        }

        $acp = array();

        if ($response->hasBody() && isset($response->body->Owner, $response->body->Owner->ID)) {
            $acp['owner'] = array(
                'id' => (string) $response->body->Owner->ID,
            );
        }

        if ($response->hasBody() && isset($response->body->AccessControlList)) {
            $acp['acl'] = array();

            foreach ($response->body->AccessControlList->Grant as $grant) {
                foreach ($grant->Grantee as $grantee) {
                    if (isset($grantee->ID)) { // CanonicalUser
                        $acp['acl'][] = array(
                            'type' => 'CanonicalUser',
                            'id' => (string) $grantee->ID,
                            'permission' => (string) $grant->Permission,
                        );
                    } elseif (isset($grantee->EmailAddress)) { // AmazonCustomerByEmail
                        $acp['acl'][] = array(
                            'type' => 'AmazonCustomerByEmail',
                            'email' => (string) $grantee->EmailAddress,
                            'permission' => (string) $grant->Permission,
                        );
                    } elseif (isset($grantee->URI)) { // Group
                        $acp['acl'][] = array(
                            'type' => 'Group',
                            'uri' => (string) $grantee->URI,
                            'permission' => (string) $grant->Permission,
                        );
                    } else {
                        continue;
                    }
                }
            }
        }

        return $acp;
    }

    public function getAuthenticatedURL($uri, $lifetime = null, $https = false)
    {
        $client = $this->getClient();
        
        try {
            $url = $client->getAuthenticatedURL($this->bucket, $uri, $lifetime, $https);
        } catch (Exception $e) {
            return false;
        }

        return $url;
    }

    public function getObjectURL($uri)
    {
        $host = $this->ssl ? 'https://' : 'http://';
        $base = $this->bucket . '.' . $this->endpoint;

        if ($this->cname && $this->cname !== $this->bucket) {
            $base = $this->cname;
        }

        $path = trim($uri, '/');

        // create public url
        $url = $host . $base . '/' . $path;

        // check ACL
        $acl = $this->getAccessControlPolicy($path);

        $public = false;

        if (!empty($acl['acl'])) {
            foreach ($acl['acl'] as $rule) {
                if ($rule['permission'] === 'READ' && (!empty($rule['uri']) && $rule['uri'] === 'http://acs.amazonaws.com/groups/global/AllUsers')) {
                    $public = true;
                    break;
                }
            }
        }

        // get authenticated url if not public
         if (!$public) {
            $url = $this->getAuthenticatedURL($path, $this->timeout, $this->ssl);
        }

        return $url;
    }

    public function getBucket($prefix = null, $marker = null, $maxKeys = null, $delimiter = '/', $returnCommonPrefixes = false)
    {
        $client = $this->getClient();
        
        $results = array();

        try {
            $results = $client->getBucket($this->bucket, $prefix, $marker, $maxKeys, $delimiter, $returnCommonPrefixes);
        } catch (Exception $e) {
        }

        return $results;
    }

    public function getObject($uri)
    {
        $client = $this->getClient();
        
        try {
            $object = $client->getObject($this->bucket, $uri);
        } catch (Exception $e) {
            return false;
        }

        return $object;
    }

    public function putObject($input, $uri, $acl = 'private', $requestHeaders = array())
    {        	        
        $client = $this->getClient();
        
        if (is_file($input)) {
            $input = Akeeba\Engine\Postproc\Connector\S3v4\Input::createFromFile($input);
        } else {            
            $input = Akeeba\Engine\Postproc\Connector\S3v4\Input::createFromData($input);
        }

        try {
            $client->putObject($input, $this->bucket, $uri, $acl);
        } catch (Exception $e) {
            return false;
        }

        return true;
    }

    public function deleteObject($uri)
    {
        $client = $this->getClient();
        
        try {
            $client->deleteObject($this->bucket, $uri);
        } catch (Exception $e) {
            return false;
        }

        return true;
    }

    public function copyObject($source, $destination)
    {
        try {
            $source = $this->getObject($source);

            if (!$source) {
                return false;
            }

            return $this->putObject($source, $destination);
        } catch (Exception $e) {
            return false;
        }

        return true;
    }

    public function renameObject($source, $destination)
    {
        try {
            if ($this->copyObject($source, $destination)) {
                return $this->deleteObject($source);
            }
        } catch (Exception $e) {
            return false;
        }

        return true;
    }
}