<?php

namespace DevOwl\RealThumbnailGenerator\Vendor\DevOwl\RealUtils\cross;

use DevOwl\RealThumbnailGenerator\Vendor\DevOwl\RealUtils\Core;
// @codeCoverageIgnoreStart
\defined('ABSPATH') or die('No script kiddies please!');
// Avoid direct file request
// @codeCoverageIgnoreEnd
/**
 * Cross-selling for Real Cookie Banner.
 *
 * @see https://app.clickup.com/2088/v/dc/218-92/218-1114
 */
class CrossRealCookieBanner extends \DevOwl\RealThumbnailGenerator\Vendor\DevOwl\RealUtils\cross\AbstractCrossSelling
{
    const SLUG = 'real-cookie-banner';
    const FILE_LITE = 'real-cookie-banner/index.php';
    const FILE_PRO = 'real-cookie-banner-pro/index.php';
    const NON_GDPR_PLUGINS = ['luckywp-cookie-notice-gdpr', 'ninja-gdpr-compliance', 'ninja-gdpr', 'gdpr-compliance-by-supsystic', 'surbma-gdpr-proof-google-analytics', 'easy-wp-cookie-popup', 'smart-cookie-kit', 'italy-cookie-choices', 'shapepress-dsgvo', 'uk-cookie-consent'];
    const GDPR_PLUGINS = ['cookiebot', 'iubenda-cookie-law-solution', 'gdpr-cookie-compliance', 'cookie-law-info', 'gdpr-cookie-consent', 'cookie-notice', 'pixelmate'];
    const SALE_READY_GDPR_PLUGINS = ['borlabs-cookie', 'complianz-gdpr', 'complianz-gdpr-premium'];
    const GDPR_LANGUAGES = ['de'];
    /**
     * Documented in AbstractCrossSelling.
     *
     * @codeCoverageIgnore
     */
    public function getSlug()
    {
        return self::SLUG;
    }
    /**
     * Documented in AbstractCrossSelling.
     */
    public function skip()
    {
        // Always skip when RCB is installed
        $handler = \DevOwl\RealThumbnailGenerator\Vendor\DevOwl\RealUtils\Core::getInstance()->getCrossSellingHandler();
        if ($handler->isInstalled(self::FILE_LITE) || $handler->isInstalled(self::FILE_PRO)) {
            return \true;
        }
        // Do not skip when a GDPR plugin is installed
        $hasGdprPluginInstalled = $this->hasGdprPluginInstalled(\array_merge(self::NON_GDPR_PLUGINS, self::GDPR_PLUGINS, self::SALE_READY_GDPR_PLUGINS));
        if ($hasGdprPluginInstalled !== \false) {
            return \false;
        }
        // Do not skip when the blog needs a GDPR solution
        return !$this->hasGdprBlog();
    }
    /**
     * Documented in AbstractCrossSelling.
     *
     * @codeCoverageIgnore
     */
    public function getMeta()
    {
        $gdprCompliantTexts = $this->getTitleAndDescription();
        return ['gdpr-compliant' => ['title' => $gdprCompliantTexts[0], 'image' => $this->getAssetsUrl(__('cookie-banner-frontend.png', REAL_UTILS_TD)), 'description' => $gdprCompliantTexts[1], 'link' => __('https://devowl.io/go/real-cookie-banner?source=cross-rcb', REAL_UTILS_TD)]];
    }
    /**
     * Get the popover title and description depending on the installed solution.
     *
     * @codeCoverageIgnore
     */
    public function getTitleAndDescription()
    {
        $solution = $this->hasGdprPluginInstalled(self::SALE_READY_GDPR_PLUGINS);
        if ($solution !== \false) {
            $strongSolution = \sprintf('<strong>%s</strong>', $solution);
            return [\sprintf(
                // translators:
                __('Is %s really set up in compliance with GDPR?', REAL_UTILS_TD),
                $strongSolution
            ), \sprintf(
                // translators:
                __('%s is not easy to set up so that you provide all cookies on your website completely and correctly. How about a simpler alternative like Real Cookie Banner?', REAL_UTILS_TD),
                $strongSolution
            )];
        }
        $solution = $this->hasGdprPluginInstalled(self::GDPR_PLUGINS);
        if ($solution !== \false) {
            $strongSolution = \sprintf('<strong>%s</strong>', $solution);
            return [__('Is your website GDPR compliant?', REAL_UTILS_TD), \sprintf(
                // translators:
                __('You use %s as cookie banner. This plugin may not meet all legal requirements in its default configuration. Real Cookie Banner could be a better alternative for you!', REAL_UTILS_TD),
                $strongSolution
            )];
        }
        $solution = $this->hasGdprPluginInstalled(self::NON_GDPR_PLUGINS);
        if ($solution !== \false) {
            $strongSolution = \sprintf('<strong>%s</strong>', $solution);
            return [__('Is your website GDPR compliant?', REAL_UTILS_TD), \sprintf(
                // translators:
                __('You use %s as cookie banner. This plugin is not a full opt-in consent cookie banner as required by the ePrivacy Directive in the EU. Real Cookie Banner could be a better alternative for you!', REAL_UTILS_TD),
                $strongSolution
            )];
        }
        return [__('Is your website GDPR compliant?', REAL_UTILS_TD), __('Websites targeting users from the EU and setting non-essential cookies require an opt-in cookie banner. We recommend Real Cookie Banner as ePrivacy Directive and GDPR compliant solution for WordPress.', REAL_UTILS_TD)];
    }
    /**
     * Check if the current WordPress instance has a plugin installed, which
     * is not GDPR-compliant.
     *
     * @param string[] $plugins Use one of the class constants
     */
    public function hasGdprPluginInstalled($plugins)
    {
        foreach ($plugins as $slug) {
            $file = \DevOwl\RealThumbnailGenerator\Vendor\DevOwl\RealUtils\Core::getInstance()->getCrossSellingHandler()->isInstalled($slug, \true);
            if ($file !== \false) {
                return get_plugin_data($file)['Name'];
            }
        }
        return \false;
    }
    /**
     * Check if the current WordPress instance generally needs a GDPR solution.
     */
    public function hasGdprBlog()
    {
        $current = get_bloginfo('language');
        foreach (self::GDPR_LANGUAGES as $lang) {
            if (\substr($current, 0, \strlen($lang)) === $lang) {
                return \true;
            }
        }
        return \false;
    }
}
