<?php

class HappyForms_Message_Admin_Edit {

	private static $instance;

	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		self::$instance->hook();

		return self::$instance;
	}

	public function hook() {
		$controller = happyforms_get_message_controller();
		$post_type = $controller->post_type;

		add_action( 'load-post.php', array( $this, 'reply_and_mark' ) );
		add_action( 'load-post.php', array( $this, 'handle_action' ) );
		add_action( 'admin_head', array( $this, 'screen_title' ) );
		add_action( 'admin_head', array( $this, 'highlight_menu' ) );
		add_action( 'edit_form_after_title', array( $this, 'edit_screen' ) );
		add_action( 'do_meta_boxes', array( $this, 'setup_metaboxes' ) );
		add_filter( 'admin_footer_text', 'happyforms_admin_footer' );
		add_action( 'happyforms_message_edit_field', array( $this, 'edit_field' ), 10, 4 );
		add_action( 'post_submitbox_minor_actions', array( $this, 'publish_box_meta_minor' ) );
		add_action( 'post_submitbox_misc_actions', array( $this, 'publish_box_meta_misc' ) );
		add_filter( 'screen_options_show_screen', array( $this, 'show_screen_options' ) );
		add_action( 'save_post_' . $post_type, array( $this, 'save_post' ) );
		add_filter( 'happyforms_message_sanitize_field', array( $this, 'sanitize_field' ), 10, 3 );
		add_filter( 'happyforms_message_field_is_editable', array( $this, 'field_is_editable' ), 10, 3 );
		add_filter( 'mce_external_plugins', array( $this, 'mce_plugin' ) );
		add_filter( 'happyforms_dashboard_data', array( $this, 'dashboard_data' ) );
	}

	/**
	 * Handles a reply-and-mark-as-read link.
	 *
	 * @return void
	 */
	public function reply_and_mark() {
		$message_controller = happyforms_get_message_controller();
		$action = $message_controller->reply_and_mark_action;

		if ( ! isset( $_REQUEST[ $action ] )
			|| 1 !== intval( $_REQUEST[ $action ] ) ) {
			return;
		}

		$post = get_post( $_REQUEST['post'] );

		if ( ! current_user_can( 'happyforms_manage_activity' ) ) {
			wp_die( __( 'Sorry, you are not allowed to access this page.' ), 403 );
		}

		if ( $message_controller->post_type !== $post->post_type ) {
			return;
		}

		happyforms_update_meta( $post->ID, 'read', 1 );
		happyforms_get_message_controller()->update_counters();

		$form_controller = happyforms_get_form_controller();
		$form = $form_controller->get( happyforms_get_meta( $post->ID, 'form_id', true ) );

		if ( ! $form ) {
			return;
		}

		$email_part = $form_controller->get_first_part_by_type( $form, 'email' );
		$email_part_id = $email_part['id'];

		if ( ! $email_part ) {
			return;
		}

		$message = $message_controller->get( $post->ID );
		$email = happyforms_get_message_part_value( $message['parts'][$email_part_id], $email_part );
		$url = "mailto: {$email}";

		if ( wp_redirect( $url ) ) {
			exit;
		}
	}

	public function handle_action() {
		$controller = happyforms_get_message_controller();
		$action = $controller->mark_action;

		if ( ! isset( $_GET['action'] )
			|| $action !== $_GET['action']
			|| ! isset( $_GET['_wpnonce'] )
			|| ! isset( $_GET['status'] ) ) {

			return;
		}

		$nonce = $_GET['_wpnonce'];
		$status = $_GET['status'];

		if ( ! isset( $_GET['post'] ) ) {
			return;
		}

		$response_id = $_GET['post'];

		if ( ! wp_verify_nonce( $nonce, "{$action}-{$response_id}" ) ) {
			return;
		}

		$response = $controller->get( $response_id );

		if ( ! $response ) {
			return;
		}

		if ( ! in_array( $status, array( 'read', 'unread', 'spam', 'not_spam' ) ) ) {
			return;
		}

		wp_untrash_post( $response_id );

		$status_value = '';

		if ( in_array( $status, array( 'read', 'unread', 'spam' ) ) ) {
			$status_values = array( 'read' => 1, 'unread' => '', 'spam' => 2 );
			$status_value = $status_values[$status];
			$current_status = happyforms_get_meta( $response_id, 'read', true );

			if ( 2 !== $current_status && 2 === $status_value ) {
				happyforms_update_meta( $response_id, 'previously_read', $current_status );
			}
		} elseif( happyforms_meta_exists( $response_id, 'previously_read' ) ) {
			$status_value = happyforms_get_meta( $response_id, 'previously_read', true );
		}

		happyforms_update_meta( $response_id, 'read', $status_value );

		$controller->update_counters();

		if ( wp_redirect( wp_get_referer() ) ) {
			exit;
		}
	}

	public function screen_title() {
		global $post, $title;

		$title = __( 'Edit Activity', 'happyforms-ugprade' );
	}

	public function highlight_menu() {
		global $parent_file, $submenu_file, $submenu;

		$parent_file = 'happyforms';
		$submenu_file = 'edit.php?post_type=happyforms-message';
	}

	/**
	 * Action: output custom markup for the
	 * Message Edit admin screen.
	 *
	 * @since 1.0
	 *
	 * @hooked action edit_form_after_title
	 *
	 * @param WP_Post $post The message post object.
	 *
	 * @return void
	 */
	public function edit_screen( $post ) {
		global $message, $form;

		$message = happyforms_get_message_controller()->get( $post->ID );
		$form = happyforms_get_form_controller()->get( $message['form_id'] );

		require_once( happyforms_get_include_folder() . '/templates/admin-message-edit.php' );
	}

	public function setup_metaboxes( $post_type ) {
		global $wp_meta_boxes;

		$publish_metabox = $wp_meta_boxes[$post_type]['side']['core']['submitdiv'];
		$publish_metabox['title'] = __( 'Save', 'happyforms' );

		$wp_meta_boxes[$post_type] = array(
			'side' => array(
				'core' => array(
					'submitdiv' => $publish_metabox
				)
			)
		);

		add_meta_box(
			'happyforms-message-details',
			__( 'Details', 'happyforms' ),
			array( $this, 'metabox_message_details' ),
			$post_type,
			'side',
			'low'
		);

		add_meta_box(
			'happyforms-message-send-email',
			__( 'Email Confirmation', 'happyforms' ),
			array( $this, 'metabox_message_send_email' ),
			$post_type,
			'side',
			'low'
		);
	}

	public function metabox_message_details( $post, $metabox ) {
		$submitted = sprintf(
			__( '%1$s at %2$s', 'happyforms' ),
			date_i18n( __( 'M j, Y' ), strtotime( $post->post_date ) ),
			date_i18n( __( 'H:i' ), strtotime( $post->post_date ) )
		);
		?>
		<div class="misc-pub-section curtime misc-pub-curtime">
			<span id="timestamp">
				<?php _e( 'Submitted on', 'happyforms' ); ?>: <b><?php echo $submitted; ?></b>
			</span>
		</div>
		<?php do_action( 'happyforms_response_metabox_details', $post->ID );
	}

	public function metabox_message_send_email( $post, $metabox ) {
		$action = happyforms_get_message_controller()->send_email_action;
		$url = admin_url( 'admin-ajax.php' );
		$url = wp_nonce_url( $url, $action );
		$url = add_query_arg( array(
			'action' => $action,
			'response_id' => $post->ID,
		), $url );
		?>
		<div class="happyforms-metabox-content">
			<p class="post-attributes-label-wrapper">
				<label for="happyforms-message-send-email-field"><?php _e( 'Email address', 'happyforms' ); ?></label>
			</p>
			<input type="text" name="" id="happyforms-message-send-email-field" class="widefat">
			<p class="post-attributes-help-text"><?php _e( 'Send to multiple email addresses by separating each with a comma.' ); ?></p>
		</div>
		<div class="happyforms-metabox-actions">
			<span class="spinner"></span>
			<input type="button" class="button button-primary button-large" id="happyforms-message-send-email-submit" value="<?php _e( 'Send Email', 'happyforms' ); ?>" data-url="<?php echo $url; ?>" />
			<div class="clear"></div>
		</div>
		<?php
	}

	public function edit_field( $value, $part, $message, $form ) {
		$visible = apply_filters( 'happyforms_message_part_visible', true, $part );

		if ( ! $visible ) {
			return;
		}

		switch( $part['type'] ) {
			case 'attachment':
			case 'payments':
				include( happyforms_get_include_folder() . '/templates/partials/admin-message-edit-text.php' );
				break;
			case 'signature':
				include( happyforms_get_include_folder() . '/templates/partials/admin-message-edit-signature-disabled.php' );
				break;
			case 'legal':
			case 'email_integration':
				include( happyforms_get_include_folder() . '/templates/partials/admin-message-edit-input-text-disabled.php' );
				break;
			case 'table':
			case 'multi_line_text':
				include( happyforms_get_include_folder() . '/templates/partials/admin-message-edit-input-textarea.php' );
				break;
			case 'rich_text':
				include( happyforms_get_include_folder() . '/templates/partials/admin-message-edit-input-editor.php' );
				break;
			default:
				include( happyforms_get_include_folder() . '/templates/partials/admin-message-edit-input-text.php' );
				break;
		}
	}

	public function publish_box_meta_minor() {
		global $post, $form;

		$status = happyforms_get_meta( $post->ID, 'read', true );

		if ( 2 != $status ) {
			happyforms_update_meta( $post->ID, 'read', 1 );
			happyforms_get_message_controller()->update_counters();
			$status = 1;
		}

		$status_label = '';

		switch ( $status ) {
			case 1:
				$status_label = __( 'Read', 'happyforms' );
				break;
			case 2:
				$status_label = __( 'Spam', 'happyforms' );
				break;
			default:
				$status_label = __( 'Unread', 'happyforms' );
				break;
		}
		?>
		<div class="misc-pub-section misc-pub-comment-status" id="comment-status">
			<fieldset id="comment-status-radio">
				<legend class="screen-reader-text"><?php _e( 'Activity status', 'happyforms' ); ?></legend>
				<label>
					<input type="radio" name="response_status" value="read" <?php checked( $status, 1 ); ?>><?php _e( 'Read', 'happyforms' ); ?>
				</label><br>
				<label>
					<input type="radio" name="response_status" value="unread" <?php checked( $status, '' ); ?>><?php _e( 'Unread', 'happyforms' ); ?>
				</label><br>
				<label>
					<input type="radio" name="response_status" value="spam" <?php checked( $status, 2 ); ?>><?php _e( 'Spam', 'happyforms' ); ?>
				</label>
			</fieldset>
		</div>
		<?php
	}

	public function publish_box_meta_misc() {
		global $form;

		?>
		<div class="misc-pub-section misc-pub-response-to">
			<?php _e( 'In response to:', 'happyforms' ); ?>

			<?php
			if ( 'publish' === $form['post_status'] && current_user_can( 'happyforms_manage_forms' ) ) {
				printf(
					'<b><a href="%s">%s</a></b>',
					happyforms_get_form_edit_link( $form['ID'] ),
					$form['post_title']
				);
			} else {
				echo "<b>{$form['post_title']}</b>";
			}
			?>
		</div>
		<?php
	}

	public function show_screen_options() {
		$screen = get_current_screen();
		$post_type = happyforms_get_message_controller()->post_type;

		if ( $screen->id === $post_type ) {
			return false;
		}
	}

	private function get_field_allowed_tags() {
		$tags = array(
			'br' => array(),
			'b' => array(),
			'strong' => array(),
			'i' => array(),
			'em' => array(),
			'ul' => array(),
			'ol' => array(),
			'li' => array(),
			'p' => array(),
			'a' => array( 'href' => array() ),
			'pre' => array(),
			'hr' => array(),
			'u' => array(),
			'strike' => array(),
			'del' => array(),
			'blockquote' => array(),
		);

		return $tags;
	}

	public function save_post( $post_id ) {
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		if ( ! isset( $_POST['action'] ) || ( 'editpost' !== $_POST['action'] ) ) {
			return;
		}

		if ( ! isset( $_POST['parts'] ) ) {
			return;
		}

		$parts = $_POST['parts'];

		foreach( $parts as $part_id => $part ) {
			$type = $part['type'];
			$value = $part['value'];
			$is_editable = apply_filters( 'happyforms_message_field_is_editable', true, $value, $type, $post_id );

			if ( ! $is_editable ) {
				continue;
			}

			$class = happyforms_get_part_library()->get_part( $type );
			$value = apply_filters( 'happyforms_message_sanitize_field', $value, $value, $type, $post_id );

			happyforms_update_meta( $post_id, $part_id, $value );
		}

		if ( isset( $_POST['response_status'] )
			&& in_array( $_POST['response_status'], array( 'read', 'unread', 'spam' ) ) ) {

			$status = $_POST['response_status'];
			$status_values = array( 'read' => 1, 'unread' => '', 'spam' => 2 );
			$status_value = $status_values[$status];
			$current_status = happyforms_get_meta( $post_id, 'read', true );

			if ( 2 !== $current_status && 2 === $status_value ) {
				happyforms_update_meta( $post_id, 'previously_read', $current_status );
			}

			happyforms_update_meta( $post_id, 'read', $status_value );
			happyforms_get_message_controller()->update_counters();

			if ( in_array( $status, array( 'unread', 'spam' ) ) ) {
				wp_redirect( wp_get_original_referer() );
				exit;
			}
		}
	}

	public function sanitize_field( $value, $original_value, $type ) {
		switch( $type ) {
			case 'table':
			case 'rich_text':
				$value = wp_kses( $original_value, $this->get_field_allowed_tags() );
				$value = str_replace( array( "\r\n", "\n", "\r" ), '<br>', $value );
				$value = wp_slash( $value );
				break;
			default:
				$value = sanitize_text_field( $original_value );
				break;
		}

		return $value;
	}

	public function field_is_editable( $is_editable, $value, $type ) {
		$not_editable = in_array( $type, array(
			'attachment',
			'signature',
			'legal',
		) );

		return ! $not_editable;
	}

	public function mce_plugin( $plugins ) {
		$plugins = array(
			'hfcode' => happyforms_get_plugin_url() . 'inc/assets/js/admin/editor-plugins.js',
		);

		return $plugins;
	}

	public function dashboard_data( $data ) {
		$data['textNoticeDismiss'] = __( 'Dismiss this notice.', 'happyforms' );

		return $data;
	}

}

HappyForms_Message_Admin_Edit::instance();
