<?php

class HappyForms_Part_Signature extends HappyForms_Form_Part {

	public $type = 'signature';
	public $template_id = 'happyforms-signature-template';

	public function __construct() {
		$this->label = __( 'Signature', 'happyforms' );
		$this->description = __( 'For requiring authorization before accepting submission.', 'happyforms' );

		add_filter( 'happyforms_message_part_value', array( $this, 'message_part_value' ), 10, 4 );
		add_filter( 'happyforms_email_part_value', array( $this, 'email_part_value' ), 10, 5 );
		add_filter( 'happyforms_part_class', array( $this, 'html_part_class' ), 10, 3 );
		add_filter( 'happyforms_frontend_dependencies', array( $this, 'script_dependencies' ), 10, 2 );
	}

	/**
	 * Get all part meta fields defaults.
	 *
	 * @since 1.0.0.
	 *
	 * @return array
	 */
	public function get_customize_fields() {
		$fields = array(
			'type' => array(
				'default' => $this->type,
				'sanitize' => 'sanitize_text_field',
			),
			'label' => array(
				'default' => __( '', 'happyforms' ),
				'sanitize' => 'sanitize_text_field',
			),
			'label_placement' => array(
				'default' => '',
				'sanitize' => 'sanitize_text_field'
			),
			'description' => array(
				'default' => '',
				'sanitize' => 'sanitize_text_field'
			),
			'description_mode' => array(
				'default' => '',
				'sanitize' => 'sanitize_text_field'
			),
			'placeholder' => array(
				'default' => __( 'Legal name', 'happyforms' ),
				'sanitize' => 'sanitize_text_field'
			),
			'intent_text' => array(
				'default' => __( 'I agree that my typed name below will be as valid as a handwritten signature to the extent allowed by local law', 'happyforms' ),
				'sanitize' => 'esc_html'
			),
			'width' => array(
				'default' => 'full',
				'sanitize' => 'sanitize_key'
			),
			'css_class' => array(
				'default' => '',
				'sanitize' => 'sanitize_text_field'
			),
			'required' => array(
				'default' => 1,
				'sanitize' => 'happyforms_sanitize_checkbox',
			),
		);

		return happyforms_get_part_customize_fields( $fields, $this->type );
	}

	public function get_default_value( $part_data = array() ) {
		return array(
			'intent' => '',
			'signature' => ''
		);
	}

	/**
	 * Get template for part item in customize pane.
	 *
	 * @since 1.0.0.
	 *
	 * @return string
	 */
	public function customize_templates() {
		$template_path = happyforms_get_include_folder() . '/templates/parts/customize-signature.php';
		$template_path = happyforms_get_part_customize_template_path( $template_path, $this->type );

		require_once( $template_path );
	}

	/**
	 * Get front end part template with parsed data.
	 *
	 * @since 1.0.0.
	 *
	 * @param array	$part_data 	Form part data.
	 * @param array	$form_data	Form (post) data.
	 *
	 * @return string	Markup for the form part.
	 */
	public function frontend_template( $part_data = array(), $form_data = array() ) {
		$part = wp_parse_args( $part_data, $this->get_customize_defaults() );
		$form = $form_data;

		include( happyforms_get_include_folder() . '/templates/parts/frontend-signature.php' );
	}

	/**
	 * Sanitize submitted value before storing it.
	 *
	 * @since 1.0.0.
	 *
	 * @param array $part_data Form part data.
	 *
	 * @return string
	 */
	public function sanitize_value( $part_data = array(), $form_data = array(), $request = array() ) {
		$sanitized_value = $this->get_default_value( $part_data );
		$part_name = happyforms_get_part_name( $part_data, $form_data );

		if ( isset( $request[$part_name] ) ) {
			$sanitized_value = wp_parse_args( $request[$part_name], $sanitized_value );
			$sanitized_value = array_map( 'sanitize_text_field', $sanitized_value );
		}

		return $sanitized_value;
	}

	/**
	 * Validate value before submitting it. If it fails validation, return WP_Error object, showing respective error message.
	 *
	 * @since 1.0.0.
	 *
	 * @param array $part Form part data.
	 * @param string $value Submitted value.
	 *
	 * @return string|object
	 */
	public function validate_value( $value, $part = array(), $form = array() ) {
		$validated_value = $value;

		$intent = $validated_value['intent'];
		$signature_string = $validated_value['signature'];

		if ( 1 === $part['required'] && ( $intent !== 'yes' || empty( $signature_string ) ) ) {
			$validated_value = new WP_Error( 'error', happyforms_get_validation_message( 'field_empty' ) );
			return $validated_value;
		} else if ( ( $intent === 'yes' && empty( $signature_string ) ) || ( $intent !== 'yes' && $signature_string ) ) {
			$validated_value = new WP_Error( 'error', happyforms_get_validation_message( 'field_invalid' ) );
			return $validated_value;
		}

		return $validated_value;
	}

	public function message_part_value( $value, $original_value, $part, $destination ) {
		if ( isset( $part['type'] ) && $this->type === $part['type'] ) {
			$original_value = maybe_unserialize( $original_value );
			$value = 'yes' === $original_value['intent'] ? $original_value['signature'] : '';

			if( $destination == 'admin-column' ) {
				$value = 'yes' === $original_value['intent'] ? esc_html( $part['intent_text'] ) . '<br />' . $value : '';
			}
		}

		return $value;
	}

	public function email_part_value( $value, $message, $part, $form, $context ) {
		if ( $this->type === $part['type'] ) {
			$part_name = happyforms_get_part_name( $part, $form );
			$original_value = $message['request'][$part_name];

			$value = 'yes' === $original_value['intent'] ? $part['intent_text'] . "<br><br>" . $original_value['signature'] : '';
		}

		return $value;
	}

	/**
	 * Enqueue scripts in customizer area.
	 *
	 * @since 1.0.0.
	 *
	 * @param array	List of dependencies.
	 *
	 * @return void
	 */
	public function customize_enqueue_scripts( $deps = array() ) {
		wp_enqueue_script(
			'part-signature',
			happyforms_get_plugin_url() . 'inc/assets/js/parts/part-signature.js',
			$deps, HAPPYFORMS_VERSION, true
		);
	}

	public function html_part_class( $class, $part, $form ) {
		if ( $part['type'] === $this->type ) {
			if ( 'above_signature' === $part['description_mode'] ) {
				$class[] = 'happyforms-part--description-above-signature';
			}
		}

		return $class;
	}

	public function script_dependencies( $deps, $forms ) {
		$contains_signature = false;
		$form_controller = happyforms_get_form_controller();

		foreach ( $forms as $form ) {
			if ( $form_controller->get_first_part_by_type( $form, $this->type ) ) {
				$contains_signature = true;
				break;
			}
		}

		if ( ! happyforms_is_preview() && ! $contains_signature ) {
			return $deps;
		}

		wp_register_script(
			'happyforms-part-signature',
			happyforms_get_plugin_url() . 'inc/assets/js/frontend/signature.js',
			array(), HAPPYFORMS_VERSION, true
		);

		$deps[] = 'happyforms-part-signature';

		return $deps;
	}

}
